/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.clientregistration;

import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.Response;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.keycloak.Config;
import org.keycloak.authentication.AuthenticationProcessor;
import org.keycloak.common.util.Time;
import org.keycloak.events.EventBuilder;
import org.keycloak.models.AdminRoles;
import org.keycloak.models.ClientInitialAccessModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.protocol.oidc.utils.AuthorizeClientUtil;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.services.ErrorResponseException;
import org.keycloak.services.clientpolicy.ClientPolicyContext;
import org.keycloak.services.clientpolicy.ClientPolicyException;
import org.keycloak.services.clientpolicy.context.DynamicClientRegisterContext;
import org.keycloak.services.clientpolicy.context.DynamicClientUnregisterContext;
import org.keycloak.services.clientpolicy.context.DynamicClientUpdateContext;
import org.keycloak.services.clientpolicy.context.DynamicClientViewContext;
import org.keycloak.services.clientregistration.ClientRegistrationContext;
import org.keycloak.services.clientregistration.ClientRegistrationProvider;
import org.keycloak.services.clientregistration.ClientRegistrationTokenUtils;
import org.keycloak.services.clientregistration.policy.ClientRegistrationPolicyException;
import org.keycloak.services.clientregistration.policy.ClientRegistrationPolicyManager;
import org.keycloak.services.clientregistration.policy.RegistrationAuth;
import org.keycloak.services.managers.AuthenticationManager;

public class ClientRegistrationAuth {
    private final KeycloakSession session;
    private final ClientRegistrationProvider provider;
    private final EventBuilder event;
    private RealmModel realm;
    private AccessToken jwt;
    private ClientInitialAccessModel initialAccessModel;
    private String kid;
    private String token;
    private String endpoint;

    public ClientRegistrationAuth(KeycloakSession session, ClientRegistrationProvider provider, EventBuilder event, String endpoint) {
        this.session = session;
        this.provider = provider;
        this.event = event;
        this.endpoint = endpoint;
    }

    private void init() {
        this.realm = this.session.getContext().getRealm();
        String authorizationHeader = (String)this.session.getContext().getRequestHeaders().getRequestHeaders().getFirst((Object)"Authorization");
        if (authorizationHeader == null) {
            return;
        }
        String[] split = authorizationHeader.split(" ");
        if (!split[0].equalsIgnoreCase("bearer")) {
            return;
        }
        this.token = split[1];
        ClientRegistrationTokenUtils.TokenVerification tokenVerification = ClientRegistrationTokenUtils.verifyToken(this.session, this.realm, this.token);
        if (tokenVerification.getError() != null) {
            throw this.unauthorized(tokenVerification.getError().getMessage());
        }
        this.kid = tokenVerification.getKid();
        this.jwt = tokenVerification.getJwt();
        if (this.isInitialAccessToken()) {
            this.initialAccessModel = this.session.realms().getClientInitialAccessModel(this.session.getContext().getRealm(), this.jwt.getId());
            if (this.initialAccessModel == null) {
                throw this.unauthorized("Initial Access Token not found");
            }
        }
    }

    public String getToken() {
        return this.token;
    }

    public String getKid() {
        return this.kid;
    }

    public AccessToken getJwt() {
        return this.jwt;
    }

    private boolean isBearerToken() {
        return this.jwt != null && "Bearer".equals(this.jwt.getType());
    }

    public boolean isInitialAccessToken() {
        return this.jwt != null && "InitialAccessToken".equals(this.jwt.getType());
    }

    public boolean isRegistrationAccessToken() {
        return this.jwt != null && "RegistrationAccessToken".equals(this.jwt.getType());
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public RegistrationAuth requireCreate(ClientRegistrationContext context) {
        this.init();
        RegistrationAuth registrationAuth = RegistrationAuth.ANONYMOUS;
        if (this.isBearerToken()) {
            if (!this.hasRole(AdminRoles.MANAGE_CLIENTS, AdminRoles.CREATE_CLIENT)) throw this.forbidden();
            registrationAuth = RegistrationAuth.AUTHENTICATED;
        } else if (this.isInitialAccessToken()) {
            if (this.initialAccessModel.getRemainingCount() <= 0) throw this.unauthorized("No remaining count on initial access token");
            if (this.initialAccessModel.getExpiration() != 0 && this.initialAccessModel.getTimestamp() + this.initialAccessModel.getExpiration() <= Time.currentTime()) throw this.unauthorized("Expired initial access token");
            registrationAuth = RegistrationAuth.AUTHENTICATED;
        }
        try {
            this.session.clientPolicy().triggerOnEvent((ClientPolicyContext)new DynamicClientRegisterContext(context, (JsonWebToken)this.jwt, this.realm));
            ClientRegistrationPolicyManager.triggerBeforeRegister(context, registrationAuth);
            return registrationAuth;
        }
        catch (ClientPolicyException | ClientRegistrationPolicyException crpe) {
            throw this.forbidden(crpe.getMessage());
        }
    }

    public void requireView(ClientModel client) {
        this.requireView(client, false);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void requireView(ClientModel client, boolean allowPublicClient) {
        RegistrationAuth authType = null;
        boolean authenticated = false;
        this.init();
        if (this.isBearerToken()) {
            this.checkClientProtocol();
            if (!this.hasRole(AdminRoles.MANAGE_CLIENTS, AdminRoles.VIEW_CLIENTS)) throw this.forbidden();
            if (client == null) {
                throw this.notFound();
            }
            authenticated = true;
            authType = RegistrationAuth.AUTHENTICATED;
        } else if (this.isRegistrationAccessToken()) {
            if (client != null && client.getRegistrationToken() != null && client.getRegistrationToken().equals(this.jwt.getId())) {
                this.checkClientProtocol(client);
                authenticated = true;
                authType = this.getRegistrationAuth();
            }
        } else {
            if (this.isInitialAccessToken()) {
                throw this.unauthorized("Not initial access token allowed");
            }
            if (allowPublicClient && this.authenticatePublicClient(client)) {
                authenticated = true;
                authType = RegistrationAuth.AUTHENTICATED;
            }
        }
        if (!authenticated) throw this.unauthorized("Not authorized to view client. Not valid token or client credentials provided.");
        try {
            this.session.clientPolicy().triggerOnEvent((ClientPolicyContext)new DynamicClientViewContext(this.session, client, (JsonWebToken)this.jwt, this.realm));
            ClientRegistrationPolicyManager.triggerBeforeView(this.session, this.provider, authType, client);
            return;
        }
        catch (ClientPolicyException | ClientRegistrationPolicyException crpe) {
            throw this.forbidden(crpe.getMessage());
        }
    }

    public RegistrationAuth getRegistrationAuth() {
        String str = (String)this.jwt.getOtherClaims().get("registration_auth");
        return RegistrationAuth.fromString(str);
    }

    public RegistrationAuth requireUpdate(ClientRegistrationContext context, ClientModel client) {
        RegistrationAuth regAuth = this.requireUpdateAuth(client);
        try {
            this.session.clientPolicy().triggerOnEvent((ClientPolicyContext)new DynamicClientUpdateContext(context, client, (JsonWebToken)this.jwt, this.realm));
            ClientRegistrationPolicyManager.triggerBeforeUpdate(context, regAuth, client);
        }
        catch (ClientPolicyException | ClientRegistrationPolicyException crpe) {
            throw this.forbidden(crpe.getMessage());
        }
        return regAuth;
    }

    public void requireDelete(ClientModel client) {
        RegistrationAuth chainType = this.requireUpdateAuth(client);
        try {
            this.session.clientPolicy().triggerOnEvent((ClientPolicyContext)new DynamicClientUnregisterContext(this.session, client, (JsonWebToken)this.jwt, this.realm));
            ClientRegistrationPolicyManager.triggerBeforeRemove(this.session, this.provider, chainType, client);
        }
        catch (ClientPolicyException | ClientRegistrationPolicyException crpe) {
            throw this.forbidden(crpe.getMessage());
        }
    }

    private void checkClientProtocol() {
        ClientModel client = this.session.getContext().getRealm().getClientByClientId(this.jwt.getIssuedFor());
        this.checkClientProtocol(client);
    }

    private void checkClientProtocol(ClientModel client) {
        if ((this.endpoint.equals("openid-connect") || this.endpoint.equals("saml2-entity-descriptor")) && client != null && !this.endpoint.contains(client.getProtocol())) {
            throw new ErrorResponseException("invalid_client", "Wrong client protocol.", Response.Status.BAD_REQUEST);
        }
    }

    private RegistrationAuth requireUpdateAuth(ClientModel client) {
        this.init();
        if (this.isBearerToken()) {
            this.checkClientProtocol();
            if (this.hasRole(AdminRoles.MANAGE_CLIENTS)) {
                if (client == null) {
                    throw this.notFound();
                }
                return RegistrationAuth.AUTHENTICATED;
            }
            throw this.forbidden();
        }
        if (this.isRegistrationAccessToken() && client != null && client.getRegistrationToken() != null && client.getRegistrationToken().equals(this.jwt.getId())) {
            return this.getRegistrationAuth();
        }
        throw this.unauthorized("Not authorized to update client. Maybe missing token or bad token type.");
    }

    public ClientInitialAccessModel getInitialAccessModel() {
        return this.initialAccessModel;
    }

    private boolean hasRole(String ... roles) {
        try {
            AuthenticationManager.resolveLightweightAccessTokenRoles(this.session, this.jwt, this.session.getContext().getRealm());
            return this.hasRoleInToken(roles);
        }
        catch (Throwable t) {
            return false;
        }
    }

    private boolean hasRoleInToken(String[] role) {
        Map resourceAccess = this.jwt.getResourceAccess();
        if (resourceAccess == null) {
            return false;
        }
        String clientId = this.realm.getName().equals(Config.getAdminRealm()) ? this.realm.getMasterAdminClient().getClientId() : "realm-management";
        Set roles = Optional.ofNullable((AccessToken.Access)resourceAccess.get(clientId)).map(AccessToken.Access::getRoles).orElse(Collections.emptySet());
        return Arrays.stream(role).anyMatch(roles::contains);
    }

    private boolean authenticatePublicClient(ClientModel client) {
        if (client == null) {
            return false;
        }
        if (client.isPublicClient()) {
            return true;
        }
        AuthenticationProcessor processor = AuthorizeClientUtil.getAuthenticationProcessor(this.session, this.event);
        Response response = processor.authenticateClient();
        if (response != null) {
            this.event.client(client.getClientId()).error("not_allowed");
            throw this.unauthorized("Failed to authenticate client");
        }
        ClientModel authClient = processor.getClient();
        if (authClient == null) {
            this.event.client(client.getClientId()).error("not_allowed");
            throw this.unauthorized("No client authenticated");
        }
        if (!authClient.getClientId().equals(client.getClientId())) {
            this.event.client(client.getClientId()).error("not_allowed");
            throw this.unauthorized("Different client authenticated");
        }
        this.checkClientProtocol(authClient);
        return true;
    }

    private WebApplicationException unauthorized(String errorDescription) {
        this.event.detail("reason", errorDescription).error("invalid_token");
        throw new ErrorResponseException("invalid_token", errorDescription, Response.Status.UNAUTHORIZED);
    }

    private WebApplicationException forbidden() {
        return this.forbidden("Forbidden");
    }

    private WebApplicationException forbidden(String errorDescription) {
        this.event.error("not_allowed");
        throw new ErrorResponseException("insufficient_scope", errorDescription, Response.Status.FORBIDDEN);
    }

    private WebApplicationException notFound() {
        this.event.error("client_not_found");
        throw new ErrorResponseException("invalid_request", "Client not found", Response.Status.NOT_FOUND);
    }
}

