/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.storage;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Spliterators;
import java.util.function.Consumer;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.helper.ReflectionCache;
import org.traccar.model.Permission;

public final class QueryBuilder {
    private static final Logger LOGGER = LoggerFactory.getLogger(QueryBuilder.class);
    private final Config config;
    private final ObjectMapper objectMapper;
    private Connection connection;
    private PreparedStatement statement;
    private final String query;
    private final boolean returnGeneratedKeys;

    private QueryBuilder(Config config, DataSource dataSource, ObjectMapper objectMapper, String query, boolean returnGeneratedKeys) throws SQLException {
        this.config = config;
        this.objectMapper = objectMapper;
        this.query = query;
        this.returnGeneratedKeys = returnGeneratedKeys;
        if (query != null) {
            this.connection = dataSource.getConnection();
            try {
                this.statement = returnGeneratedKeys ? this.connection.prepareStatement(query, 1) : this.connection.prepareStatement(query);
            }
            catch (SQLException error) {
                this.connection.close();
                throw error;
            }
        }
    }

    public static QueryBuilder create(Config config, DataSource dataSource, ObjectMapper objectMapper, String query) throws SQLException {
        return new QueryBuilder(config, dataSource, objectMapper, query, false);
    }

    public static QueryBuilder create(Config config, DataSource dataSource, ObjectMapper objectMapper, String query, boolean returnGeneratedKeys) throws SQLException {
        return new QueryBuilder(config, dataSource, objectMapper, query, returnGeneratedKeys);
    }

    private QueryBuilder setValue(ValueSetter setter) throws SQLException {
        try {
            setter.invoke();
        }
        catch (SQLException error) {
            this.statement.close();
            this.connection.close();
            throw error;
        }
        return this;
    }

    public QueryBuilder setBoolean(int index, boolean value) throws SQLException {
        return this.setValue(() -> this.statement.setBoolean(index + 1, value));
    }

    public QueryBuilder setInteger(int index, int value) throws SQLException {
        return this.setValue(() -> this.statement.setInt(index + 1, value));
    }

    public QueryBuilder setLong(int index, long value) throws SQLException {
        return this.setLong(index, value, false);
    }

    public QueryBuilder setLong(int index, long value, boolean nullIfZero) throws SQLException {
        return this.setValue(() -> {
            if (value == 0L && nullIfZero) {
                this.statement.setNull(index + 1, -5);
            } else {
                this.statement.setLong(index + 1, value);
            }
        });
    }

    public QueryBuilder setDouble(int index, double value) throws SQLException {
        return this.setValue(() -> this.statement.setDouble(index + 1, value));
    }

    public QueryBuilder setString(int index, String value) throws SQLException {
        return this.setValue(() -> {
            if (value == null) {
                this.statement.setNull(index + 1, 12);
            } else {
                this.statement.setString(index + 1, value);
            }
        });
    }

    public QueryBuilder setDate(int index, Date value) throws SQLException {
        return this.setValue(() -> {
            if (value == null) {
                this.statement.setNull(index + 1, 93);
            } else {
                this.statement.setTimestamp(index + 1, new Timestamp(value.getTime()));
            }
        });
    }

    public QueryBuilder setBlob(int index, byte[] value) throws SQLException {
        return this.setValue(() -> {
            if (value == null) {
                this.statement.setNull(index + 1, 2004);
            } else {
                this.statement.setBytes(index + 1, value);
            }
        });
    }

    public QueryBuilder setValue(int index, Object value) throws SQLException {
        if (value instanceof Boolean) {
            Boolean booleanValue = (Boolean)value;
            this.setBoolean(index, booleanValue);
        } else if (value instanceof Integer) {
            Integer integerValue = (Integer)value;
            this.setInteger(index, integerValue);
        } else if (value instanceof Long) {
            Long longValue = (Long)value;
            this.setLong(index, longValue);
        } else if (value instanceof Double) {
            Double doubleValue = (Double)value;
            this.setDouble(index, doubleValue);
        } else if (value instanceof String) {
            String stringValue = (String)value;
            this.setString(index, stringValue);
        } else if (value instanceof Date) {
            Date dateValue = (Date)value;
            this.setDate(index, dateValue);
        }
        return this;
    }

    public QueryBuilder setObject(Object object, List<String> columns) throws SQLException {
        try {
            for (int index = 0; index < columns.size(); ++index) {
                String column = columns.get(index);
                Method method = ReflectionCache.getProperties(object.getClass(), "get").get(column).method();
                if (method.getReturnType().equals(Boolean.TYPE)) {
                    this.setBoolean(index, (Boolean)method.invoke(object, new Object[0]));
                    continue;
                }
                if (method.getReturnType().equals(Integer.TYPE)) {
                    this.setInteger(index, (Integer)method.invoke(object, new Object[0]));
                    continue;
                }
                if (method.getReturnType().equals(Long.TYPE)) {
                    this.setLong(index, (Long)method.invoke(object, new Object[0]), column.endsWith("Id"));
                    continue;
                }
                if (method.getReturnType().equals(Double.TYPE)) {
                    this.setDouble(index, (Double)method.invoke(object, new Object[0]));
                    continue;
                }
                if (method.getReturnType().equals(String.class)) {
                    this.setString(index, (String)method.invoke(object, new Object[0]));
                    continue;
                }
                if (method.getReturnType().equals(Date.class)) {
                    this.setDate(index, (Date)method.invoke(object, new Object[0]));
                    continue;
                }
                if (method.getReturnType().equals(byte[].class)) {
                    this.setBlob(index, (byte[])method.invoke(object, new Object[0]));
                    continue;
                }
                this.setString(index, this.objectMapper.writeValueAsString(method.invoke(object, new Object[0])));
            }
        }
        catch (JsonProcessingException | ReflectiveOperationException e) {
            LOGGER.warn("Set object error", e);
        }
        return this;
    }

    private <T> void addProcessors(List<ResultSetProcessor<T>> processors, Class<?> parameterType, Method method, String name) {
        if (parameterType.equals(Boolean.TYPE)) {
            processors.add((object, resultSet) -> method.invoke(object, resultSet.getBoolean(name)));
        } else if (parameterType.equals(Integer.TYPE)) {
            processors.add((object, resultSet) -> method.invoke(object, resultSet.getInt(name)));
        } else if (parameterType.equals(Long.TYPE)) {
            processors.add((object, resultSet) -> method.invoke(object, resultSet.getLong(name)));
        } else if (parameterType.equals(Double.TYPE)) {
            processors.add((object, resultSet) -> method.invoke(object, resultSet.getDouble(name)));
        } else if (parameterType.equals(String.class)) {
            processors.add((object, resultSet) -> method.invoke(object, resultSet.getString(name)));
        } else if (parameterType.equals(Date.class)) {
            processors.add((object, resultSet) -> {
                Timestamp timestamp = resultSet.getTimestamp(name);
                if (timestamp != null) {
                    method.invoke(object, new Date(timestamp.getTime()));
                }
            });
        } else if (parameterType.equals(byte[].class)) {
            processors.add((object, resultSet) -> method.invoke(object, new Object[]{resultSet.getBytes(name)}));
        } else {
            processors.add((object, resultSet) -> {
                String value = resultSet.getString(name);
                if (value != null && !value.isEmpty()) {
                    method.invoke(object, this.objectMapper.readValue(value, parameterType));
                }
            });
        }
    }

    private void logQuery() {
        if (this.config.getBoolean(Keys.LOGGER_QUERIES)) {
            LOGGER.info(this.query);
        }
    }

    public <T> List<T> executeQuery(Class<T> clazz) throws SQLException {
        try (Stream<T> stream = this.executeQueryStreamed(clazz);){
            List<T> list = stream.toList();
            return list;
        }
    }

    public <T> Stream<T> executeQueryStreamed(final Class<T> clazz) throws SQLException {
        if (this.query == null) {
            return Stream.empty();
        }
        ResultSet resultSet = null;
        try {
            this.logQuery();
            resultSet = this.statement.executeQuery();
            ResultSetMetaData resultMetaData = resultSet.getMetaData();
            final ArrayList<ResultSetProcessor<T>> processors = new ArrayList<ResultSetProcessor<T>>();
            for (Map.Entry<String, ReflectionCache.PropertyMethod> entry : ReflectionCache.getProperties(clazz, "set").entrySet()) {
                String name = entry.getKey();
                boolean column = false;
                for (int i = 1; i <= resultMetaData.getColumnCount(); ++i) {
                    if (!name.equalsIgnoreCase(resultMetaData.getColumnLabel(i))) continue;
                    column = true;
                    break;
                }
                if (!column) continue;
                Method method = entry.getValue().method();
                this.addProcessors(processors, method.getParameterTypes()[0], method, name);
            }
            final ResultSet retainedResultSet = resultSet;
            return (Stream)StreamSupport.stream(new Spliterators.AbstractSpliterator<T>(Long.MAX_VALUE, 16){

                @Override
                public boolean tryAdvance(Consumer<? super T> action) {
                    try {
                        if (retainedResultSet.next()) {
                            Object object = clazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                            for (ResultSetProcessor processor : processors) {
                                try {
                                    processor.process(object, retainedResultSet);
                                }
                                catch (IOException | ReflectiveOperationException error) {
                                    LOGGER.warn("Set property error", (Throwable)error);
                                }
                            }
                            action.accept(object);
                            return true;
                        }
                        return false;
                    }
                    catch (ReflectiveOperationException | SQLException e) {
                        throw new RuntimeException(e);
                    }
                }
            }, false).onClose(() -> this.close(retainedResultSet));
        }
        catch (Exception e) {
            this.close(resultSet);
            throw e;
        }
    }

    private void close(ResultSet resultSet) {
        try {
            if (resultSet != null) {
                resultSet.close();
            }
            this.statement.close();
            this.connection.close();
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long executeUpdate() throws SQLException {
        block12: {
            if (this.query != null) {
                try {
                    this.logQuery();
                    this.statement.execute();
                    if (!this.returnGeneratedKeys) break block12;
                    try (ResultSet resultSet = this.statement.getGeneratedKeys();){
                        if (resultSet.next()) {
                            long l = resultSet.getLong(1);
                            return l;
                        }
                    }
                }
                finally {
                    this.statement.close();
                    this.connection.close();
                }
            }
        }
        return 0L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<Permission> executePermissionsQuery() throws SQLException {
        LinkedList<Permission> result = new LinkedList<Permission>();
        if (this.query != null) {
            try {
                this.logQuery();
                try (ResultSet resultSet = this.statement.executeQuery();){
                    ResultSetMetaData resultMetaData = resultSet.getMetaData();
                    while (resultSet.next()) {
                        LinkedHashMap<String, Long> map = new LinkedHashMap<String, Long>();
                        for (int i = 1; i <= resultMetaData.getColumnCount(); ++i) {
                            String label = resultMetaData.getColumnLabel(i);
                            map.put(label, resultSet.getLong(label));
                        }
                        result.add(new Permission(map));
                    }
                }
            }
            finally {
                this.statement.close();
                this.connection.close();
            }
        }
        return result;
    }

    private static interface ValueSetter {
        public void invoke() throws SQLException;
    }

    private static interface ResultSetProcessor<T> {
        public void process(T var1, ResultSet var2) throws ReflectiveOperationException, IOException, SQLException;
    }
}

