/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.firebase.internal.NonNull;
import com.google.firebase.internal.Nullable;
import com.google.firebase.remoteconfig.AndCondition;
import com.google.firebase.remoteconfig.CustomSignalCondition;
import com.google.firebase.remoteconfig.CustomSignalOperator;
import com.google.firebase.remoteconfig.KeysAndValues;
import com.google.firebase.remoteconfig.MicroPercentRange;
import com.google.firebase.remoteconfig.OneOfCondition;
import com.google.firebase.remoteconfig.OrCondition;
import com.google.firebase.remoteconfig.PercentCondition;
import com.google.firebase.remoteconfig.PercentConditionOperator;
import com.google.firebase.remoteconfig.ServerCondition;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;
import java.util.function.IntPredicate;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class ConditionEvaluator {
    private static final int MAX_CONDITION_RECURSION_DEPTH = 10;
    private static final Logger logger = LoggerFactory.getLogger(ConditionEvaluator.class);
    private static final BigInteger MICRO_PERCENT_MODULO = BigInteger.valueOf(100000000L);
    private static final Pattern SEMVER_PATTERN = Pattern.compile("^[0-9]+(?:\\.[0-9]+){0,4}$");

    ConditionEvaluator() {
    }

    @NonNull
    Map<String, Boolean> evaluateConditions(@NonNull List<ServerCondition> conditions, @Nullable KeysAndValues context) {
        Preconditions.checkNotNull(conditions, (Object)"List of conditions must not be null.");
        Preconditions.checkArgument((!conditions.isEmpty() ? 1 : 0) != 0, (Object)"List of conditions must not be empty.");
        if (context == null || conditions.isEmpty()) {
            return ImmutableMap.of();
        }
        KeysAndValues evaluationContext = context != null ? context : new KeysAndValues.Builder().build();
        Map evaluatedConditions = (Map)conditions.stream().collect(ImmutableMap.toImmutableMap(ServerCondition::getName, condition -> this.evaluateCondition(condition.getCondition(), evaluationContext, 0)));
        return evaluatedConditions;
    }

    private boolean evaluateCondition(OneOfCondition condition, KeysAndValues context, int nestingLevel) {
        if (nestingLevel > 10) {
            logger.warn("Maximum condition recursion depth exceeded.");
            return false;
        }
        if (condition.getOrCondition() != null) {
            return this.evaluateOrCondition(condition.getOrCondition(), context, nestingLevel + 1);
        }
        if (condition.getAndCondition() != null) {
            return this.evaluateAndCondition(condition.getAndCondition(), context, nestingLevel + 1);
        }
        if (condition.isTrue() != null) {
            return true;
        }
        if (condition.isFalse() != null) {
            return false;
        }
        if (condition.getCustomSignal() != null) {
            return this.evaluateCustomSignalCondition(condition.getCustomSignal(), context);
        }
        if (condition.getPercent() != null) {
            return this.evaluatePercentCondition(condition.getPercent(), context);
        }
        logger.atWarn().log("Received invalid condition for evaluation.");
        return false;
    }

    private boolean evaluateOrCondition(OrCondition condition, KeysAndValues context, int nestingLevel) {
        return condition.getConditions().stream().anyMatch(subCondition -> this.evaluateCondition((OneOfCondition)subCondition, context, nestingLevel + 1));
    }

    private boolean evaluateAndCondition(AndCondition condition, KeysAndValues context, int nestingLevel) {
        return condition.getConditions().stream().allMatch(subCondition -> this.evaluateCondition((OneOfCondition)subCondition, context, nestingLevel + 1));
    }

    private boolean evaluateCustomSignalCondition(CustomSignalCondition condition, KeysAndValues context) {
        CustomSignalOperator customSignalOperator = condition.getCustomSignalOperator();
        String customSignalKey = condition.getCustomSignalKey();
        ImmutableList targetCustomSignalValues = ImmutableList.copyOf(condition.getTargetCustomSignalValues());
        if (targetCustomSignalValues.isEmpty()) {
            logger.warn(String.format("Values must be assigned to all custom signal fields. Operator:%s, Key:%s, Values:%s", new Object[]{customSignalOperator, customSignalKey, targetCustomSignalValues}));
            return false;
        }
        String customSignalValue = context.get(customSignalKey);
        if (customSignalValue == null) {
            return false;
        }
        switch (customSignalOperator) {
            case STRING_CONTAINS: {
                return this.compareStrings((ImmutableList<String>)targetCustomSignalValues, customSignalValue, (customSignal, targetSignal) -> customSignal.contains((CharSequence)targetSignal));
            }
            case STRING_DOES_NOT_CONTAIN: {
                return !this.compareStrings((ImmutableList<String>)targetCustomSignalValues, customSignalValue, (customSignal, targetSignal) -> customSignal.contains((CharSequence)targetSignal));
            }
            case STRING_EXACTLY_MATCHES: {
                return this.compareStrings((ImmutableList<String>)targetCustomSignalValues, customSignalValue, (customSignal, targetSignal) -> customSignal.equals(targetSignal));
            }
            case STRING_CONTAINS_REGEX: {
                return this.compareStrings((ImmutableList<String>)targetCustomSignalValues, customSignalValue, (customSignal, targetSignal) -> this.compareStringRegex((String)customSignal, (String)targetSignal));
            }
            case NUMERIC_LESS_THAN: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result < 0);
            }
            case NUMERIC_LESS_EQUAL: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result <= 0);
            }
            case NUMERIC_EQUAL: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result == 0);
            }
            case NUMERIC_NOT_EQUAL: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result != 0);
            }
            case NUMERIC_GREATER_THAN: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result > 0);
            }
            case NUMERIC_GREATER_EQUAL: {
                return this.compareNumbers((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result >= 0);
            }
            case SEMANTIC_VERSION_EQUAL: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result == 0);
            }
            case SEMANTIC_VERSION_GREATER_EQUAL: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result >= 0);
            }
            case SEMANTIC_VERSION_GREATER_THAN: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result > 0);
            }
            case SEMANTIC_VERSION_LESS_EQUAL: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result <= 0);
            }
            case SEMANTIC_VERSION_LESS_THAN: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result < 0);
            }
            case SEMANTIC_VERSION_NOT_EQUAL: {
                return this.compareSemanticVersions((ImmutableList<String>)targetCustomSignalValues, customSignalValue, result -> result != 0);
            }
        }
        return false;
    }

    private boolean evaluatePercentCondition(PercentCondition condition, KeysAndValues context) {
        if (!context.containsKey("randomizationId")) {
            logger.warn("Percentage operation must not be performed without randomizationId");
            return false;
        }
        PercentConditionOperator operator = condition.getPercentConditionOperator();
        MicroPercentRange microPercentRange = condition.getMicroPercentRange();
        int microPercentUpperBound = microPercentRange != null ? microPercentRange.getMicroPercentUpperBound() : 0;
        int microPercentLowerBound = microPercentRange != null ? microPercentRange.getMicroPercentLowerBound() : 0;
        int microPercent = condition.getMicroPercent();
        BigInteger microPercentile = this.getMicroPercentile(condition.getSeed(), context.get("randomizationId"));
        switch (operator) {
            case LESS_OR_EQUAL: {
                return microPercentile.compareTo(BigInteger.valueOf(microPercent)) <= 0;
            }
            case GREATER_THAN: {
                return microPercentile.compareTo(BigInteger.valueOf(microPercent)) > 0;
            }
            case BETWEEN: {
                return microPercentile.compareTo(BigInteger.valueOf(microPercentLowerBound)) > 0 && microPercentile.compareTo(BigInteger.valueOf(microPercentUpperBound)) <= 0;
            }
        }
        return false;
    }

    private BigInteger getMicroPercentile(String seed, String randomizationId) {
        String seedPrefix = seed != null && !seed.isEmpty() ? seed + "." : "";
        String stringToHash = seedPrefix + randomizationId;
        BigInteger hash = this.hashSeededRandomizationId(stringToHash);
        BigInteger microPercentile = hash.mod(MICRO_PERCENT_MODULO);
        return microPercentile;
    }

    private BigInteger hashSeededRandomizationId(String seededRandomizationId) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hashBytes = digest.digest(seededRandomizationId.getBytes(StandardCharsets.UTF_8));
            return new BigInteger(1, hashBytes);
        }
        catch (NoSuchAlgorithmException e) {
            logger.error("SHA-256 algorithm not found", (Throwable)e);
            throw new RuntimeException("SHA-256 algorithm not found", e);
        }
    }

    private boolean compareStrings(ImmutableList<String> targetValues, String customSignal, BiPredicate<String, String> compareFunction) {
        return targetValues.stream().anyMatch(targetValue -> compareFunction.test(customSignal, (String)targetValue));
    }

    private boolean compareStringRegex(String customSignal, String targetSignal) {
        try {
            return Pattern.compile(targetSignal).matcher(customSignal).matches();
        }
        catch (PatternSyntaxException e) {
            return false;
        }
    }

    private boolean compareNumbers(ImmutableList<String> targetValues, String customSignal, IntPredicate compareFunction) {
        if (targetValues.size() != 1) {
            logger.warn(String.format("Target values must contain 1 element for numeric operations. Target Value: %s", targetValues));
            return false;
        }
        try {
            double customSignalDouble = Double.parseDouble(customSignal);
            double targetValue = Double.parseDouble((String)targetValues.get(0));
            int comparisonResult = Double.compare(customSignalDouble, targetValue);
            return compareFunction.test(comparisonResult);
        }
        catch (NumberFormatException e) {
            logger.warn("Error parsing numeric values: customSignal=%s, targetValue=%s", new Object[]{customSignal, targetValues.get(0), e});
            return false;
        }
    }

    private boolean compareSemanticVersions(ImmutableList<String> targetValues, String customSignal, IntPredicate compareFunction) {
        if (targetValues.size() != 1) {
            logger.warn(String.format("Target values must contain 1 element for semantic operation.", new Object[0]));
            return false;
        }
        String targetValueString = (String)targetValues.get(0);
        if (!this.validateSemanticVersion(targetValueString) || !this.validateSemanticVersion(customSignal)) {
            return false;
        }
        List<Integer> targetVersion = this.parseSemanticVersion(targetValueString);
        List<Integer> customSignalVersion = this.parseSemanticVersion(customSignal);
        int maxLength = 5;
        if (targetVersion.size() > maxLength || customSignalVersion.size() > maxLength) {
            logger.warn("Semantic version max length(%s) exceeded. Target: %s, Custom Signal: %s", new Object[]{maxLength, targetValueString, customSignal});
            return false;
        }
        int comparison = this.compareSemanticVersions(customSignalVersion, targetVersion);
        return compareFunction.test(comparison);
    }

    private int compareSemanticVersions(List<Integer> version1, List<Integer> version2) {
        int maxLength = Math.max(version1.size(), version2.size());
        int version1Size = version1.size();
        int version2Size = version2.size();
        for (int i = 0; i < maxLength; ++i) {
            int v2;
            int v1 = i < version1Size ? version1.get(i) : 0;
            int comparison = Integer.compare(v1, v2 = i < version2Size ? version2.get(i) : 0);
            if (comparison == 0) continue;
            return comparison;
        }
        return 0;
    }

    private List<Integer> parseSemanticVersion(String versionString) {
        return Arrays.stream(versionString.split("\\.")).map(Integer::parseInt).collect(Collectors.toList());
    }

    private boolean validateSemanticVersion(String version) {
        return SEMVER_PATTERN.matcher(version).matches();
    }
}

