/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server;

import ghidra.framework.store.local.LocalFileSystem;
import ghidra.server.Repository;
import ghidra.server.RepositoryManager;
import ghidra.server.UserManager;
import ghidra.util.exception.DuplicateNameException;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import javax.security.auth.x500.X500Principal;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import utilities.util.FileUtilities;

public class CommandProcessor {
    static final Logger log = LogManager.getLogger(CommandProcessor.class);
    static final String ADD_USER_COMMAND = "-add";
    static final String REMOVE_USER_COMMAND = "-remove";
    static final String RESET_USER_COMMAND = "-reset";
    static final String SET_USER_DN_COMMAND = "-dn";
    static final String GRANT_USER_COMMAND = "-grant";
    static final String REVOKE_USER_COMMAND = "-revoke";
    static final String PASSWORD_OPTION = "--p";
    private static final String ADMIN_CMD_DIR = LocalFileSystem.HIDDEN_DIR_PREFIX + "admin";
    private static final String COMMAND_FILE_EXT = ".cmd";
    static final FileFilter CMD_FILE_FILTER = f -> f.isFile() && f.getName().endsWith(COMMAND_FILE_EXT);
    static final Comparator<File> FILE_DATE_COMPARATOR = (f1, f2) -> {
        long t2;
        long t1 = f1.lastModified();
        long diff = t1 - (t2 = f2.lastModified());
        if (diff == 0L) {
            return 0;
        }
        return diff < 0L ? -1 : 1;
    };

    private CommandProcessor() {
    }

    private static String[] splitCommand(String cmd) {
        ArrayList<String> argList = new ArrayList<String>();
        int startIx = 0;
        int endIx = 0;
        int len = cmd.length();
        boolean insideQuote = false;
        while (endIx < len) {
            char c = cmd.charAt(endIx);
            if (!(insideQuote || startIx != endIx || c != ' ' && c != '\"')) {
                insideQuote = c == '\"';
                startIx = ++endIx;
                continue;
            }
            if (c == (insideQuote ? (char)'\"' : ' ')) {
                argList.add(cmd.substring(startIx, endIx));
                startIx = ++endIx;
                insideQuote = false;
                continue;
            }
            ++endIx;
        }
        if (startIx != endIx) {
            argList.add(cmd.substring(startIx, endIx));
        }
        String[] args = new String[argList.size()];
        argList.toArray(args);
        return args;
    }

    private static void processCommand(RepositoryManager repositoryMgr, String cmd) throws IOException {
        UserManager userMgr = repositoryMgr.getUserManager();
        String[] args = CommandProcessor.splitCommand(cmd);
        switch (args[0]) {
            case "-add": {
                String sid = args[1];
                char[] pwdHash = null;
                if (args.length == 4 && args[2].contentEquals(PASSWORD_OPTION)) {
                    pwdHash = args[3].toCharArray();
                }
                try {
                    userMgr.addUser(sid, pwdHash);
                }
                catch (DuplicateNameException e) {
                    log.error("Add User Failed: " + e.getMessage());
                }
                break;
            }
            case "-remove": {
                String sid = args[1];
                if (userMgr.removeUser(sid)) break;
                log.info("User not found: '" + sid + "'");
                break;
            }
            case "-reset": {
                String sid = args[1];
                char[] pwdHash = null;
                if (args.length == 4 && args[2].contentEquals(PASSWORD_OPTION)) {
                    pwdHash = args[3].toCharArray();
                }
                if (!userMgr.resetPassword(sid, pwdHash)) {
                    log.info("Failed to reset password for user '" + sid + "'");
                    break;
                }
                if (pwdHash != null) {
                    log.info("User '" + sid + "' password reset to specified password");
                    break;
                }
                log.info("User '" + sid + "' password reset to default password");
                break;
            }
            case "-dn": {
                String sid = args[1];
                X500Principal x500User = new X500Principal(args[2]);
                if (userMgr.isValidUser(sid)) {
                    userMgr.setDistinguishedName(sid, x500User);
                } else {
                    try {
                        userMgr.addUser(sid, x500User);
                    }
                    catch (DuplicateNameException e) {
                        log.error("Add User Failed: " + e.getMessage());
                        return;
                    }
                }
                log.info("User '" + sid + "' DN set (" + x500User.getName() + ")");
                break;
            }
            case "-grant": {
                String sid = args[1];
                int permission = CommandProcessor.parsePermission(args[2]);
                String repName = args[3];
                if (!userMgr.isValidUser(sid)) {
                    log.error("Failed to grant access for '" + sid + "', user has not been added to server.");
                    return;
                }
                if (permission < 0) {
                    log.error("Failed to process grant command.  Invalid permission: " + args[2]);
                    return;
                }
                Repository rep = repositoryMgr.getRepository(repName);
                if (rep == null) {
                    log.error("Failed to grant access for '" + sid + "', repository '" + repName + "' not found.");
                    return;
                }
                rep.setUserPermission(sid, permission);
                break;
            }
            case "-revoke": {
                String sid = args[1];
                String repName = args[2];
                Repository rep = repositoryMgr.getRepository(repName);
                if (rep == null) {
                    log.error("Failed to revoke access for '" + sid + "', repository '" + repName + "' not found.");
                    return;
                }
                rep.removeUser(sid);
                break;
            }
            default: {
                log.error("Failed to process unrecognized command: " + args[0]);
            }
        }
    }

    static int parsePermission(String permissionStr) {
        if ("+r".equals(permissionStr)) {
            return 0;
        }
        if ("+w".equals(permissionStr)) {
            return 1;
        }
        if ("+a".equals(permissionStr)) {
            return 2;
        }
        return -1;
    }

    static File getCommandDir(File serverRootDir) {
        return new File(serverRootDir, ADMIN_CMD_DIR);
    }

    static File getOrCreateCommandDir(RepositoryManager repositoryMgr) {
        File cmdDir = CommandProcessor.getCommandDir(repositoryMgr.getRootDir());
        if (!cmdDir.exists()) {
            cmdDir.mkdir();
        }
        return cmdDir;
    }

    static void processCommands(RepositoryManager repositoryMgr) throws IOException {
        File cmdDir = CommandProcessor.getOrCreateCommandDir(repositoryMgr);
        File[] files = cmdDir.listFiles(CMD_FILE_FILTER);
        if (files == null) {
            log.error("Failed to access command queue " + cmdDir.getAbsolutePath() + ": possible permission problem");
            return;
        }
        if (files.length == 0) {
            return;
        }
        log.info("Processing queued commands");
        Arrays.sort(files, FILE_DATE_COMPARATOR);
        for (File file : files) {
            List cmdList = FileUtilities.getLines((File)file);
            for (String cmdStr : cmdList) {
                if (cmdStr.isBlank()) continue;
                try {
                    CommandProcessor.processCommand(repositoryMgr, cmdStr.trim());
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    log.error("Error occured processing command: " + cmdStr);
                }
            }
            file.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void writeCommands(List<String> cmdList, File cmdDir) throws IOException {
        File cmdTempFile = null;
        try {
            cmdTempFile = File.createTempFile("adm", ".tmp", cmdDir);
            FileUtils.writeLines((File)cmdTempFile, cmdList);
            Object cmdFilename = cmdTempFile.getName();
            cmdFilename = ((String)cmdFilename).substring(0, ((String)cmdFilename).length() - 4) + COMMAND_FILE_EXT;
            File cmdFile = new File(cmdTempFile.getParentFile(), (String)cmdFilename);
            if (!cmdTempFile.renameTo(cmdFile)) {
                throw new IOException("file error");
            }
            cmdTempFile = null;
        }
        finally {
            if (cmdTempFile != null) {
                cmdTempFile.delete();
            }
        }
    }
}

