﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2Request.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2_EXPORTS.h>
#include <aws/kinesisanalyticsv2/model/VpcConfiguration.h>

#include <utility>

namespace Aws {
namespace KinesisAnalyticsV2 {
namespace Model {

/**
 */
class AddApplicationVpcConfigurationRequest : public KinesisAnalyticsV2Request {
 public:
  AWS_KINESISANALYTICSV2_API AddApplicationVpcConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddApplicationVpcConfiguration"; }

  AWS_KINESISANALYTICSV2_API Aws::String SerializePayload() const override;

  AWS_KINESISANALYTICSV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of an existing application.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  AddApplicationVpcConfigurationRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the application to which you want to add the VPC
   * configuration. You must provide the <code>CurrentApplicationVersionId</code> or
   * the <code>ConditionalToken</code>. You can use the <a>DescribeApplication</a>
   * operation to get the current application version. If the version specified is
   * not the current version, the <code>ConcurrentModificationException</code> is
   * returned. For better concurrency support, use the <code>ConditionalToken</code>
   * parameter instead of <code>CurrentApplicationVersionId</code>.</p>
   */
  inline long long GetCurrentApplicationVersionId() const { return m_currentApplicationVersionId; }
  inline bool CurrentApplicationVersionIdHasBeenSet() const { return m_currentApplicationVersionIdHasBeenSet; }
  inline void SetCurrentApplicationVersionId(long long value) {
    m_currentApplicationVersionIdHasBeenSet = true;
    m_currentApplicationVersionId = value;
  }
  inline AddApplicationVpcConfigurationRequest& WithCurrentApplicationVersionId(long long value) {
    SetCurrentApplicationVersionId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Description of the VPC to add to the application.</p>
   */
  inline const VpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = VpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = VpcConfiguration>
  AddApplicationVpcConfigurationRequest& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value you use to implement strong concurrency for application updates. You
   * must provide the <code>ApplicationVersionID</code> or the
   * <code>ConditionalToken</code>. You get the application's current
   * <code>ConditionalToken</code> using <a>DescribeApplication</a>. For better
   * concurrency support, use the <code>ConditionalToken</code> parameter instead of
   * <code>CurrentApplicationVersionId</code>.</p>
   */
  inline const Aws::String& GetConditionalToken() const { return m_conditionalToken; }
  inline bool ConditionalTokenHasBeenSet() const { return m_conditionalTokenHasBeenSet; }
  template <typename ConditionalTokenT = Aws::String>
  void SetConditionalToken(ConditionalTokenT&& value) {
    m_conditionalTokenHasBeenSet = true;
    m_conditionalToken = std::forward<ConditionalTokenT>(value);
  }
  template <typename ConditionalTokenT = Aws::String>
  AddApplicationVpcConfigurationRequest& WithConditionalToken(ConditionalTokenT&& value) {
    SetConditionalToken(std::forward<ConditionalTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;
  bool m_applicationNameHasBeenSet = false;

  long long m_currentApplicationVersionId{0};
  bool m_currentApplicationVersionIdHasBeenSet = false;

  VpcConfiguration m_vpcConfiguration;
  bool m_vpcConfigurationHasBeenSet = false;

  Aws::String m_conditionalToken;
  bool m_conditionalTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalyticsV2
}  // namespace Aws
