﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/discovery/ApplicationDiscoveryServiceRequest.h>
#include <aws/discovery/ApplicationDiscoveryService_EXPORTS.h>
#include <aws/discovery/model/Filter.h>

#include <utility>

namespace Aws {
namespace ApplicationDiscoveryService {
namespace Model {

/**
 */
class DescribeAgentsRequest : public ApplicationDiscoveryServiceRequest {
 public:
  AWS_APPLICATIONDISCOVERYSERVICE_API DescribeAgentsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeAgents"; }

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The agent or the collector IDs for which you want information. If you specify
   * no IDs, the system returns information about all agents/collectors associated
   * with your user.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAgentIds() const { return m_agentIds; }
  inline bool AgentIdsHasBeenSet() const { return m_agentIdsHasBeenSet; }
  template <typename AgentIdsT = Aws::Vector<Aws::String>>
  void SetAgentIds(AgentIdsT&& value) {
    m_agentIdsHasBeenSet = true;
    m_agentIds = std::forward<AgentIdsT>(value);
  }
  template <typename AgentIdsT = Aws::Vector<Aws::String>>
  DescribeAgentsRequest& WithAgentIds(AgentIdsT&& value) {
    SetAgentIds(std::forward<AgentIdsT>(value));
    return *this;
  }
  template <typename AgentIdsT = Aws::String>
  DescribeAgentsRequest& AddAgentIds(AgentIdsT&& value) {
    m_agentIdsHasBeenSet = true;
    m_agentIds.emplace_back(std::forward<AgentIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>You can filter the request using various logical operators and a
   * <i>key</i>-<i>value</i> format. For example: </p> <p> <code>{"key":
   * "collectionStatus", "value": "STARTED"}</code> </p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeAgentsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeAgentsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total number of agents/collectors to return in a single page of output.
   * The maximum value is 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeAgentsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Token to retrieve the next set of results. For example, if you previously
   * specified 100 IDs for <code>DescribeAgentsRequest$agentIds</code> but set
   * <code>DescribeAgentsRequest$maxResults</code> to 10, you received a set of 10
   * results along with a token. Use that token in this query to get the next set of
   * 10.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeAgentsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_agentIds;
  bool m_agentIdsHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationDiscoveryService
}  // namespace Aws
