﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class AcceptReservedNodeExchangeRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API AcceptReservedNodeExchangeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AcceptReservedNodeExchange"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A string representing the node identifier of the DC1 Reserved Node to be
   * exchanged.</p>
   */
  inline const Aws::String& GetReservedNodeId() const { return m_reservedNodeId; }
  inline bool ReservedNodeIdHasBeenSet() const { return m_reservedNodeIdHasBeenSet; }
  template <typename ReservedNodeIdT = Aws::String>
  void SetReservedNodeId(ReservedNodeIdT&& value) {
    m_reservedNodeIdHasBeenSet = true;
    m_reservedNodeId = std::forward<ReservedNodeIdT>(value);
  }
  template <typename ReservedNodeIdT = Aws::String>
  AcceptReservedNodeExchangeRequest& WithReservedNodeId(ReservedNodeIdT&& value) {
    SetReservedNodeId(std::forward<ReservedNodeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the DC2 Reserved Node offering to be used for the
   * exchange. You can obtain the value for the parameter by calling
   * <a>GetReservedNodeExchangeOfferings</a> </p>
   */
  inline const Aws::String& GetTargetReservedNodeOfferingId() const { return m_targetReservedNodeOfferingId; }
  inline bool TargetReservedNodeOfferingIdHasBeenSet() const { return m_targetReservedNodeOfferingIdHasBeenSet; }
  template <typename TargetReservedNodeOfferingIdT = Aws::String>
  void SetTargetReservedNodeOfferingId(TargetReservedNodeOfferingIdT&& value) {
    m_targetReservedNodeOfferingIdHasBeenSet = true;
    m_targetReservedNodeOfferingId = std::forward<TargetReservedNodeOfferingIdT>(value);
  }
  template <typename TargetReservedNodeOfferingIdT = Aws::String>
  AcceptReservedNodeExchangeRequest& WithTargetReservedNodeOfferingId(TargetReservedNodeOfferingIdT&& value) {
    SetTargetReservedNodeOfferingId(std::forward<TargetReservedNodeOfferingIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_reservedNodeId;

  Aws::String m_targetReservedNodeOfferingId;
  bool m_reservedNodeIdHasBeenSet = false;
  bool m_targetReservedNodeOfferingIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
