#
# Copyright (C) 2018 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows various methods of getting attributes from the data set
# inside the Advisor project.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
data = project.load(advisor.ALL)


def test_getattr_bottomup():
    global data

    print("START test_getattr_bottomup")
    try:
        for row in data.bottomup:
            for col in row:
                print("{:.<45}: {}".format(col, getattr(row, col)))
    except:
        print("FAIL. test_getattr_bottomup" + "\n")

        return False
    return True


def test_getattr_bottomup_flat():
    global data

    print("START test_getattr_bottomup_flat")
    try:
        for row in data.bottomup_flat:
            for col in row:
                print("{:.<45}: {}".format(col, getattr(row, col)))
    except:
        print("FAIL. test_getattr_bottomup_flat" + "\n")

        return False
    return True


def test_getattr_bottomup_full():
    global data

    print("START test_getattr_bottomup_full")
    try:
        for row in data.bottomup_full:
            for col in row:
                print("{:.<45}: {}".format(col, getattr(row, col)))
    except:
        print("FAIL. test_getattr_bottomup_full" + "\n")

        return False
    return True


def test_getattr_topdown():
    global data

    print("START test_getattr_topdown")
    try:
        for row in data.topdown:
            for col in row:
                print("{:.<45}: {}".format(col, getattr(row, col)))
    except:
        print("FAIL. test_getattr_topdown" + "\n")

        return False
    return True


def test_getattr_dependencies():
    global data

    print("START test_getattr_dependencies")
    try:
        for site in data.dependencies:
            print("\n" + "SITE: {}, {}".format(site.site_name, site.source))
            for problem in site.problems:
                print(" " * 4 + "PROBLEM: {}, {}".format(problem.id, problem.type))
                for observation in problem.observations:
                    print(" " * 8 + "OBSERVATION: {}, {}".format(observation.id, observation.description))
    except:
        print("FAIL. test_getattr_dependencies" + "\n")

        return False
    return True


def test_getattr_map():
    global data

    print("START test_getattr_map")
    try:
        for row in data.map:
            for col in row:
                print("{:.<45}: {}".format(col, getattr(row, col)))
    except:
        print("FAIL. test_getattr_map" + "\n")

        return False
    return True


if __name__ == "__main__":

    num_of_tests = 6
    failed_tests = []

    if not test_getattr_bottomup():
        failed_tests.append("test_getattr_bottomup")
    elif not test_getattr_bottomup_flat():
        failed_tests.append("test_getattr_bottomup_flat")
    elif not test_getattr_bottomup_full():
        failed_tests.append("test_getattr_bottomup_full")
    elif not test_getattr_topdown():
        failed_tests.append("test_getattr_topdown")
    elif not test_getattr_dependencies():
        failed_tests.append("test_getattr_dependencies")
    elif not test_getattr_map():
        failed_tests.append("test_getattr_map")

    print("\n" + "The result of getattr testing is:")
    print("PASSED tests : {}".format(num_of_tests - len(failed_tests)))
    print("FAILED tests : {}".format(len(failed_tests)))

    if failed_tests:
        print("\n" + "List of failed tests:")
        for test in failed_tests:
            print(test)
