#
# Copyright (C) 2017 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to collect roofline data programatically with
# Advisor PythonAPI.
# ------------------------------------------------------------------------------

import os
import sys

try:

    import advisor

except ImportError:

    print(
        """Import Error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 3:
    print('Usage: "python {} path_to_project_dir path_to_binary"'.format(__file__))
    sys.exit(2)

# Create or open the Advisor Project, then collect survey data for the
# specified binary.
project_dir = sys.argv[1]
app = sys.argv[2]

if os.path.exists(project_dir):

    project = advisor.open_project(project_dir)

else:

    project = advisor.create_project(project_dir)

res = project.collect(advisor.ROOFLINE, app)

# Note that advisor.ROOFLINE is a shortcut for two different collections:
#
# (1) Collect Survey
# (2) Collect Tripcounts with FLOPS
#
# To get roofline with stacks or to specify other flags, avoid using the
# shortcut and manually collect (1) and (2) independently.
#
# res = project.collect(advisor.SURVEY, app, collection_args = ['stackwalk-mode=online', 'no-stack-stitching', ...])
#
# if res[0] == 0:
#
#     res = project.collect(advisor.TRIPCOUNTS, app, collection_args = ['flop', 'stacks', 'auto-finalize'])

# Collection successful.
if res[0] == 0:

    # Load the survey results and print the metrics.
    data = project.load(advisor.SURVEY)

    print(95 * "-")
    print(
        "{:^15}|{:^15}|{:^15}|{:^15}|{:^15}|{:^15}".format(
            "Loop ID", "Trip Count", "Self GFLOP", "Self GFLOPs", "Self Memory GB", "Self Time",
        )
    )
    print(95 * "-")

    for loop in data.bottomup:

        print(
            "{:>15}|{:>15}|{:>15}|{:>15}|{:>15}|{:>15}".format(
                loop["loop_function_id"],
                loop["average_trip_count"],
                loop["self_gflop"],
                loop["self_gflops"],
                loop["self_memory_gb"],
                loop["self_time"],
            )
        )

# Collection failed.
else:

    print("Error: Collection returned a non-zero return code.")
