﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesis/Kinesis_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Kinesis {
namespace Model {

/**
 * <p>Represents the output for <code>PutRecords</code>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/kinesis-2013-12-02/PutRecordsRequestEntry">AWS
 * API Reference</a></p>
 */
class PutRecordsRequestEntry {
 public:
  AWS_KINESIS_API PutRecordsRequestEntry() = default;
  AWS_KINESIS_API PutRecordsRequestEntry(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESIS_API PutRecordsRequestEntry& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_KINESIS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The data blob to put into the record, which is base64-encoded when the blob
   * is serialized. When the data blob (the payload before base64-encoding) is added
   * to the partition key size, the total size must not exceed the maximum record
   * size (1 MiB).</p>
   */
  inline const Aws::Utils::ByteBuffer& GetData() const { return m_data; }
  inline bool DataHasBeenSet() const { return m_dataHasBeenSet; }
  template <typename DataT = Aws::Utils::ByteBuffer>
  void SetData(DataT&& value) {
    m_dataHasBeenSet = true;
    m_data = std::forward<DataT>(value);
  }
  template <typename DataT = Aws::Utils::ByteBuffer>
  PutRecordsRequestEntry& WithData(DataT&& value) {
    SetData(std::forward<DataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The hash value used to determine explicitly the shard that the data record is
   * assigned to by overriding the partition key hash.</p>
   */
  inline const Aws::String& GetExplicitHashKey() const { return m_explicitHashKey; }
  inline bool ExplicitHashKeyHasBeenSet() const { return m_explicitHashKeyHasBeenSet; }
  template <typename ExplicitHashKeyT = Aws::String>
  void SetExplicitHashKey(ExplicitHashKeyT&& value) {
    m_explicitHashKeyHasBeenSet = true;
    m_explicitHashKey = std::forward<ExplicitHashKeyT>(value);
  }
  template <typename ExplicitHashKeyT = Aws::String>
  PutRecordsRequestEntry& WithExplicitHashKey(ExplicitHashKeyT&& value) {
    SetExplicitHashKey(std::forward<ExplicitHashKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines which shard in the stream the data record is assigned to.
   * Partition keys are Unicode strings with a maximum length limit of 256 characters
   * for each key. Amazon Kinesis Data Streams uses the partition key as input to a
   * hash function that maps the partition key and associated data to a specific
   * shard. Specifically, an MD5 hash function is used to map partition keys to
   * 128-bit integer values and to map associated data records to shards. As a result
   * of this hashing mechanism, all data records with the same partition key map to
   * the same shard within the stream.</p>
   */
  inline const Aws::String& GetPartitionKey() const { return m_partitionKey; }
  inline bool PartitionKeyHasBeenSet() const { return m_partitionKeyHasBeenSet; }
  template <typename PartitionKeyT = Aws::String>
  void SetPartitionKey(PartitionKeyT&& value) {
    m_partitionKeyHasBeenSet = true;
    m_partitionKey = std::forward<PartitionKeyT>(value);
  }
  template <typename PartitionKeyT = Aws::String>
  PutRecordsRequestEntry& WithPartitionKey(PartitionKeyT&& value) {
    SetPartitionKey(std::forward<PartitionKeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::ByteBuffer m_data{};
  bool m_dataHasBeenSet = false;

  Aws::String m_explicitHashKey;
  bool m_explicitHashKeyHasBeenSet = false;

  Aws::String m_partitionKey;
  bool m_partitionKeyHasBeenSet = false;
};

}  // namespace Model
}  // namespace Kinesis
}  // namespace Aws
