﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Tag.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/CreateOptionGroupMessage">AWS
 * API Reference</a></p>
 */
class CreateOptionGroupRequest : public RDSRequest {
 public:
  AWS_RDS_API CreateOptionGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateOptionGroup"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Specifies the name of the option group to be created.</p> <p>Constraints:</p>
   * <ul> <li> <p>Must be 1 to 255 letters, numbers, or hyphens</p> </li> <li>
   * <p>First character must be a letter</p> </li> <li> <p>Can't end with a hyphen or
   * contain two consecutive hyphens</p> </li> </ul> <p>Example:
   * <code>myoptiongroup</code> </p>
   */
  inline const Aws::String& GetOptionGroupName() const { return m_optionGroupName; }
  inline bool OptionGroupNameHasBeenSet() const { return m_optionGroupNameHasBeenSet; }
  template <typename OptionGroupNameT = Aws::String>
  void SetOptionGroupName(OptionGroupNameT&& value) {
    m_optionGroupNameHasBeenSet = true;
    m_optionGroupName = std::forward<OptionGroupNameT>(value);
  }
  template <typename OptionGroupNameT = Aws::String>
  CreateOptionGroupRequest& WithOptionGroupName(OptionGroupNameT&& value) {
    SetOptionGroupName(std::forward<OptionGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the engine to associate this option group with.</p> <p>Valid
   * Values:</p> <ul> <li> <p> <code>db2-ae</code> </p> </li> <li> <p>
   * <code>db2-se</code> </p> </li> <li> <p> <code>mariadb</code> </p> </li> <li> <p>
   * <code>mysql</code> </p> </li> <li> <p> <code>oracle-ee</code> </p> </li> <li>
   * <p> <code>oracle-ee-cdb</code> </p> </li> <li> <p> <code>oracle-se2</code> </p>
   * </li> <li> <p> <code>oracle-se2-cdb</code> </p> </li> <li> <p>
   * <code>postgres</code> </p> </li> <li> <p> <code>sqlserver-ee</code> </p> </li>
   * <li> <p> <code>sqlserver-se</code> </p> </li> <li> <p> <code>sqlserver-ex</code>
   * </p> </li> <li> <p> <code>sqlserver-web</code> </p> </li> </ul>
   */
  inline const Aws::String& GetEngineName() const { return m_engineName; }
  inline bool EngineNameHasBeenSet() const { return m_engineNameHasBeenSet; }
  template <typename EngineNameT = Aws::String>
  void SetEngineName(EngineNameT&& value) {
    m_engineNameHasBeenSet = true;
    m_engineName = std::forward<EngineNameT>(value);
  }
  template <typename EngineNameT = Aws::String>
  CreateOptionGroupRequest& WithEngineName(EngineNameT&& value) {
    SetEngineName(std::forward<EngineNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the major version of the engine that this option group should be
   * associated with.</p>
   */
  inline const Aws::String& GetMajorEngineVersion() const { return m_majorEngineVersion; }
  inline bool MajorEngineVersionHasBeenSet() const { return m_majorEngineVersionHasBeenSet; }
  template <typename MajorEngineVersionT = Aws::String>
  void SetMajorEngineVersion(MajorEngineVersionT&& value) {
    m_majorEngineVersionHasBeenSet = true;
    m_majorEngineVersion = std::forward<MajorEngineVersionT>(value);
  }
  template <typename MajorEngineVersionT = Aws::String>
  CreateOptionGroupRequest& WithMajorEngineVersion(MajorEngineVersionT&& value) {
    SetMajorEngineVersion(std::forward<MajorEngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the option group.</p>
   */
  inline const Aws::String& GetOptionGroupDescription() const { return m_optionGroupDescription; }
  inline bool OptionGroupDescriptionHasBeenSet() const { return m_optionGroupDescriptionHasBeenSet; }
  template <typename OptionGroupDescriptionT = Aws::String>
  void SetOptionGroupDescription(OptionGroupDescriptionT&& value) {
    m_optionGroupDescriptionHasBeenSet = true;
    m_optionGroupDescription = std::forward<OptionGroupDescriptionT>(value);
  }
  template <typename OptionGroupDescriptionT = Aws::String>
  CreateOptionGroupRequest& WithOptionGroupDescription(OptionGroupDescriptionT&& value) {
    SetOptionGroupDescription(std::forward<OptionGroupDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags to assign to the option group.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateOptionGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateOptionGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_optionGroupName;

  Aws::String m_engineName;

  Aws::String m_majorEngineVersion;

  Aws::String m_optionGroupDescription;

  Aws::Vector<Tag> m_tags;
  bool m_optionGroupNameHasBeenSet = false;
  bool m_engineNameHasBeenSet = false;
  bool m_majorEngineVersionHasBeenSet = false;
  bool m_optionGroupDescriptionHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
