﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/NodeAggregator.h>
#include <aws/ssm/model/NodeFilter.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class ListNodesSummaryRequest : public SSMRequest {
 public:
  AWS_SSM_API ListNodesSummaryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListNodesSummary"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the Amazon Web Services managed resource data sync to retrieve
   * information about.</p> <p>For cross-account/cross-Region configurations, this
   * parameter is required, and the name of the supported resource data sync is
   * <code>AWS-QuickSetup-ManagedNode</code>.</p> <p>For single account/single-Region
   * configurations, the parameter is not required.</p>
   */
  inline const Aws::String& GetSyncName() const { return m_syncName; }
  inline bool SyncNameHasBeenSet() const { return m_syncNameHasBeenSet; }
  template <typename SyncNameT = Aws::String>
  void SetSyncName(SyncNameT&& value) {
    m_syncNameHasBeenSet = true;
    m_syncName = std::forward<SyncNameT>(value);
  }
  template <typename SyncNameT = Aws::String>
  ListNodesSummaryRequest& WithSyncName(SyncNameT&& value) {
    SetSyncName(std::forward<SyncNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters. Use a filter to generate a summary that matches your
   * specified filter criteria.</p>
   */
  inline const Aws::Vector<NodeFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<NodeFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<NodeFilter>>
  ListNodesSummaryRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = NodeFilter>
  ListNodesSummaryRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify one or more aggregators to return a count of managed nodes that match
   * that expression. For example, a count of managed nodes by operating system.</p>
   */
  inline const Aws::Vector<NodeAggregator>& GetAggregators() const { return m_aggregators; }
  inline bool AggregatorsHasBeenSet() const { return m_aggregatorsHasBeenSet; }
  template <typename AggregatorsT = Aws::Vector<NodeAggregator>>
  void SetAggregators(AggregatorsT&& value) {
    m_aggregatorsHasBeenSet = true;
    m_aggregators = std::forward<AggregatorsT>(value);
  }
  template <typename AggregatorsT = Aws::Vector<NodeAggregator>>
  ListNodesSummaryRequest& WithAggregators(AggregatorsT&& value) {
    SetAggregators(std::forward<AggregatorsT>(value));
    return *this;
  }
  template <typename AggregatorsT = NodeAggregator>
  ListNodesSummaryRequest& AddAggregators(AggregatorsT&& value) {
    m_aggregatorsHasBeenSet = true;
    m_aggregators.emplace_back(std::forward<AggregatorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of items to return. (You received this token from
   * a previous call.) The call also returns a token that you can specify in a
   * subsequent call to get the next set of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListNodesSummaryRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return for this call. The call also returns a
   * token that you can specify in a subsequent call to get the next set of
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListNodesSummaryRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_syncName;
  bool m_syncNameHasBeenSet = false;

  Aws::Vector<NodeFilter> m_filters;
  bool m_filtersHasBeenSet = false;

  Aws::Vector<NodeAggregator> m_aggregators;
  bool m_aggregatorsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
