﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-contacts/SSMContactsRequest.h>
#include <aws/ssm-contacts/SSMContacts_EXPORTS.h>
#include <aws/ssm-contacts/model/RecurrenceSettings.h>
#include <aws/ssm-contacts/model/Tag.h>

#include <utility>

namespace Aws {
namespace SSMContacts {
namespace Model {

/**
 */
class CreateRotationRequest : public SSMContactsRequest {
 public:
  AWS_SSMCONTACTS_API CreateRotationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRotation"; }

  AWS_SSMCONTACTS_API Aws::String SerializePayload() const override;

  AWS_SSMCONTACTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the rotation.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRotationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Names (ARNs) of the contacts to add to the rotation.</p>
   *  <p>Only the <code>PERSONAL</code> contact type is supported. The contact
   * types <code>ESCALATION</code> and <code>ONCALL_SCHEDULE</code> are not supported
   * for this operation. </p>  <p>The order that you list the contacts in is
   * their shift order in the rotation schedule. To change the order of the contact's
   * shifts, use the <a>UpdateRotation</a> operation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetContactIds() const { return m_contactIds; }
  inline bool ContactIdsHasBeenSet() const { return m_contactIdsHasBeenSet; }
  template <typename ContactIdsT = Aws::Vector<Aws::String>>
  void SetContactIds(ContactIdsT&& value) {
    m_contactIdsHasBeenSet = true;
    m_contactIds = std::forward<ContactIdsT>(value);
  }
  template <typename ContactIdsT = Aws::Vector<Aws::String>>
  CreateRotationRequest& WithContactIds(ContactIdsT&& value) {
    SetContactIds(std::forward<ContactIdsT>(value));
    return *this;
  }
  template <typename ContactIdsT = Aws::String>
  CreateRotationRequest& AddContactIds(ContactIdsT&& value) {
    m_contactIdsHasBeenSet = true;
    m_contactIds.emplace_back(std::forward<ContactIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that the rotation goes into effect.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  CreateRotationRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time zone to base the rotation’s activity on in Internet Assigned Numbers
   * Authority (IANA) format. For example: "America/Los_Angeles", "UTC", or
   * "Asia/Seoul". For more information, see the <a
   * href="https://www.iana.org/time-zones">Time Zone Database</a> on the IANA
   * website.</p>  <p>Designators for time zones that don’t support Daylight
   * Savings Time rules, such as Pacific Standard Time (PST), are not supported.</p>
   *
   */
  inline const Aws::String& GetTimeZoneId() const { return m_timeZoneId; }
  inline bool TimeZoneIdHasBeenSet() const { return m_timeZoneIdHasBeenSet; }
  template <typename TimeZoneIdT = Aws::String>
  void SetTimeZoneId(TimeZoneIdT&& value) {
    m_timeZoneIdHasBeenSet = true;
    m_timeZoneId = std::forward<TimeZoneIdT>(value);
  }
  template <typename TimeZoneIdT = Aws::String>
  CreateRotationRequest& WithTimeZoneId(TimeZoneIdT&& value) {
    SetTimeZoneId(std::forward<TimeZoneIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the rule that specifies when a shift's team members
   * rotate.</p>
   */
  inline const RecurrenceSettings& GetRecurrence() const { return m_recurrence; }
  inline bool RecurrenceHasBeenSet() const { return m_recurrenceHasBeenSet; }
  template <typename RecurrenceT = RecurrenceSettings>
  void SetRecurrence(RecurrenceT&& value) {
    m_recurrenceHasBeenSet = true;
    m_recurrence = std::forward<RecurrenceT>(value);
  }
  template <typename RecurrenceT = RecurrenceSettings>
  CreateRotationRequest& WithRecurrence(RecurrenceT&& value) {
    SetRecurrence(std::forward<RecurrenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional metadata to assign to the rotation. Tags enable you to categorize a
   * resource in different ways, such as by purpose, owner, or environment. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/incident-manager/latest/userguide/tagging.html">Tagging
   * Incident Manager resources</a> in the <i>Incident Manager User Guide</i>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateRotationRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateRotationRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that ensures that the operation is called only once with the
   * specified details.</p>
   */
  inline const Aws::String& GetIdempotencyToken() const { return m_idempotencyToken; }
  inline bool IdempotencyTokenHasBeenSet() const { return m_idempotencyTokenHasBeenSet; }
  template <typename IdempotencyTokenT = Aws::String>
  void SetIdempotencyToken(IdempotencyTokenT&& value) {
    m_idempotencyTokenHasBeenSet = true;
    m_idempotencyToken = std::forward<IdempotencyTokenT>(value);
  }
  template <typename IdempotencyTokenT = Aws::String>
  CreateRotationRequest& WithIdempotencyToken(IdempotencyTokenT&& value) {
    SetIdempotencyToken(std::forward<IdempotencyTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Vector<Aws::String> m_contactIds;
  bool m_contactIdsHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::String m_timeZoneId;
  bool m_timeZoneIdHasBeenSet = false;

  RecurrenceSettings m_recurrence;
  bool m_recurrenceHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_idempotencyToken;
  bool m_idempotencyTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMContacts
}  // namespace Aws
