﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/event/EventStreamDecoder.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/qbusiness/QBusinessRequest.h>
#include <aws/qbusiness/QBusiness_EXPORTS.h>
#include <aws/qbusiness/model/ChatHandler.h>
#include <aws/qbusiness/model/ChatInputStream.h>

#include <memory>
#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace QBusiness {
namespace Model {

/**
 */
class ChatRequest : public QBusinessRequest {
 public:
  AWS_QBUSINESS_API ChatRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "Chat"; }

  inline virtual bool IsEventStreamRequest() const override { return true; }
  inline virtual bool HasEventStreamResponse() const override { return true; }
  // SerializePayload will not be invoked.
  // This request is sent by encoding its data in event-streams which is sent as IOStream via GetBody()
  AWS_QBUSINESS_API Aws::String SerializePayload() const override { return {}; }
  AWS_QBUSINESS_API std::shared_ptr<Aws::IOStream> GetBody() const override;
  AWS_QBUSINESS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  /**
   * Underlying Event Stream Decoder.
   */
  inline Aws::Utils::Event::EventStreamDecoder& GetEventStreamDecoder() { return m_decoder; }

  /**
   * Underlying Event Stream Handler which is used to define callback functions.
   */
  inline ChatHandler& GetEventStreamHandler() { return m_handler; }

  /**
   * Underlying Event Stream Handler which is used to define callback functions.
   */
  inline void SetEventStreamHandler(const ChatHandler& value) {
    m_handler = value;
    m_decoder.ResetEventStreamHandler(&m_handler);
  }

  /**
   * Underlying Event Stream Handler which is used to define callback functions.
   */
  inline ChatRequest& WithEventStreamHandler(const ChatHandler& value) {
    SetEventStreamHandler(value);
    return *this;
  }

  ///@{
  /**
   * <p>The identifier of the Amazon Q Business application linked to a streaming
   * Amazon Q Business conversation.</p>
   */
  inline const Aws::String& GetApplicationId() const { return m_applicationId; }
  inline bool ApplicationIdHasBeenSet() const { return m_applicationIdHasBeenSet; }
  template <typename ApplicationIdT = Aws::String>
  void SetApplicationId(ApplicationIdT&& value) {
    m_applicationIdHasBeenSet = true;
    m_applicationId = std::forward<ApplicationIdT>(value);
  }
  template <typename ApplicationIdT = Aws::String>
  ChatRequest& WithApplicationId(ApplicationIdT&& value) {
    SetApplicationId(std::forward<ApplicationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the user attached to the chat input. </p>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  ChatRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The group names that a user associated with the chat input belongs to.</p>
   */
  inline const Aws::Vector<Aws::String>& GetUserGroups() const { return m_userGroups; }
  inline bool UserGroupsHasBeenSet() const { return m_userGroupsHasBeenSet; }
  template <typename UserGroupsT = Aws::Vector<Aws::String>>
  void SetUserGroups(UserGroupsT&& value) {
    m_userGroupsHasBeenSet = true;
    m_userGroups = std::forward<UserGroupsT>(value);
  }
  template <typename UserGroupsT = Aws::Vector<Aws::String>>
  ChatRequest& WithUserGroups(UserGroupsT&& value) {
    SetUserGroups(std::forward<UserGroupsT>(value));
    return *this;
  }
  template <typename UserGroupsT = Aws::String>
  ChatRequest& AddUserGroups(UserGroupsT&& value) {
    m_userGroupsHasBeenSet = true;
    m_userGroups.emplace_back(std::forward<UserGroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Q Business conversation.</p>
   */
  inline const Aws::String& GetConversationId() const { return m_conversationId; }
  inline bool ConversationIdHasBeenSet() const { return m_conversationIdHasBeenSet; }
  template <typename ConversationIdT = Aws::String>
  void SetConversationId(ConversationIdT&& value) {
    m_conversationIdHasBeenSet = true;
    m_conversationId = std::forward<ConversationIdT>(value);
  }
  template <typename ConversationIdT = Aws::String>
  ChatRequest& WithConversationId(ConversationIdT&& value) {
    SetConversationId(std::forward<ConversationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier used to associate a user message with a AI generated
   * response.</p>
   */
  inline const Aws::String& GetParentMessageId() const { return m_parentMessageId; }
  inline bool ParentMessageIdHasBeenSet() const { return m_parentMessageIdHasBeenSet; }
  template <typename ParentMessageIdT = Aws::String>
  void SetParentMessageId(ParentMessageIdT&& value) {
    m_parentMessageIdHasBeenSet = true;
    m_parentMessageId = std::forward<ParentMessageIdT>(value);
  }
  template <typename ParentMessageIdT = Aws::String>
  ChatRequest& WithParentMessageId(ParentMessageIdT&& value) {
    SetParentMessageId(std::forward<ParentMessageIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that you provide to identify the chat input.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  ChatRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The streaming input for the <code>Chat</code> API.</p>
   */
  AWS_QBUSINESS_API std::shared_ptr<ChatInputStream> GetInputStream() const { return m_inputStream; }
  AWS_QBUSINESS_API void SetInputStream(const std::shared_ptr<ChatInputStream>& value) {
    m_inputStreamHasBeenSet = true;
    m_inputStream = value;
  }
  AWS_QBUSINESS_API ChatRequest& WithInputStream(const std::shared_ptr<ChatInputStream>& value) {
    SetInputStream(value);
    return *this;
  }

  ///@}
 private:
  Aws::String m_applicationId;
  bool m_applicationIdHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::Vector<Aws::String> m_userGroups;
  bool m_userGroupsHasBeenSet = false;

  Aws::String m_conversationId;
  bool m_conversationIdHasBeenSet = false;

  Aws::String m_parentMessageId;
  bool m_parentMessageIdHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  std::shared_ptr<ChatInputStream> m_inputStream;
  bool m_inputStreamHasBeenSet = false;
  ChatHandler m_handler;
  Aws::Utils::Event::EventStreamDecoder m_decoder{Utils::Event::EventStreamDecoder(&m_handler)};
};

}  // namespace Model
}  // namespace QBusiness
}  // namespace Aws
