﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/managedblockchain/ManagedBlockchainRequest.h>
#include <aws/managedblockchain/ManagedBlockchain_EXPORTS.h>
#include <aws/managedblockchain/model/NodeStatus.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace ManagedBlockchain {
namespace Model {

/**
 */
class ListNodesRequest : public ManagedBlockchainRequest {
 public:
  AWS_MANAGEDBLOCKCHAIN_API ListNodesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListNodes"; }

  AWS_MANAGEDBLOCKCHAIN_API Aws::String SerializePayload() const override;

  AWS_MANAGEDBLOCKCHAIN_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The unique identifier of the network for which to list nodes.</p>
   */
  inline const Aws::String& GetNetworkId() const { return m_networkId; }
  inline bool NetworkIdHasBeenSet() const { return m_networkIdHasBeenSet; }
  template <typename NetworkIdT = Aws::String>
  void SetNetworkId(NetworkIdT&& value) {
    m_networkIdHasBeenSet = true;
    m_networkId = std::forward<NetworkIdT>(value);
  }
  template <typename NetworkIdT = Aws::String>
  ListNodesRequest& WithNetworkId(NetworkIdT&& value) {
    SetNetworkId(std::forward<NetworkIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the member who owns the nodes to list.</p>
   * <p>Applies only to Hyperledger Fabric and is required for Hyperledger
   * Fabric.</p>
   */
  inline const Aws::String& GetMemberId() const { return m_memberId; }
  inline bool MemberIdHasBeenSet() const { return m_memberIdHasBeenSet; }
  template <typename MemberIdT = Aws::String>
  void SetMemberId(MemberIdT&& value) {
    m_memberIdHasBeenSet = true;
    m_memberId = std::forward<MemberIdT>(value);
  }
  template <typename MemberIdT = Aws::String>
  ListNodesRequest& WithMemberId(MemberIdT&& value) {
    SetMemberId(std::forward<MemberIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional status specifier. If provided, only nodes currently in this
   * status are listed.</p>
   */
  inline NodeStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(NodeStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListNodesRequest& WithStatus(NodeStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of nodes to list.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListNodesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token that indicates the next set of results to retrieve.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListNodesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_networkId;
  bool m_networkIdHasBeenSet = false;

  Aws::String m_memberId;
  bool m_memberIdHasBeenSet = false;

  NodeStatus m_status{NodeStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ManagedBlockchain
}  // namespace Aws
