﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/globalaccelerator/GlobalAcceleratorRequest.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>

#include <utility>

namespace Aws {
namespace GlobalAccelerator {
namespace Model {

/**
 */
class ListCustomRoutingPortMappingsRequest : public GlobalAcceleratorRequest {
 public:
  AWS_GLOBALACCELERATOR_API ListCustomRoutingPortMappingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListCustomRoutingPortMappings"; }

  AWS_GLOBALACCELERATOR_API Aws::String SerializePayload() const override;

  AWS_GLOBALACCELERATOR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the accelerator to list the custom routing
   * port mappings for.</p>
   */
  inline const Aws::String& GetAcceleratorArn() const { return m_acceleratorArn; }
  inline bool AcceleratorArnHasBeenSet() const { return m_acceleratorArnHasBeenSet; }
  template <typename AcceleratorArnT = Aws::String>
  void SetAcceleratorArn(AcceleratorArnT&& value) {
    m_acceleratorArnHasBeenSet = true;
    m_acceleratorArn = std::forward<AcceleratorArnT>(value);
  }
  template <typename AcceleratorArnT = Aws::String>
  ListCustomRoutingPortMappingsRequest& WithAcceleratorArn(AcceleratorArnT&& value) {
    SetAcceleratorArn(std::forward<AcceleratorArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the endpoint group to list the custom
   * routing port mappings for.</p>
   */
  inline const Aws::String& GetEndpointGroupArn() const { return m_endpointGroupArn; }
  inline bool EndpointGroupArnHasBeenSet() const { return m_endpointGroupArnHasBeenSet; }
  template <typename EndpointGroupArnT = Aws::String>
  void SetEndpointGroupArn(EndpointGroupArnT&& value) {
    m_endpointGroupArnHasBeenSet = true;
    m_endpointGroupArn = std::forward<EndpointGroupArnT>(value);
  }
  template <typename EndpointGroupArnT = Aws::String>
  ListCustomRoutingPortMappingsRequest& WithEndpointGroupArn(EndpointGroupArnT&& value) {
    SetEndpointGroupArn(std::forward<EndpointGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of destination port mappings that you want to return with this
   * call. The default value is 10.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListCustomRoutingPortMappingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of results. You receive this token from a previous
   * call.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListCustomRoutingPortMappingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_acceleratorArn;
  bool m_acceleratorArnHasBeenSet = false;

  Aws::String m_endpointGroupArn;
  bool m_endpointGroupArnHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
