﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticache/ElastiCacheRequest.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the input of a <code>DescribeEngineDefaultParameters</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/DescribeEngineDefaultParametersMessage">AWS
 * API Reference</a></p>
 */
class DescribeEngineDefaultParametersRequest : public ElastiCacheRequest {
 public:
  AWS_ELASTICACHE_API DescribeEngineDefaultParametersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeEngineDefaultParameters"; }

  AWS_ELASTICACHE_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICACHE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the cache parameter group family.</p> <p>Valid values are:
   * <code>memcached1.4</code> | <code>memcached1.5</code> |
   * <code>memcached1.6</code> | <code>redis2.6</code> | <code>redis2.8</code> |
   * <code>redis3.2</code> | <code>redis4.0</code> | <code>redis5.0</code> |
   * <code>redis6.x</code> | <code>redis6.2</code> | <code>redis7</code> </p>
   */
  inline const Aws::String& GetCacheParameterGroupFamily() const { return m_cacheParameterGroupFamily; }
  inline bool CacheParameterGroupFamilyHasBeenSet() const { return m_cacheParameterGroupFamilyHasBeenSet; }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  void SetCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    m_cacheParameterGroupFamilyHasBeenSet = true;
    m_cacheParameterGroupFamily = std::forward<CacheParameterGroupFamilyT>(value);
  }
  template <typename CacheParameterGroupFamilyT = Aws::String>
  DescribeEngineDefaultParametersRequest& WithCacheParameterGroupFamily(CacheParameterGroupFamilyT&& value) {
    SetCacheParameterGroupFamily(std::forward<CacheParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a marker is included in
   * the response so that the remaining results can be retrieved.</p> <p>Default:
   * 100</p> <p>Constraints: minimum 20; maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeEngineDefaultParametersRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional marker returned from a prior request. Use this marker for
   * pagination of results from this operation. If this parameter is specified, the
   * response includes only records beyond the marker, up to the value specified by
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeEngineDefaultParametersRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cacheParameterGroupFamily;
  bool m_cacheParameterGroupFamilyHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
