﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock-agent/BedrockAgentRequest.h>
#include <aws/bedrock-agent/BedrockAgent_EXPORTS.h>
#include <aws/bedrock-agent/model/DataDeletionPolicy.h>
#include <aws/bedrock-agent/model/DataSourceConfiguration.h>
#include <aws/bedrock-agent/model/ServerSideEncryptionConfiguration.h>
#include <aws/bedrock-agent/model/VectorIngestionConfiguration.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace BedrockAgent {
namespace Model {

/**
 */
class CreateDataSourceRequest : public BedrockAgentRequest {
 public:
  AWS_BEDROCKAGENT_API CreateDataSourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDataSource"; }

  AWS_BEDROCKAGENT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier of the knowledge base to which to add the data
   * source.</p>
   */
  inline const Aws::String& GetKnowledgeBaseId() const { return m_knowledgeBaseId; }
  inline bool KnowledgeBaseIdHasBeenSet() const { return m_knowledgeBaseIdHasBeenSet; }
  template <typename KnowledgeBaseIdT = Aws::String>
  void SetKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    m_knowledgeBaseIdHasBeenSet = true;
    m_knowledgeBaseId = std::forward<KnowledgeBaseIdT>(value);
  }
  template <typename KnowledgeBaseIdT = Aws::String>
  CreateDataSourceRequest& WithKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    SetKnowledgeBaseId(std::forward<KnowledgeBaseIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier to ensure that the API request completes
   * no more than one time. If this token matches a previous request, Amazon Bedrock
   * ignores the request, but does not return an error. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateDataSourceRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the data source.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateDataSourceRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the data source.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateDataSourceRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The connection configuration for the data source.</p>
   */
  inline const DataSourceConfiguration& GetDataSourceConfiguration() const { return m_dataSourceConfiguration; }
  inline bool DataSourceConfigurationHasBeenSet() const { return m_dataSourceConfigurationHasBeenSet; }
  template <typename DataSourceConfigurationT = DataSourceConfiguration>
  void SetDataSourceConfiguration(DataSourceConfigurationT&& value) {
    m_dataSourceConfigurationHasBeenSet = true;
    m_dataSourceConfiguration = std::forward<DataSourceConfigurationT>(value);
  }
  template <typename DataSourceConfigurationT = DataSourceConfiguration>
  CreateDataSourceRequest& WithDataSourceConfiguration(DataSourceConfigurationT&& value) {
    SetDataSourceConfiguration(std::forward<DataSourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data deletion policy for the data source.</p> <p>You can set the data
   * deletion policy to:</p> <ul> <li> <p>DELETE: Deletes all data from your data
   * source that’s converted into vector embeddings upon deletion of a knowledge base
   * or data source resource. Note that the <b>vector store itself is not
   * deleted</b>, only the data. This flag is ignored if an Amazon Web Services
   * account is deleted.</p> </li> <li> <p>RETAIN: Retains all data from your data
   * source that’s converted into vector embeddings upon deletion of a knowledge base
   * or data source resource. Note that the <b>vector store itself is not deleted</b>
   * if you delete a knowledge base or data source resource.</p> </li> </ul>
   */
  inline DataDeletionPolicy GetDataDeletionPolicy() const { return m_dataDeletionPolicy; }
  inline bool DataDeletionPolicyHasBeenSet() const { return m_dataDeletionPolicyHasBeenSet; }
  inline void SetDataDeletionPolicy(DataDeletionPolicy value) {
    m_dataDeletionPolicyHasBeenSet = true;
    m_dataDeletionPolicy = value;
  }
  inline CreateDataSourceRequest& WithDataDeletionPolicy(DataDeletionPolicy value) {
    SetDataDeletionPolicy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains details about the server-side encryption for the data source.</p>
   */
  inline const ServerSideEncryptionConfiguration& GetServerSideEncryptionConfiguration() const {
    return m_serverSideEncryptionConfiguration;
  }
  inline bool ServerSideEncryptionConfigurationHasBeenSet() const { return m_serverSideEncryptionConfigurationHasBeenSet; }
  template <typename ServerSideEncryptionConfigurationT = ServerSideEncryptionConfiguration>
  void SetServerSideEncryptionConfiguration(ServerSideEncryptionConfigurationT&& value) {
    m_serverSideEncryptionConfigurationHasBeenSet = true;
    m_serverSideEncryptionConfiguration = std::forward<ServerSideEncryptionConfigurationT>(value);
  }
  template <typename ServerSideEncryptionConfigurationT = ServerSideEncryptionConfiguration>
  CreateDataSourceRequest& WithServerSideEncryptionConfiguration(ServerSideEncryptionConfigurationT&& value) {
    SetServerSideEncryptionConfiguration(std::forward<ServerSideEncryptionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains details about how to ingest the documents in the data source.</p>
   */
  inline const VectorIngestionConfiguration& GetVectorIngestionConfiguration() const { return m_vectorIngestionConfiguration; }
  inline bool VectorIngestionConfigurationHasBeenSet() const { return m_vectorIngestionConfigurationHasBeenSet; }
  template <typename VectorIngestionConfigurationT = VectorIngestionConfiguration>
  void SetVectorIngestionConfiguration(VectorIngestionConfigurationT&& value) {
    m_vectorIngestionConfigurationHasBeenSet = true;
    m_vectorIngestionConfiguration = std::forward<VectorIngestionConfigurationT>(value);
  }
  template <typename VectorIngestionConfigurationT = VectorIngestionConfiguration>
  CreateDataSourceRequest& WithVectorIngestionConfiguration(VectorIngestionConfigurationT&& value) {
    SetVectorIngestionConfiguration(std::forward<VectorIngestionConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_knowledgeBaseId;
  bool m_knowledgeBaseIdHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  DataSourceConfiguration m_dataSourceConfiguration;
  bool m_dataSourceConfigurationHasBeenSet = false;

  DataDeletionPolicy m_dataDeletionPolicy{DataDeletionPolicy::NOT_SET};
  bool m_dataDeletionPolicyHasBeenSet = false;

  ServerSideEncryptionConfiguration m_serverSideEncryptionConfiguration;
  bool m_serverSideEncryptionConfigurationHasBeenSet = false;

  VectorIngestionConfiguration m_vectorIngestionConfiguration;
  bool m_vectorIngestionConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace BedrockAgent
}  // namespace Aws
