﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class CreateEndpointAccessRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API CreateEndpointAccessRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEndpointAccess"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The cluster identifier of the cluster to access.</p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  CreateEndpointAccessRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID of the owner of the cluster. This is only
   * required if the cluster is in another Amazon Web Services account.</p>
   */
  inline const Aws::String& GetResourceOwner() const { return m_resourceOwner; }
  inline bool ResourceOwnerHasBeenSet() const { return m_resourceOwnerHasBeenSet; }
  template <typename ResourceOwnerT = Aws::String>
  void SetResourceOwner(ResourceOwnerT&& value) {
    m_resourceOwnerHasBeenSet = true;
    m_resourceOwner = std::forward<ResourceOwnerT>(value);
  }
  template <typename ResourceOwnerT = Aws::String>
  CreateEndpointAccessRequest& WithResourceOwner(ResourceOwnerT&& value) {
    SetResourceOwner(std::forward<ResourceOwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Redshift-managed VPC endpoint name.</p> <p>An endpoint name must contain
   * 1-30 characters. Valid characters are A-Z, a-z, 0-9, and hyphen(-). The first
   * character must be a letter. The name can't contain two consecutive hyphens or
   * end with a hyphen.</p>
   */
  inline const Aws::String& GetEndpointName() const { return m_endpointName; }
  inline bool EndpointNameHasBeenSet() const { return m_endpointNameHasBeenSet; }
  template <typename EndpointNameT = Aws::String>
  void SetEndpointName(EndpointNameT&& value) {
    m_endpointNameHasBeenSet = true;
    m_endpointName = std::forward<EndpointNameT>(value);
  }
  template <typename EndpointNameT = Aws::String>
  CreateEndpointAccessRequest& WithEndpointName(EndpointNameT&& value) {
    SetEndpointName(std::forward<EndpointNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subnet group from which Amazon Redshift chooses the subnet to deploy the
   * endpoint.</p>
   */
  inline const Aws::String& GetSubnetGroupName() const { return m_subnetGroupName; }
  inline bool SubnetGroupNameHasBeenSet() const { return m_subnetGroupNameHasBeenSet; }
  template <typename SubnetGroupNameT = Aws::String>
  void SetSubnetGroupName(SubnetGroupNameT&& value) {
    m_subnetGroupNameHasBeenSet = true;
    m_subnetGroupName = std::forward<SubnetGroupNameT>(value);
  }
  template <typename SubnetGroupNameT = Aws::String>
  CreateEndpointAccessRequest& WithSubnetGroupName(SubnetGroupNameT&& value) {
    SetSubnetGroupName(std::forward<SubnetGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security group that defines the ports, protocols, and sources for inbound
   * traffic that you are authorizing into your endpoint.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVpcSecurityGroupIds() const { return m_vpcSecurityGroupIds; }
  inline bool VpcSecurityGroupIdsHasBeenSet() const { return m_vpcSecurityGroupIdsHasBeenSet; }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds = std::forward<VpcSecurityGroupIdsT>(value);
  }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  CreateEndpointAccessRequest& WithVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    SetVpcSecurityGroupIds(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  template <typename VpcSecurityGroupIdsT = Aws::String>
  CreateEndpointAccessRequest& AddVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds.emplace_back(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterIdentifier;
  bool m_clusterIdentifierHasBeenSet = false;

  Aws::String m_resourceOwner;
  bool m_resourceOwnerHasBeenSet = false;

  Aws::String m_endpointName;
  bool m_endpointNameHasBeenSet = false;

  Aws::String m_subnetGroupName;
  bool m_subnetGroupNameHasBeenSet = false;

  Aws::Vector<Aws::String> m_vpcSecurityGroupIds;
  bool m_vpcSecurityGroupIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
