/****
    Copyright (C) 2005 Intel Corporation.  All Rights Reserved.

    This file is part of SEP Development Kit.

    SEP Development Kit is free software; you can redistribute it
    and/or modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    SEP Development Kit is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with SEP Development Kit; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    As a special exception, you may use this file as part of a free software
    library without restriction.  Specifically, if other files instantiate
    templates or use macros or inline functions from this file, or you compile
    this file and link it with other files to produce an executable, this
    file does not by itself cause the resulting executable to be covered by
    the GNU General Public License.  This exception does not however
    invalidate any other reasons why the executable file might be covered by
    the GNU General Public License.
****/





#include "lwpmudrv_defines.h"
#include <linux/version.h>
#include <linux/interrupt.h>
#include <asm/msr.h>
#include <asm/apic.h>
#if defined(CONFIG_XEN) && LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 32)
#include <xen/xen.h>
#endif
#if defined(CONFIG_XEN_DOM0) && LINUX_VERSION_CODE > KERNEL_VERSION(3, 3, 0)
#include <xen/interface/platform.h>
#include <asm/xen/hypercall.h>
#endif

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "apic.h"
#include "lwpmudrv.h"
#include "control.h"
#include "utility.h"

static DEFINE_PER_CPU(U32, saved_apic_lvtpc);
U32 apic_lvtpc_val = APIC_DM_NMI;

VOID apic_Get_APIC_ID(S32);

/*!
 * @fn          VOID apic_Get_APIC_ID(S32 cpu)
 *
 * @brief       Obtain APIC ID
 *
 * @param       S32 cpuid - cpu index
 *
 * @return      U32 APIC ID
 */
VOID
apic_Get_APIC_ID(S32 cpu)
{
	U32       apic_id = 0;
	CPU_STATE pcpu;

	SEP_DRV_LOG_TRACE_IN("CPU: %d.", cpu);
	pcpu = &pcb[cpu];

#if defined(CONFIG_XEN_DOM0) && LINUX_VERSION_CODE > KERNEL_VERSION(3, 3, 0)
	if (xen_initial_domain()) {
		S32                    ret = 0;
		struct xen_platform_op op  = {
			 .cmd                   = XENPF_get_cpuinfo,
			 .interface_version     = XENPF_INTERFACE_VERSION,
			 .u.pcpu_info.xen_cpuid = cpu,
		};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 5, 0)
		ret = HYPERVISOR_platform_op(&op);
#else
		ret = HYPERVISOR_dom0_op(&op);
#endif
		if (ret) {
			SEP_DRV_LOG_ERROR("apic_Get_APIC_ID: Error in reading APIC ID on Xen PV.");
			apic_id = 0;
		} else {
			apic_id = op.u.pcpu_info.apic_id;
		}
	} else {
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 6, 0) || defined(DRV_DECLARE_APIC_CALL_PRESENT)
		apic_id = APIC_Read(APIC_ID);
		if (!apic_is_x2apic_enabled()) {
			apic_id = (apic_id >> 24) & 0xFF;
		}
#else
		apic_id = read_apic_id();
#endif
#if defined(CONFIG_XEN_DOM0) && LINUX_VERSION_CODE > KERNEL_VERSION(3, 3, 0)
	}
#endif

	CPU_STATE_apic_id(pcpu) = apic_id;

	SEP_DRV_LOG_TRACE_OUT("Apic_id[%d] is %d.", cpu,
			      CPU_STATE_apic_id(pcpu));
}

/*!
 * @fn          extern VOID APIC_Init(param)
 *
 * @brief       initialize the local APIC
 *
 * @param       int cpu_idx - The cpu to deinit
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 *              This routine is expected to be called via the CONTROL_Parallel routine
 */
extern VOID
APIC_Init(PVOID param)
{
	int me;

	SEP_DRV_LOG_TRACE_IN("Param: %p.", param);

	preempt_disable();
	me = CONTROL_THIS_CPU();
	preempt_enable();

	apic_Get_APIC_ID(me);

#if defined(X86_FEATURE_NMI_SOURCE) && defined(NMI_SOURCE_VEC_PMI) && defined(DRV_USE_NMI_SOURCE)
	if (cpu_feature_enabled(X86_FEATURE_NMI_SOURCE)) {
		apic_lvtpc_val |= NMI_SOURCE_VEC_PMI;
	}
#endif

	SEP_DRV_LOG_TRACE_OUT("");
}

/*!
 * @fn          extern VOID APIC_Install_Interrupt_Handler(param)
 *
 * @brief       Install the interrupt handler
 *
 * @param       int param - The linear address of the Local APIC
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 *             The linear address is necessary if the LAPIC is used.  If X2APIC is
 *             used the linear address is not necessary.
 */
extern VOID
APIC_Install_Interrupt_Handler(PVOID param)
{
	U32 this_cpu;

	SEP_DRV_LOG_TRACE_IN("Param: %p.", param);
	preempt_disable();
	this_cpu = CONTROL_THIS_CPU();
	preempt_enable();

	per_cpu(saved_apic_lvtpc, this_cpu) = APIC_Read(APIC_LVTPC);
	APIC_Write(APIC_LVTPC, apic_lvtpc_val);

	SEP_DRV_LOG_TRACE_OUT("");
}

/*!
 * @fn          extern VOID APIC_Enable_PMI(void)
 *
 * @brief       Enable the PMU interrupt
 *
 * @param       None
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 *             <NONE>
 */
extern VOID
APIC_Enable_Pmi(VOID)
{
	SEP_DRV_LOG_TRACE_IN("");

	APIC_Write(APIC_LVTPC, apic_lvtpc_val);

	SEP_DRV_LOG_TRACE_OUT("");
}

/*!
 * @fn          extern VOID APIC_Restore_LVTPC(void)
 *
 * @brief       Restore APIC LVTPC value
 *
 * @param       None
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 *             <NONE>
 */
extern VOID
APIC_Restore_LVTPC(PVOID param)
{
	U32 this_cpu;

	SEP_DRV_LOG_TRACE_IN("");
	preempt_disable();
	this_cpu = CONTROL_THIS_CPU();
	preempt_enable();

	APIC_Write(APIC_LVTPC, per_cpu(saved_apic_lvtpc, this_cpu));

	SEP_DRV_LOG_TRACE_OUT("");
}

/*!
 * @fn          extern U32 APIC_Read(U32 reg)
 *
 * @brief       Read APIC register value
 *
 * @param       U32 reg - APIC register
 *
 * @return      APIC register value
 *
 * <I>Special Notes:</I>
 *             <NONE>
 */
extern U32
APIC_Read(U32 reg)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 6, 0) || defined(DRV_DECLARE_APIC_CALL_PRESENT)
	if (apic_is_x2apic_enabled()) {
		return native_apic_msr_read(reg);
	} else {
		return native_apic_mem_read(reg);
	}
#else
	return apic_read(reg);
#endif
}

/*!
 * @fn          extern VOID APIC_Write(U32 reg, U32 val)
 *
 * @brief       Write value to APIC register
 *
 * @param       U32 reg - APIC register
 *              U32 val - value to write in APIC register
 *
 * @return      None
 *
 * <I>Special Notes:</I>
 *             <NONE>
 */
extern VOID
APIC_Write(U32 reg, U32 val)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 6, 0) || defined(DRV_DECLARE_APIC_CALL_PRESENT)
	if (apic_is_x2apic_enabled()) {
		native_apic_msr_write(reg, val);
	} else {
		native_apic_mem_write(reg, val);
	}
#else
	apic_write(reg, val);
#endif
}

