﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/CreateTransitGatewayPeeringAttachmentRequestOptions.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateTransitGatewayPeeringAttachmentRequest : public EC2Request {
 public:
  AWS_EC2_API CreateTransitGatewayPeeringAttachmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTransitGatewayPeeringAttachment"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the transit gateway.</p>
   */
  inline const Aws::String& GetTransitGatewayId() const { return m_transitGatewayId; }
  inline bool TransitGatewayIdHasBeenSet() const { return m_transitGatewayIdHasBeenSet; }
  template <typename TransitGatewayIdT = Aws::String>
  void SetTransitGatewayId(TransitGatewayIdT&& value) {
    m_transitGatewayIdHasBeenSet = true;
    m_transitGatewayId = std::forward<TransitGatewayIdT>(value);
  }
  template <typename TransitGatewayIdT = Aws::String>
  CreateTransitGatewayPeeringAttachmentRequest& WithTransitGatewayId(TransitGatewayIdT&& value) {
    SetTransitGatewayId(std::forward<TransitGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the peer transit gateway with which to create the peering
   * attachment.</p>
   */
  inline const Aws::String& GetPeerTransitGatewayId() const { return m_peerTransitGatewayId; }
  inline bool PeerTransitGatewayIdHasBeenSet() const { return m_peerTransitGatewayIdHasBeenSet; }
  template <typename PeerTransitGatewayIdT = Aws::String>
  void SetPeerTransitGatewayId(PeerTransitGatewayIdT&& value) {
    m_peerTransitGatewayIdHasBeenSet = true;
    m_peerTransitGatewayId = std::forward<PeerTransitGatewayIdT>(value);
  }
  template <typename PeerTransitGatewayIdT = Aws::String>
  CreateTransitGatewayPeeringAttachmentRequest& WithPeerTransitGatewayId(PeerTransitGatewayIdT&& value) {
    SetPeerTransitGatewayId(std::forward<PeerTransitGatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Amazon Web Services account that owns the peer transit
   * gateway.</p>
   */
  inline const Aws::String& GetPeerAccountId() const { return m_peerAccountId; }
  inline bool PeerAccountIdHasBeenSet() const { return m_peerAccountIdHasBeenSet; }
  template <typename PeerAccountIdT = Aws::String>
  void SetPeerAccountId(PeerAccountIdT&& value) {
    m_peerAccountIdHasBeenSet = true;
    m_peerAccountId = std::forward<PeerAccountIdT>(value);
  }
  template <typename PeerAccountIdT = Aws::String>
  CreateTransitGatewayPeeringAttachmentRequest& WithPeerAccountId(PeerAccountIdT&& value) {
    SetPeerAccountId(std::forward<PeerAccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Region where the peer transit gateway is located.</p>
   */
  inline const Aws::String& GetPeerRegion() const { return m_peerRegion; }
  inline bool PeerRegionHasBeenSet() const { return m_peerRegionHasBeenSet; }
  template <typename PeerRegionT = Aws::String>
  void SetPeerRegion(PeerRegionT&& value) {
    m_peerRegionHasBeenSet = true;
    m_peerRegion = std::forward<PeerRegionT>(value);
  }
  template <typename PeerRegionT = Aws::String>
  CreateTransitGatewayPeeringAttachmentRequest& WithPeerRegion(PeerRegionT&& value) {
    SetPeerRegion(std::forward<PeerRegionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Requests a transit gateway peering attachment.</p>
   */
  inline const CreateTransitGatewayPeeringAttachmentRequestOptions& GetOptions() const { return m_options; }
  inline bool OptionsHasBeenSet() const { return m_optionsHasBeenSet; }
  template <typename OptionsT = CreateTransitGatewayPeeringAttachmentRequestOptions>
  void SetOptions(OptionsT&& value) {
    m_optionsHasBeenSet = true;
    m_options = std::forward<OptionsT>(value);
  }
  template <typename OptionsT = CreateTransitGatewayPeeringAttachmentRequestOptions>
  CreateTransitGatewayPeeringAttachmentRequest& WithOptions(OptionsT&& value) {
    SetOptions(std::forward<OptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to apply to the transit gateway peering attachment.</p>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  CreateTransitGatewayPeeringAttachmentRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  CreateTransitGatewayPeeringAttachmentRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateTransitGatewayPeeringAttachmentRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_transitGatewayId;

  Aws::String m_peerTransitGatewayId;

  Aws::String m_peerAccountId;

  Aws::String m_peerRegion;

  CreateTransitGatewayPeeringAttachmentRequestOptions m_options;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  bool m_dryRun{false};
  bool m_transitGatewayIdHasBeenSet = false;
  bool m_peerTransitGatewayIdHasBeenSet = false;
  bool m_peerAccountIdHasBeenSet = false;
  bool m_peerRegionHasBeenSet = false;
  bool m_optionsHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
