﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/workmail/WorkMailRequest.h>
#include <aws/workmail/WorkMail_EXPORTS.h>
#include <aws/workmail/model/ImpersonationRoleType.h>
#include <aws/workmail/model/ImpersonationRule.h>

#include <utility>

namespace Aws {
namespace WorkMail {
namespace Model {

/**
 */
class UpdateImpersonationRoleRequest : public WorkMailRequest {
 public:
  AWS_WORKMAIL_API UpdateImpersonationRoleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateImpersonationRole"; }

  AWS_WORKMAIL_API Aws::String SerializePayload() const override;

  AWS_WORKMAIL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The WorkMail organization that contains the impersonation role to update.</p>
   */
  inline const Aws::String& GetOrganizationId() const { return m_organizationId; }
  inline bool OrganizationIdHasBeenSet() const { return m_organizationIdHasBeenSet; }
  template <typename OrganizationIdT = Aws::String>
  void SetOrganizationId(OrganizationIdT&& value) {
    m_organizationIdHasBeenSet = true;
    m_organizationId = std::forward<OrganizationIdT>(value);
  }
  template <typename OrganizationIdT = Aws::String>
  UpdateImpersonationRoleRequest& WithOrganizationId(OrganizationIdT&& value) {
    SetOrganizationId(std::forward<OrganizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the impersonation role to update.</p>
   */
  inline const Aws::String& GetImpersonationRoleId() const { return m_impersonationRoleId; }
  inline bool ImpersonationRoleIdHasBeenSet() const { return m_impersonationRoleIdHasBeenSet; }
  template <typename ImpersonationRoleIdT = Aws::String>
  void SetImpersonationRoleId(ImpersonationRoleIdT&& value) {
    m_impersonationRoleIdHasBeenSet = true;
    m_impersonationRoleId = std::forward<ImpersonationRoleIdT>(value);
  }
  template <typename ImpersonationRoleIdT = Aws::String>
  UpdateImpersonationRoleRequest& WithImpersonationRoleId(ImpersonationRoleIdT&& value) {
    SetImpersonationRoleId(std::forward<ImpersonationRoleIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated impersonation role name.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateImpersonationRoleRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated impersonation role type.</p>
   */
  inline ImpersonationRoleType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ImpersonationRoleType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline UpdateImpersonationRoleRequest& WithType(ImpersonationRoleType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated impersonation role description.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateImpersonationRoleRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated list of rules.</p>
   */
  inline const Aws::Vector<ImpersonationRule>& GetRules() const { return m_rules; }
  inline bool RulesHasBeenSet() const { return m_rulesHasBeenSet; }
  template <typename RulesT = Aws::Vector<ImpersonationRule>>
  void SetRules(RulesT&& value) {
    m_rulesHasBeenSet = true;
    m_rules = std::forward<RulesT>(value);
  }
  template <typename RulesT = Aws::Vector<ImpersonationRule>>
  UpdateImpersonationRoleRequest& WithRules(RulesT&& value) {
    SetRules(std::forward<RulesT>(value));
    return *this;
  }
  template <typename RulesT = ImpersonationRule>
  UpdateImpersonationRoleRequest& AddRules(RulesT&& value) {
    m_rulesHasBeenSet = true;
    m_rules.emplace_back(std::forward<RulesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_organizationId;

  Aws::String m_impersonationRoleId;

  Aws::String m_name;

  ImpersonationRoleType m_type{ImpersonationRoleType::NOT_SET};

  Aws::String m_description;

  Aws::Vector<ImpersonationRule> m_rules;
  bool m_organizationIdHasBeenSet = false;
  bool m_impersonationRoleIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_rulesHasBeenSet = false;
};

}  // namespace Model
}  // namespace WorkMail
}  // namespace Aws
