﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/shield/ShieldRequest.h>
#include <aws/shield/Shield_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Shield {
namespace Model {

/**
 */
class ListResourcesInProtectionGroupRequest : public ShieldRequest {
 public:
  AWS_SHIELD_API ListResourcesInProtectionGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListResourcesInProtectionGroup"; }

  AWS_SHIELD_API Aws::String SerializePayload() const override;

  AWS_SHIELD_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the protection group. You use this to identify the protection
   * group in lists and to manage the protection group, for example to update,
   * delete, or describe it. </p>
   */
  inline const Aws::String& GetProtectionGroupId() const { return m_protectionGroupId; }
  inline bool ProtectionGroupIdHasBeenSet() const { return m_protectionGroupIdHasBeenSet; }
  template <typename ProtectionGroupIdT = Aws::String>
  void SetProtectionGroupId(ProtectionGroupIdT&& value) {
    m_protectionGroupIdHasBeenSet = true;
    m_protectionGroupId = std::forward<ProtectionGroupIdT>(value);
  }
  template <typename ProtectionGroupIdT = Aws::String>
  ListResourcesInProtectionGroupRequest& WithProtectionGroupId(ProtectionGroupIdT&& value) {
    SetProtectionGroupId(std::forward<ProtectionGroupIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When you request a list of objects from Shield Advanced, if the response does
   * not include all of the remaining available objects, Shield Advanced includes a
   * <code>NextToken</code> value in the response. You can retrieve the next batch of
   * objects by requesting the list again and providing the token that was returned
   * by the prior call in your request. </p> <p>You can indicate the maximum number
   * of objects that you want Shield Advanced to return for a single call with the
   * <code>MaxResults</code> setting. Shield Advanced will not return more than
   * <code>MaxResults</code> objects, but may return fewer, even if more objects are
   * still available.</p> <p>Whenever more objects remain that Shield Advanced has
   * not yet returned to you, the response will include a <code>NextToken</code>
   * value.</p> <p>On your first call to a list operation, leave this setting
   * empty.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListResourcesInProtectionGroupRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The greatest number of objects that you want Shield Advanced to return to the
   * list request. Shield Advanced might return fewer objects than you indicate in
   * this setting, even if more objects are available. If there are more objects
   * remaining, Shield Advanced will always also return a <code>NextToken</code>
   * value in the response.</p> <p>The default setting is 20.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListResourcesInProtectionGroupRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_protectionGroupId;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_protectionGroupIdHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Shield
}  // namespace Aws
