/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.utils;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.SyncFailedException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Random;
import java.util.UUID;
import org.apache.ratis.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DiskCheckUtil {
    private static DiskChecks impl = new DiskChecksImpl();

    private DiskCheckUtil() {
    }

    @VisibleForTesting
    public static void setTestImpl(DiskChecks diskChecks) {
        impl = diskChecks;
    }

    @VisibleForTesting
    public static void clearTestImpl() {
        impl = new DiskChecksImpl();
    }

    public static boolean checkExistence(File storageDir) {
        return impl.checkExistence(storageDir);
    }

    public static boolean checkPermissions(File storageDir) {
        return impl.checkPermissions(storageDir);
    }

    public static boolean checkReadWrite(File storageDir, File testFileDir, int numBytesToWrite) {
        return impl.checkReadWrite(storageDir, testFileDir, numBytesToWrite);
    }

    public static interface DiskChecks {
        default public boolean checkExistence(File storageDir) {
            return true;
        }

        default public boolean checkPermissions(File storageDir) {
            return true;
        }

        default public boolean checkReadWrite(File storageDir, File testFileDir, int numBytesToWrite) {
            return true;
        }
    }

    private static class DiskChecksImpl
    implements DiskChecks {
        private static final Logger LOG = LoggerFactory.getLogger(DiskCheckUtil.class);
        private static final Random RANDOM = new Random();

        private DiskChecksImpl() {
        }

        @Override
        public boolean checkExistence(File diskDir) {
            if (!diskDir.exists()) {
                this.logError(diskDir, "Directory does not exist.");
                return false;
            }
            return true;
        }

        @Override
        public boolean checkPermissions(File storageDir) {
            boolean permissionsCorrect = true;
            if (!storageDir.canRead()) {
                this.logError(storageDir, "Datanode does not have read permission on volume.");
                permissionsCorrect = false;
            }
            if (!storageDir.canWrite()) {
                this.logError(storageDir, "Datanode does not have write permission on volume.");
                permissionsCorrect = false;
            }
            if (!storageDir.canExecute()) {
                this.logError(storageDir, "Datanode does not have executepermission on volume.");
                permissionsCorrect = false;
            }
            return permissionsCorrect;
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public boolean checkReadWrite(File storageDir, File testFileDir, int numBytesToWrite) {
            File testFile = new File(testFileDir, "disk-check-" + UUID.randomUUID());
            byte[] writtenBytes = new byte[numBytesToWrite];
            RANDOM.nextBytes(writtenBytes);
            try (OutputStream fos = FileUtils.newOutputStreamForceAtClose((File)testFile, (OpenOption[])new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE});){
                fos.write(writtenBytes);
            }
            catch (FileNotFoundException | NoSuchFileException notFoundEx) {
                this.logError(storageDir, String.format("Could not find file %s for volume check.", testFile.getAbsolutePath()), notFoundEx);
                return false;
            }
            catch (SyncFailedException syncEx) {
                this.logError(storageDir, String.format("Could sync file %s to disk.", testFile.getAbsolutePath()), syncEx);
                return false;
            }
            catch (IOException ioEx) {
                this.logError(storageDir, String.format("Could not write file %s for volume check.", testFile.getAbsolutePath()), ioEx);
                return false;
            }
            byte[] readBytes = new byte[numBytesToWrite];
            try (InputStream fis = Files.newInputStream(testFile.toPath(), new OpenOption[0]);){
                int numBytesRead = fis.read(readBytes);
                if (numBytesRead != numBytesToWrite) {
                    this.logError(storageDir, String.format("%d bytes written to file %s but %d bytes were read back.", numBytesToWrite, testFile.getAbsolutePath(), numBytesRead));
                    boolean bl = false;
                    return bl;
                }
            }
            catch (FileNotFoundException | NoSuchFileException notFoundEx) {
                this.logError(storageDir, String.format("Could not find file %s for volume check.", testFile.getAbsolutePath()), notFoundEx);
                return false;
            }
            catch (IOException ioEx) {
                this.logError(storageDir, String.format("Could not read file %s for volume check.", testFile.getAbsolutePath()), ioEx);
                return false;
            }
            if (!Arrays.equals(writtenBytes, readBytes)) {
                this.logError(storageDir, String.format("%d Bytes read from file %s do not match the %d bytes that were written.", writtenBytes.length, testFile.getAbsolutePath(), readBytes.length));
                return false;
            }
            if (testFile.delete()) return true;
            this.logError(storageDir, String.format("Could not delete file %s for volume check.", testFile.getAbsolutePath()));
            return false;
        }

        private void logError(File storageDir, String message) {
            LOG.error("Volume {} failed health check. {}", (Object)storageDir, (Object)message);
        }

        private void logError(File storageDir, String message, Exception ex) {
            LOG.error("Volume {} failed health check. {}", new Object[]{storageDir, message, ex});
        }
    }
}

