/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.base.Preconditions;
import com.google.common.primitives.UnsignedBytes;
import java.io.Closeable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.hdds.StringUtils;
import org.apache.hadoop.hdds.utils.IOUtils;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.CodecBuffer;
import org.apache.hadoop.hdds.utils.db.RocksDatabase;
import org.apache.hadoop.hdds.utils.db.RocksDatabaseException;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteBatch;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.apache.ratis.util.TraditionalBinaryPrefix;
import org.apache.ratis.util.UncheckedAutoCloseable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBBatchOperation
implements BatchOperation {
    static final Logger LOG = LoggerFactory.getLogger(RDBBatchOperation.class);
    private static final AtomicInteger BATCH_COUNT = new AtomicInteger();
    private final String name = "Batch-" + BATCH_COUNT.getAndIncrement();
    private final ManagedWriteBatch writeBatch;
    private final OpCache opCache = new OpCache();

    private static void debug(Supplier<String> message) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("\n{}", (Object)message.get());
        }
    }

    private static String byteSize2String(long length) {
        return TraditionalBinaryPrefix.long2String((long)length, (String)"B", (int)2);
    }

    private static String countSize2String(int count, long size) {
        return count + " (" + RDBBatchOperation.byteSize2String(size) + ")";
    }

    public RDBBatchOperation() {
        this.writeBatch = new ManagedWriteBatch();
    }

    public RDBBatchOperation(ManagedWriteBatch writeBatch) {
        this.writeBatch = writeBatch;
    }

    public String toString() {
        return this.name;
    }

    public void commit(RocksDatabase db) throws RocksDatabaseException {
        RDBBatchOperation.debug(() -> String.format("%s: commit %s", this.name, this.opCache.getCommitString()));
        try (UncheckedAutoCloseable ignored = this.opCache.prepareBatchWrite();){
            db.batchWrite(this.writeBatch);
        }
    }

    public void commit(RocksDatabase db, ManagedWriteOptions writeOptions) throws RocksDatabaseException {
        RDBBatchOperation.debug(() -> String.format("%s: commit-with-writeOptions %s", this.name, this.opCache.getCommitString()));
        try (UncheckedAutoCloseable ignored = this.opCache.prepareBatchWrite();){
            db.batchWrite(this.writeBatch, writeOptions);
        }
    }

    @Override
    public void close() {
        RDBBatchOperation.debug(() -> String.format("%s: close", this.name));
        this.writeBatch.close();
        this.opCache.clear();
    }

    public void delete(RocksDatabase.ColumnFamily family, byte[] key) {
        this.opCache.delete(family, key);
    }

    public void put(RocksDatabase.ColumnFamily family, CodecBuffer key, CodecBuffer value) {
        this.opCache.put(family, key, value);
    }

    public void put(RocksDatabase.ColumnFamily family, byte[] key, byte[] value) {
        this.opCache.put(family, key, value);
    }

    public void deleteRange(RocksDatabase.ColumnFamily family, byte[] startKey, byte[] endKey) {
        this.opCache.deleteRange(family, startKey, endKey);
    }

    private class OpCache {
        private final Map<String, FamilyCache> name2cache = new HashMap<String, FamilyCache>();

        private OpCache() {
        }

        void put(RocksDatabase.ColumnFamily f, CodecBuffer key, CodecBuffer value) {
            this.name2cache.computeIfAbsent(f.getName(), k -> new FamilyCache(f)).put(key, value);
        }

        void put(RocksDatabase.ColumnFamily f, byte[] key, byte[] value) {
            this.name2cache.computeIfAbsent(f.getName(), k -> new FamilyCache(f)).put(key, value);
        }

        void delete(RocksDatabase.ColumnFamily family, byte[] key) {
            this.name2cache.computeIfAbsent(family.getName(), k -> new FamilyCache(family)).delete(key);
        }

        void deleteRange(RocksDatabase.ColumnFamily family, byte[] startKey, byte[] endKey) {
            this.name2cache.computeIfAbsent(family.getName(), k -> new FamilyCache(family)).deleteRange(startKey, endKey);
        }

        UncheckedAutoCloseable prepareBatchWrite() throws RocksDatabaseException {
            for (Map.Entry<String, FamilyCache> e : this.name2cache.entrySet()) {
                e.getValue().prepareBatchWrite();
            }
            return this::clear;
        }

        void clear() {
            for (Map.Entry<String, FamilyCache> e : this.name2cache.entrySet()) {
                e.getValue().clear();
            }
            this.name2cache.clear();
        }

        String getCommitString() {
            int putCount = 0;
            int delCount = 0;
            int opSize = 0;
            int discardedCount = 0;
            int discardedSize = 0;
            int delRangeCount = 0;
            for (FamilyCache f : this.name2cache.values()) {
                putCount += f.putCount;
                delCount += f.delCount;
                opSize = (int)((long)opSize + f.batchSize);
                discardedCount += f.discardedCount;
                discardedSize = (int)((long)discardedSize + f.discardedSize);
                delRangeCount += f.delRangeCount;
            }
            int opCount = putCount + delCount;
            return String.format("#put=%s, #del=%s, #delRange=%s, batchSize: %s, discarded: %s, committed: %s", putCount, delCount, delRangeCount, RDBBatchOperation.countSize2String(opCount, opSize), RDBBatchOperation.countSize2String(discardedCount, discardedSize), RDBBatchOperation.countSize2String(opCount - discardedCount, opSize - discardedSize));
        }

        private class FamilyCache {
            private final RocksDatabase.ColumnFamily family;
            private final Map<Bytes, Integer> opsKeys = new HashMap<Bytes, Integer>();
            private final Map<Integer, Operation> batchOps = new HashMap<Integer, Operation>();
            private boolean isCommit;
            private long batchSize;
            private long discardedSize;
            private int discardedCount;
            private int putCount;
            private int delCount;
            private int delRangeCount;
            private AtomicInteger opIndex;

            FamilyCache(RocksDatabase.ColumnFamily family) {
                this.family = family;
                this.opIndex = new AtomicInteger(0);
            }

            void prepareBatchWrite() throws RocksDatabaseException {
                Preconditions.checkState((!this.isCommit ? 1 : 0) != 0, (String)"%s is already committed.", (Object)this);
                this.isCommit = true;
                List ops = this.batchOps.entrySet().stream().sorted(Comparator.comparingInt(Map.Entry::getKey)).map(Map.Entry::getValue).collect(Collectors.toList());
                ArrayList<List<Object>> deleteRangeIndices = new ArrayList<List<Object>>();
                int index = 0;
                int prevIndex = -2;
                for (Operation op : ops) {
                    if (Op.DELETE_RANGE == op.getOpType()) {
                        if (index - prevIndex > 1) {
                            deleteRangeIndices.add(new ArrayList());
                        }
                        List list = (List)deleteRangeIndices.get(deleteRangeIndices.size() - 1);
                        list.add(index);
                        prevIndex = index;
                    }
                    ++index;
                }
                deleteRangeIndices.add(Collections.emptyList());
                int startIndex = 0;
                for (List list : deleteRangeIndices) {
                    List deleteRangeOps = list.stream().map(i -> (DeleteRangeOperation)ops.get((int)i)).collect(Collectors.toList());
                    List deleteRangeOpsRanges = list.stream().map(i -> (DeleteRangeOperation)ops.get((int)i)).map(i -> Pair.of((Object)new Bytes(((DeleteRangeOperation)i).startKey), (Object)new Bytes(((DeleteRangeOperation)i).endKey))).collect(Collectors.toList());
                    int firstOpIndex = list.isEmpty() ? ops.size() : ((Integer)list.get(0)).intValue();
                    for (int i2 = startIndex; i2 < firstOpIndex; ++i2) {
                        Operation op = (Operation)ops.get(i2);
                        Bytes key = op.getKey();
                        boolean keyInRange = false;
                        Pair deleteRange = null;
                        for (Pair deleteRangeOp : deleteRangeOpsRanges) {
                            if (key.compareTo((Bytes)deleteRangeOp.getLeft()) < 0 || key.compareTo((Bytes)deleteRangeOp.getRight()) >= 0) continue;
                            keyInRange = true;
                            deleteRange = deleteRangeOp;
                            break;
                        }
                        if (!keyInRange) {
                            op.apply(this.family, RDBBatchOperation.this.writeBatch);
                            continue;
                        }
                        Pair finalDeleteRange = deleteRange;
                        RDBBatchOperation.debug(() -> String.format("Discarding Operation with Key: %s as it falls within the range of [%s, %s)", StringUtils.bytes2String((ByteBuffer)key.asReadOnlyByteBuffer()), StringUtils.bytes2String((ByteBuffer)((Bytes)finalDeleteRange.getKey()).asReadOnlyByteBuffer()), StringUtils.bytes2String((ByteBuffer)((Bytes)finalDeleteRange.getRight()).asReadOnlyByteBuffer())));
                        ++this.discardedCount;
                        this.discardedSize += (long)op.totalLength();
                    }
                    for (DeleteRangeOperation deleteRangeOp : deleteRangeOps) {
                        deleteRangeOp.apply(this.family, RDBBatchOperation.this.writeBatch);
                    }
                    startIndex = firstOpIndex + list.size();
                }
                RDBBatchOperation.debug(this::summary);
            }

            private String summary() {
                return String.format("  %s %s, #put=%s, #del=%s", this, this.batchSizeDiscardedString(), this.putCount, this.delCount);
            }

            void clear() {
                boolean warn = !this.isCommit && this.batchSize > 0L;
                String details = warn ? this.summary() : null;
                IOUtils.close((Logger)LOG, this.batchOps.values());
                this.batchOps.clear();
                if (warn) {
                    LOG.warn("discarding changes {}", (Object)details);
                }
            }

            private void deleteIfExist(Bytes key, boolean removeFromIndexMap) {
                if (this.opsKeys.containsKey(key)) {
                    int previousIndex = removeFromIndexMap ? this.opsKeys.remove(key) : this.opsKeys.get(key);
                    Operation previous = this.batchOps.remove(previousIndex);
                    previous.close();
                    this.discardedSize += (long)previous.totalLength();
                    ++this.discardedCount;
                    RDBBatchOperation.debug(() -> String.format("%s overwriting a previous %s[valLen => %s]", new Object[]{this, previous.getOpType(), previous.valLen()}));
                }
            }

            void overWriteOpIfExist(Bytes key, Operation operation) {
                Preconditions.checkState((!this.isCommit ? 1 : 0) != 0, (String)"%s is already committed.", (Object)this);
                this.deleteIfExist(key, true);
                this.batchSize += (long)operation.totalLength();
                int newIndex = this.opIndex.getAndIncrement();
                Integer overwritten = this.opsKeys.put(key, newIndex);
                this.batchOps.put(newIndex, operation);
                Preconditions.checkState((overwritten == null || !this.batchOps.containsKey(overwritten) ? 1 : 0) != 0);
                RDBBatchOperation.debug(() -> String.format("%s %s, %s; key=%s", this, Op.DELETE == operation.getOpType() ? this.delString(operation.totalLength()) : this.putString(operation.keyLen(), operation.valLen()), this.batchSizeDiscardedString(), key));
            }

            void put(CodecBuffer key, CodecBuffer value) {
                ++this.putCount;
                Bytes keyBytes = new Bytes(key);
                this.overWriteOpIfExist(keyBytes, new CodecBufferPutOperation(key, value, keyBytes));
            }

            void put(byte[] key, byte[] value) {
                ++this.putCount;
                Bytes keyBytes = new Bytes(key);
                this.overWriteOpIfExist(keyBytes, new ByteArrayPutOperation(key, value, keyBytes));
            }

            void delete(byte[] key) {
                ++this.delCount;
                Bytes keyBytes = new Bytes(key);
                this.overWriteOpIfExist(keyBytes, new DeleteOperation(key, keyBytes));
            }

            void deleteRange(byte[] startKey, byte[] endKey) {
                ++this.delRangeCount;
                this.batchOps.put(this.opIndex.getAndIncrement(), new DeleteRangeOperation(startKey, endKey));
            }

            String putString(int keySize, int valueSize) {
                return String.format("put(key: %s, value: %s), #put=%s", RDBBatchOperation.byteSize2String(keySize), RDBBatchOperation.byteSize2String(valueSize), this.putCount);
            }

            String delString(int keySize) {
                return String.format("del(key: %s), #del=%s", RDBBatchOperation.byteSize2String(keySize), this.delCount);
            }

            String batchSizeDiscardedString() {
                return String.format("batchSize=%s, discarded: %s", RDBBatchOperation.byteSize2String(this.batchSize), RDBBatchOperation.countSize2String(this.discardedCount, this.discardedSize));
            }

            public String toString() {
                return RDBBatchOperation.this.name + ": " + this.family.getName();
            }
        }
    }

    private final class DeleteRangeOperation
    extends Operation {
        private final byte[] startKey;
        private final byte[] endKey;

        private DeleteRangeOperation(byte[] startKey, byte[] endKey) {
            super(null);
            this.startKey = Objects.requireNonNull(startKey, "startKey == null");
            this.endKey = Objects.requireNonNull(endKey, "endKey == null");
        }

        @Override
        public void apply(RocksDatabase.ColumnFamily family, ManagedWriteBatch batch) throws RocksDatabaseException {
            family.batchDeleteRange(batch, this.startKey, this.endKey);
        }

        @Override
        public int keyLen() {
            return this.startKey.length + this.endKey.length;
        }

        @Override
        public int valLen() {
            return 0;
        }

        @Override
        public Op getOpType() {
            return Op.DELETE_RANGE;
        }
    }

    private final class ByteArrayPutOperation
    extends Operation {
        private final byte[] key;
        private final byte[] value;

        private ByteArrayPutOperation(byte[] key, byte[] value, Bytes keyBytes) {
            super(Objects.requireNonNull(keyBytes));
            this.key = Objects.requireNonNull(key, "key == null");
            this.value = Objects.requireNonNull(value, "value == null");
        }

        @Override
        public void apply(RocksDatabase.ColumnFamily family, ManagedWriteBatch batch) throws RocksDatabaseException {
            family.batchPut(batch, this.key, this.value);
        }

        @Override
        public int keyLen() {
            return this.key.length;
        }

        @Override
        public int valLen() {
            return this.value.length;
        }

        @Override
        public Op getOpType() {
            return Op.PUT;
        }
    }

    private final class CodecBufferPutOperation
    extends Operation {
        private final CodecBuffer key;
        private final CodecBuffer value;
        private final AtomicBoolean closed;

        private CodecBufferPutOperation(CodecBuffer key, CodecBuffer value, Bytes keyBytes) {
            super(keyBytes);
            this.closed = new AtomicBoolean(false);
            this.key = key;
            this.value = value;
        }

        @Override
        public void apply(RocksDatabase.ColumnFamily family, ManagedWriteBatch batch) throws RocksDatabaseException {
            family.batchPut(batch, this.key.asReadOnlyByteBuffer(), this.value.asReadOnlyByteBuffer());
        }

        @Override
        public int keyLen() {
            return this.key.readableBytes();
        }

        @Override
        public int valLen() {
            return this.value.readableBytes();
        }

        @Override
        public Op getOpType() {
            return Op.PUT;
        }

        @Override
        public void close() {
            if (this.closed.compareAndSet(false, true)) {
                this.key.release();
                this.value.release();
            }
            super.close();
        }
    }

    private final class DeleteOperation
    extends Operation {
        private final byte[] key;

        private DeleteOperation(byte[] key, Bytes keyBytes) {
            super(Objects.requireNonNull(keyBytes, "keyBytes == null"));
            this.key = Objects.requireNonNull(key, "key == null");
        }

        @Override
        public void apply(RocksDatabase.ColumnFamily family, ManagedWriteBatch batch) throws RocksDatabaseException {
            family.batchDelete(batch, this.key);
        }

        @Override
        public int keyLen() {
            return this.key.length;
        }

        @Override
        public int valLen() {
            return 0;
        }

        @Override
        public Op getOpType() {
            return Op.DELETE;
        }
    }

    private abstract class Operation
    implements Closeable {
        private Bytes keyBytes;

        private Operation(Bytes keyBytes) {
            this.keyBytes = keyBytes;
        }

        abstract void apply(RocksDatabase.ColumnFamily var1, ManagedWriteBatch var2) throws RocksDatabaseException;

        abstract int keyLen();

        abstract int valLen();

        Bytes getKey() {
            return this.keyBytes;
        }

        int totalLength() {
            return this.keyLen() + this.valLen();
        }

        abstract Op getOpType();

        @Override
        public void close() {
        }
    }

    static final class Bytes
    implements Comparable<Bytes> {
        private final byte[] array;
        private final CodecBuffer buffer;
        private final int hash;

        Bytes(CodecBuffer buffer) {
            this.array = null;
            this.buffer = Objects.requireNonNull(buffer, "buffer == null");
            this.hash = buffer.asReadOnlyByteBuffer().hashCode();
        }

        Bytes(byte[] array) {
            this.array = array;
            this.buffer = null;
            this.hash = ByteBuffer.wrap(array).hashCode();
        }

        ByteBuffer asReadOnlyByteBuffer() {
            return this.buffer.asReadOnlyByteBuffer();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof Bytes)) {
                return false;
            }
            Bytes that = (Bytes)obj;
            if (this.hash != that.hash) {
                return false;
            }
            ByteBuffer thisBuf = this.array != null ? ByteBuffer.wrap(this.array) : this.asReadOnlyByteBuffer();
            ByteBuffer thatBuf = that.array != null ? ByteBuffer.wrap(that.array) : that.asReadOnlyByteBuffer();
            return thisBuf.equals(thatBuf);
        }

        public int hashCode() {
            return this.hash;
        }

        public String toString() {
            return this.array != null ? StringUtils.bytes2String((byte[])this.array) : StringUtils.bytes2String((ByteBuffer)this.asReadOnlyByteBuffer());
        }

        @Override
        public int compareTo(Bytes that) {
            ByteBuffer thisBuf = this.array != null ? ByteBuffer.wrap(this.array) : this.asReadOnlyByteBuffer();
            ByteBuffer thatBuf = that.array != null ? ByteBuffer.wrap(that.array) : that.asReadOnlyByteBuffer();
            for (int i = 0; i < Math.min(thisBuf.remaining(), thatBuf.remaining()); ++i) {
                int cmp = UnsignedBytes.compare((byte)thisBuf.get(i), (byte)thatBuf.get(i));
                if (cmp == 0) continue;
                return cmp;
            }
            return thisBuf.remaining() - thatBuf.remaining();
        }
    }

    private static enum Op {
        DELETE,
        PUT,
        DELETE_RANGE;

    }
}

