/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.keys;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.Set;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.x509.keys.KeyCodec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyStorage {
    private static final Logger LOG = LoggerFactory.getLogger(KeyStorage.class);
    public static final Set<PosixFilePermission> DIR_PERMISSIONS = PosixFilePermissions.fromString("rwx------");
    public static final Set<PosixFilePermission> FILE_PERMISSIONS = PosixFilePermissions.fromString("rw-------");
    private final Path privateKeyPath;
    private final Path publicKeyPath;
    private final KeyCodec keyCodec;
    private final boolean externalKeysUsed;

    public KeyStorage(SecurityConfig config, String component) throws IOException {
        this(config, component, config.getKeyLocation(component));
    }

    public KeyStorage(SecurityConfig config, String component, String keyDirSuffix) throws IOException {
        this(config, component, Paths.get(config.getKeyLocation(component).toString() + keyDirSuffix, new String[0]));
    }

    private KeyStorage(SecurityConfig config, String component, Path keyPath) throws IOException {
        if (config.useExternalCACertificate(component)) {
            this.privateKeyPath = config.getExternalRootCaPrivateKeyPath();
            if (!Files.isReadable(this.privateKeyPath)) {
                throw new UnsupportedEncodingException("External private key path is not readable: " + this.privateKeyPath);
            }
            this.publicKeyPath = config.getExternalRootCaPublicKeyPath();
            if (!Files.isReadable(this.publicKeyPath)) {
                throw new UnsupportedEncodingException("External public key path is not readable: " + this.publicKeyPath);
            }
            this.externalKeysUsed = true;
        } else {
            this.createOrSanitizeDirectory(keyPath);
            this.privateKeyPath = keyPath.resolve(config.getPrivateKeyFileName());
            this.publicKeyPath = keyPath.resolve(config.getPublicKeyFileName());
            this.externalKeysUsed = false;
        }
        this.keyCodec = config.keyCodec();
    }

    public PrivateKey readPrivateKey() throws IOException {
        LOG.info("Reading private key from {}.", (Object)this.privateKeyPath);
        try {
            return this.keyCodec.decodePrivateKey(Files.readAllBytes(this.privateKeyPath));
        }
        catch (IOException e) {
            LOG.error("Failed to read the private key.", (Throwable)e);
            throw e;
        }
    }

    public PublicKey readPublicKey() throws IOException {
        LOG.info("Reading public key from {}.", (Object)this.publicKeyPath);
        try {
            return this.keyCodec.decodePublicKey(Files.readAllBytes(this.publicKeyPath));
        }
        catch (IOException e) {
            LOG.error("Failed to read the public key.", (Throwable)e);
            throw e;
        }
    }

    public KeyPair readKeyPair() throws IOException {
        return new KeyPair(this.readPublicKey(), this.readPrivateKey());
    }

    public void storePrivateKey(PrivateKey key) throws IOException {
        LOG.info("Storing private key to {}.", (Object)this.privateKeyPath);
        try {
            this.storeKey(this.privateKeyPath, this.keyCodec.encodePrivateKey(key));
        }
        catch (IOException e) {
            LOG.error("Failed to persist the private key.", (Throwable)e);
            throw e;
        }
    }

    public void storePublicKey(PublicKey key) throws IOException {
        LOG.info("Storing public key to {}.", (Object)this.publicKeyPath);
        try {
            this.storeKey(this.publicKeyPath, this.keyCodec.encodePublicKey(key));
        }
        catch (IOException e) {
            LOG.error("Failed to persist the public key.", (Throwable)e);
            throw e;
        }
    }

    public void storeKeyPair(KeyPair keyPair) throws IOException {
        this.storePublicKey(keyPair.getPublic());
        this.storePrivateKey(keyPair.getPrivate());
    }

    private void storeKey(Path keyPath, byte[] encodedKey) throws IOException {
        if (this.externalKeysUsed) {
            throw new UnsupportedOperationException("Attempt to override external keys.");
        }
        Files.createFile(keyPath, PosixFilePermissions.asFileAttribute(FILE_PERMISSIONS));
        Files.write(keyPath, encodedKey, new OpenOption[0]);
    }

    private void createOrSanitizeDirectory(Path dir) throws IOException {
        if (Files.exists(dir, new LinkOption[0])) {
            Files.setPosixFilePermissions(dir, DIR_PERMISSIONS);
        } else {
            Files.createDirectories(dir, PosixFilePermissions.asFileAttribute(DIR_PERMISSIONS));
        }
    }
}

