/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.x509.certificate.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.math.BigInteger;
import java.net.InetAddress;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.DomainValidator;
import org.apache.hadoop.hdds.security.SecurityConfig;
import org.apache.hadoop.hdds.security.exception.SCMSecurityException;
import org.apache.hadoop.hdds.security.x509.certificate.utils.CertificateSignRequest;
import org.apache.hadoop.hdds.security.x509.exception.CertificateException;
import org.apache.hadoop.ozone.OzoneSecurityUtil;
import org.apache.hadoop.util.Time;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DERUTF8String;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.BasicConstraints;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.KeyUsage;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SelfSignedCertificate {
    private String subject;
    private String clusterID;
    private String scmID;
    private LocalDateTime beginDate;
    private LocalDateTime endDate;
    private KeyPair key;
    private SecurityConfig config;
    private List<GeneralName> altNames;
    private static final Logger LOG = LoggerFactory.getLogger(SelfSignedCertificate.class);

    private SelfSignedCertificate(Builder builder) {
        this.subject = builder.subject;
        this.clusterID = builder.clusterID;
        this.scmID = builder.scmID;
        this.beginDate = builder.beginDate;
        this.endDate = builder.endDate;
        this.config = builder.config;
        this.key = builder.key;
        this.altNames = builder.altNames;
    }

    @VisibleForTesting
    public static String getNameFormat() {
        return CertificateSignRequest.getDistinguishedNameFormatWithSN();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private X509Certificate generateCertificate(BigInteger caCertSerialId) throws OperatorCreationException, IOException {
        byte[] encoded = this.key.getPublic().getEncoded();
        SubjectPublicKeyInfo publicKeyInfo = SubjectPublicKeyInfo.getInstance((Object)encoded);
        ContentSigner contentSigner = new JcaContentSignerBuilder(this.config.getSignatureAlgo()).setProvider(this.config.getProvider()).build(this.key.getPrivate());
        BigInteger serial = caCertSerialId == null ? new BigInteger(Long.toString(Time.monotonicNow())) : caCertSerialId;
        String dnName = String.format(SelfSignedCertificate.getNameFormat(), this.subject, this.scmID, this.clusterID, serial);
        X500Name name = new X500Name(dnName);
        Date validFrom = Date.from(this.beginDate.atZone(ZoneId.systemDefault()).toInstant());
        Date validTill = Date.from(this.endDate.atZone(ZoneId.systemDefault()).toInstant());
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(name, serial, validFrom, validTill, name, publicKeyInfo);
        if (caCertSerialId != null) {
            builder.addExtension(Extension.basicConstraints, true, (ASN1Encodable)new BasicConstraints(true));
            int keyUsageFlag = 6;
            KeyUsage keyUsage = new KeyUsage(keyUsageFlag);
            builder.addExtension(Extension.keyUsage, true, (ASN1Encodable)keyUsage);
            if (this.altNames != null && !this.altNames.isEmpty()) {
                builder.addExtension(new Extension(Extension.subjectAlternativeName, false, new GeneralNames(this.altNames.toArray(new GeneralName[this.altNames.size()])).getEncoded()));
            }
        }
        try {
            X509Certificate cert = new JcaX509CertificateConverter().getCertificate(builder.build(contentSigner));
            LOG.info("Certificate {} is issued by {} to {}, valid from {} to {}", new Object[]{cert.getSerialNumber(), cert.getIssuerDN(), cert.getSubjectDN(), cert.getNotBefore(), cert.getNotAfter()});
            return cert;
        }
        catch (java.security.cert.CertificateException e) {
            throw new CertificateException("Could not create self-signed X509Certificate.", (Throwable)e, CertificateException.ErrorCode.CERTIFICATE_ERROR);
        }
    }

    public static class Builder {
        private String subject;
        private String clusterID;
        private String scmID;
        private LocalDateTime beginDate;
        private LocalDateTime endDate;
        private KeyPair key;
        private SecurityConfig config;
        private BigInteger caCertSerialId;
        private List<GeneralName> altNames;

        public Builder setConfiguration(SecurityConfig configuration) {
            this.config = configuration;
            return this;
        }

        public Builder setKey(KeyPair keyPair) {
            this.key = keyPair;
            return this;
        }

        public Builder setSubject(String subjectString) {
            this.subject = subjectString;
            return this;
        }

        public Builder setClusterID(String s) {
            this.clusterID = s;
            return this;
        }

        public Builder setScmID(String s) {
            this.scmID = s;
            return this;
        }

        public Builder setBeginDate(LocalDateTime date) {
            this.beginDate = date;
            return this;
        }

        public Builder setEndDate(LocalDateTime date) {
            this.endDate = date;
            return this;
        }

        public Builder makeCA() {
            return this.makeCA(BigInteger.ONE);
        }

        public Builder makeCA(BigInteger serialId) {
            this.caCertSerialId = serialId;
            return this;
        }

        public Builder addInetAddresses() throws CertificateException {
            try {
                DomainValidator validator = DomainValidator.getInstance();
                List<InetAddress> inetAddresses = OzoneSecurityUtil.getValidInetsForCurrentHost();
                this.addInetAddresses(inetAddresses, validator);
            }
            catch (IOException e) {
                throw new CertificateException("Error while getting Inet addresses for the CSR builder", (Throwable)e, CertificateException.ErrorCode.CSR_ERROR);
            }
            return this;
        }

        public Builder addInetAddresses(List<InetAddress> addresses, DomainValidator validator) {
            addresses.forEach(ip -> {
                this.addIpAddress(ip.getHostAddress());
                if (validator.isValid(ip.getCanonicalHostName())) {
                    this.addDnsName(ip.getCanonicalHostName());
                } else {
                    LOG.error("Invalid domain {}", (Object)ip.getCanonicalHostName());
                }
            });
            return this;
        }

        public Builder addDnsName(String dnsName) {
            Preconditions.checkNotNull((Object)dnsName, (Object)"dnsName cannot be null");
            this.addAltName(2, dnsName);
            return this;
        }

        public Builder addIpAddress(String ip) {
            Preconditions.checkNotNull((Object)ip, (Object)"Ip address cannot be null");
            this.addAltName(7, ip);
            return this;
        }

        private Builder addAltName(int tag, String name) {
            if (this.altNames == null) {
                this.altNames = new ArrayList<GeneralName>();
            }
            if (tag == 0) {
                ASN1Object ono = this.addOtherNameAsn1Object(name);
                this.altNames.add(new GeneralName(tag, (ASN1Encodable)ono));
            } else {
                this.altNames.add(new GeneralName(tag, name));
            }
            return this;
        }

        private ASN1Object addOtherNameAsn1Object(String name) {
            String otherNameOID = "2.16.840.1.113730.3.1.34";
            ASN1EncodableVector otherName = new ASN1EncodableVector();
            otherName.add((ASN1Encodable)new ASN1ObjectIdentifier("2.16.840.1.113730.3.1.34"));
            otherName.add((ASN1Encodable)new DERTaggedObject(true, 0, (ASN1Encodable)new DERUTF8String(name)));
            return new DERTaggedObject(false, 0, (ASN1Encodable)new DERSequence(otherName));
        }

        public X509Certificate build() throws SCMSecurityException, IOException {
            Preconditions.checkNotNull((Object)this.key, (Object)"Key cannot be null");
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)this.subject), (Object)"Subject cannot be blank");
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)this.clusterID), (Object)"Cluster ID cannot be blank");
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((CharSequence)this.scmID), (Object)"SCM ID cannot be blank");
            Preconditions.checkArgument((boolean)this.beginDate.isBefore(this.endDate), (Object)"Certificate begin date should be before end date");
            Duration certDuration = Duration.between(this.beginDate, this.endDate);
            Duration maxDuration = this.config.getMaxCertificateDuration();
            if (certDuration.compareTo(maxDuration) > 0) {
                throw new SCMSecurityException("The cert duration violates the maximum configured value. Please check the hdds.x509.max.duration config key. Current Value: " + certDuration + " config: " + maxDuration);
            }
            SelfSignedCertificate rootCertificate = new SelfSignedCertificate(this);
            try {
                return rootCertificate.generateCertificate(this.caCertSerialId);
            }
            catch (CertIOException | OperatorCreationException e) {
                throw new CertificateException("Unable to create root certificate.", e.getCause());
            }
        }
    }
}

