/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.IOException;
import java.math.RoundingMode;
import java.net.InetAddress;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.management.ObjectName;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.DatanodeID;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.VersionInfo;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeMetric;
import org.apache.hadoop.hdds.scm.container.placement.metrics.SCMNodeStat;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.net.Node;
import org.apache.hadoop.hdds.scm.node.CommandQueue;
import org.apache.hadoop.hdds.scm.node.DatanodeInfo;
import org.apache.hadoop.hdds.scm.node.DatanodeUsageInfo;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStateManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.SCMNodeMetrics;
import org.apache.hadoop.hdds.scm.node.states.NodeAlreadyExistsException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.pipeline.PipelineNotFoundException;
import org.apache.hadoop.hdds.scm.server.SCMStorageConfig;
import org.apache.hadoop.hdds.scm.server.upgrade.FinalizationManager;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.apache.hadoop.hdds.upgrade.HDDSLayoutVersionManager;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.ozone.protocol.VersionResponse;
import org.apache.hadoop.ozone.protocol.commands.CommandForDatanode;
import org.apache.hadoop.ozone.protocol.commands.FinalizeNewLayoutVersionCommand;
import org.apache.hadoop.ozone.protocol.commands.RefreshVolumeUsageCommand;
import org.apache.hadoop.ozone.protocol.commands.RegisteredCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.ozone.protocol.commands.SetNodeOperationalStateCommand;
import org.apache.hadoop.ozone.upgrade.LayoutVersionManager;
import org.apache.hadoop.util.Time;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeManager
implements NodeManager {
    public static final Logger LOG = LoggerFactory.getLogger(SCMNodeManager.class);
    private final NodeStateManager nodeStateManager;
    private final VersionInfo version;
    private final CommandQueue commandQueue;
    private final SCMNodeMetrics metrics;
    private ObjectName nmInfoBean;
    private final SCMStorageConfig scmStorageConfig;
    private final NetworkTopology clusterMap;
    private final Function<String, String> nodeResolver;
    private final boolean useHostname;
    private final Map<String, Set<DatanodeID>> dnsToDnIdMap = new ConcurrentHashMap<String, Set<DatanodeID>>();
    private final int numPipelinesPerMetadataVolume;
    private final int heavyNodeCriteria;
    private final HDDSLayoutVersionManager scmLayoutVersionManager;
    private final EventPublisher scmNodeEventPublisher;
    private final SCMContext scmContext;
    private final Map<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, BiConsumer<DatanodeDetails, SCMCommand<?>>> sendCommandNotifyMap;
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private static final String OPESTATE = "OPSTATE";
    private static final String COMSTATE = "COMSTATE";
    private static final String LASTHEARTBEAT = "LASTHEARTBEAT";
    private static final String USEDSPACEPERCENT = "USEDSPACEPERCENT";
    private static final String TOTALCAPACITY = "CAPACITY";
    private static final String DNUUID = "UUID";
    private static final String VERSION = "VERSION";

    public SCMNodeManager(OzoneConfiguration conf, SCMStorageConfig scmStorageConfig, EventPublisher eventPublisher, NetworkTopology networkTopology, SCMContext scmContext, HDDSLayoutVersionManager layoutVersionManager) {
        this(conf, scmStorageConfig, eventPublisher, networkTopology, scmContext, layoutVersionManager, hostname -> null);
    }

    public SCMNodeManager(OzoneConfiguration conf, SCMStorageConfig scmStorageConfig, EventPublisher eventPublisher, NetworkTopology networkTopology, SCMContext scmContext, HDDSLayoutVersionManager layoutVersionManager, Function<String, String> nodeResolver) {
        this.scmNodeEventPublisher = eventPublisher;
        this.nodeStateManager = new NodeStateManager((ConfigurationSource)conf, eventPublisher, (LayoutVersionManager)layoutVersionManager, scmContext);
        this.version = VersionInfo.getLatestVersion();
        this.commandQueue = new CommandQueue();
        this.scmStorageConfig = scmStorageConfig;
        this.scmLayoutVersionManager = layoutVersionManager;
        LOG.info("Entering startup safe mode.");
        this.registerMXBean();
        this.metrics = SCMNodeMetrics.create(this);
        this.clusterMap = networkTopology;
        this.nodeResolver = nodeResolver;
        this.useHostname = conf.getBoolean("hdds.datanode.use.datanode.hostname", false);
        this.numPipelinesPerMetadataVolume = conf.getInt("ozone.scm.pipeline.per.metadata.disk", 2);
        String dnLimit = conf.get("ozone.scm.datanode.pipeline.limit");
        this.heavyNodeCriteria = dnLimit == null ? 0 : Integer.parseInt(dnLimit);
        this.scmContext = scmContext;
        this.sendCommandNotifyMap = new HashMap();
    }

    @Override
    public void registerSendCommandNotify(StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type type, BiConsumer<DatanodeDetails, SCMCommand<?>> scmCommand) {
        this.sendCommandNotifyMap.put(type, scmCommand);
    }

    private void registerMXBean() {
        this.nmInfoBean = MBeans.register((String)"SCMNodeManager", (String)"SCMNodeManagerInfo", (Object)this);
    }

    private void unregisterMXBean() {
        if (this.nmInfoBean != null) {
            MBeans.unregister((ObjectName)this.nmInfoBean);
            this.nmInfoBean = null;
        }
    }

    protected NodeStateManager getNodeStateManager() {
        return this.nodeStateManager;
    }

    @Override
    public List<DatanodeDetails> getNodes(NodeStatus nodeStatus) {
        return this.nodeStateManager.getNodes(nodeStatus).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getNodes(HddsProtos.NodeOperationalState opState, HddsProtos.NodeState health) {
        return this.nodeStateManager.getNodes(opState, health).stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public List<DatanodeDetails> getAllNodes() {
        return this.nodeStateManager.getAllNodes().stream().map(node -> node).collect(Collectors.toList());
    }

    @Override
    public int getNodeCount(NodeStatus nodeStatus) {
        return this.nodeStateManager.getNodeCount(nodeStatus);
    }

    @Override
    public int getNodeCount(HddsProtos.NodeOperationalState nodeOpState, HddsProtos.NodeState health) {
        return this.nodeStateManager.getNodeCount(nodeOpState, health);
    }

    @Override
    public NodeStatus getNodeStatus(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getNodeStatus(datanodeDetails);
    }

    @Override
    public void setNodeOperationalState(DatanodeDetails datanodeDetails, HddsProtos.NodeOperationalState newState) throws NodeNotFoundException {
        this.setNodeOperationalState(datanodeDetails, newState, 0L);
    }

    @Override
    public void setNodeOperationalState(DatanodeDetails datanodeDetails, HddsProtos.NodeOperationalState newState, long opStateExpiryEpocSec) throws NodeNotFoundException {
        this.nodeStateManager.setNodeOperationalState(datanodeDetails, newState, opStateExpiryEpocSec);
    }

    @Override
    public void close() throws IOException {
        this.unregisterMXBean();
        this.metrics.unRegister();
        this.nodeStateManager.close();
    }

    public VersionResponse getVersion(StorageContainerDatanodeProtocolProtos.SCMVersionRequestProto versionRequest) {
        return VersionResponse.newBuilder().setVersion(this.version.getVersion()).addValue("scmUuid", this.scmStorageConfig.getScmId()).addValue("clusterID", this.scmStorageConfig.getClusterID()).build();
    }

    @Override
    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto) {
        return this.register(datanodeDetails, nodeReport, pipelineReportsProto, StorageContainerDatanodeProtocolProtos.LayoutVersionProto.newBuilder().setMetadataLayoutVersion(this.scmLayoutVersionManager.getMetadataLayoutVersion()).setSoftwareLayoutVersion(this.scmLayoutVersionManager.getSoftwareLayoutVersion()).build());
    }

    public RegisteredCommand register(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport, StorageContainerDatanodeProtocolProtos.PipelineReportsProto pipelineReportsProto, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutInfo) {
        if (layoutInfo.getSoftwareLayoutVersion() != this.scmLayoutVersionManager.getSoftwareLayoutVersion()) {
            return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.errorNodeNotPermitted).setDatanode(datanodeDetails).setClusterID(this.scmStorageConfig.getClusterID()).build();
        }
        InetAddress dnAddress = Server.getRemoteIp();
        if (dnAddress != null) {
            if (!this.useHostname) {
                datanodeDetails.setHostName(dnAddress.getHostName());
            }
            datanodeDetails.setIpAddress(dnAddress.getHostAddress());
        }
        String ipAddress = datanodeDetails.getIpAddress();
        String hostName = datanodeDetails.getHostName();
        datanodeDetails.setNetworkName(datanodeDetails.getUuidString());
        String networkLocation = this.nodeResolver.apply(this.useHostname ? hostName : ipAddress);
        if (networkLocation != null) {
            datanodeDetails.setNetworkLocation(networkLocation);
        }
        DatanodeID dnId = datanodeDetails.getID();
        if (!this.isNodeRegistered(datanodeDetails).booleanValue()) {
            try {
                this.clusterMap.add((Node)datanodeDetails);
                this.nodeStateManager.addNode(datanodeDetails, layoutInfo);
                DatanodeInfo dn = this.nodeStateManager.getNode(datanodeDetails);
                Preconditions.checkState((dn.getParent() != null ? 1 : 0) != 0);
                this.addToDnsToDnIdMap(dnId, ipAddress, hostName);
                this.processNodeReport(datanodeDetails, nodeReport);
                LOG.info("Registered datanode: {}", (Object)datanodeDetails.toDebugString());
                this.scmNodeEventPublisher.fireEvent(SCMEvents.NEW_NODE, (Object)datanodeDetails);
            }
            catch (NodeAlreadyExistsException e) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Datanode is already registered: {}", (Object)datanodeDetails);
                }
            }
            catch (NodeNotFoundException e) {
                LOG.error("Cannot find datanode {} from nodeStateManager", (Object)datanodeDetails);
            }
        } else {
            try {
                DatanodeInfo oldNode = this.nodeStateManager.getNode(datanodeDetails);
                if (this.updateDnsToDnIdMap(oldNode.getHostName(), oldNode.getIpAddress(), hostName, ipAddress, dnId)) {
                    LOG.info("Updating datanode {} from {} to {}", new Object[]{datanodeDetails.getUuidString(), oldNode, datanodeDetails});
                    this.clusterMap.update((Node)oldNode, (Node)datanodeDetails);
                    this.nodeStateManager.updateNode(datanodeDetails, layoutInfo);
                    DatanodeInfo dn = this.nodeStateManager.getNode(datanodeDetails);
                    Preconditions.checkState((dn.getParent() != null ? 1 : 0) != 0);
                    this.processNodeReport(datanodeDetails, nodeReport);
                    LOG.info("Updated datanode to: {}", (Object)dn);
                    this.scmNodeEventPublisher.fireEvent(SCMEvents.NODE_ADDRESS_UPDATE, (Object)dn);
                } else if (this.isVersionChange(oldNode.getVersion(), datanodeDetails.getVersion())) {
                    LOG.info("Update the version for registered datanode = {}, oldVersion = {}, newVersion = {}.", new Object[]{datanodeDetails.getUuid(), oldNode.getVersion(), datanodeDetails.getVersion()});
                    this.nodeStateManager.updateNode(datanodeDetails, layoutInfo);
                }
            }
            catch (NodeNotFoundException e) {
                LOG.error("Cannot find datanode {} from nodeStateManager", (Object)datanodeDetails);
            }
        }
        return RegisteredCommand.newBuilder().setErrorCode(StorageContainerDatanodeProtocolProtos.SCMRegisteredResponseProto.ErrorCode.success).setDatanode(datanodeDetails).setClusterID(this.scmStorageConfig.getClusterID()).build();
    }

    private synchronized void addToDnsToDnIdMap(DatanodeID datanodeID, String ... addresses) {
        for (String addr : addresses) {
            if (Strings.isNullOrEmpty((String)addr)) continue;
            this.dnsToDnIdMap.computeIfAbsent(addr, k -> ConcurrentHashMap.newKeySet()).add(datanodeID);
        }
    }

    private synchronized void removeFromDnsToDnIdMap(DatanodeID datanodeID, String address) {
        Set<DatanodeID> dnSet;
        if (address != null && (dnSet = this.dnsToDnIdMap.get(address)) != null && dnSet.remove(datanodeID) && dnSet.isEmpty()) {
            this.dnsToDnIdMap.remove(address);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean updateDnsToDnIdMap(String oldHostName, String oldIpAddress, String newHostName, String newIpAddress, DatanodeID datanodeID) {
        boolean hostNameChanged;
        boolean ipChanged = !Objects.equals(oldIpAddress, newIpAddress);
        boolean bl = hostNameChanged = !Objects.equals(oldHostName, newHostName);
        if (ipChanged || hostNameChanged) {
            SCMNodeManager sCMNodeManager = this;
            synchronized (sCMNodeManager) {
                if (ipChanged) {
                    this.removeFromDnsToDnIdMap(datanodeID, oldIpAddress);
                    this.addToDnsToDnIdMap(datanodeID, newIpAddress);
                }
                if (hostNameChanged) {
                    this.removeFromDnsToDnIdMap(datanodeID, oldHostName);
                    this.addToDnsToDnIdMap(datanodeID, newHostName);
                }
            }
        }
        return ipChanged || hostNameChanged;
    }

    private boolean isVersionChange(String oldVersion, String newVersion) {
        boolean versionChanged = !Objects.equals(oldVersion, newVersion);
        return versionChanged;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<SCMCommand<?>> processHeartbeat(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.CommandQueueReportProto queueReport) {
        Preconditions.checkNotNull((Object)datanodeDetails, (Object)"Heartbeat is missing DatanodeDetails.");
        try {
            this.nodeStateManager.updateLastHeartbeatTime(datanodeDetails);
            this.metrics.incNumHBProcessed();
            this.updateDatanodeOpState(datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumHBProcessingFailed();
            LOG.error("SCM trying to process heartbeat from an unregistered node {}. Ignoring the heartbeat.", (Object)datanodeDetails);
        }
        this.writeLock().lock();
        try {
            Map<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> summary = this.commandQueue.getDatanodeCommandSummary(datanodeDetails.getUuid());
            List<SCMCommand<?>> commands = this.commandQueue.getCommand(datanodeDetails.getUuid());
            for (SCMCommand<?> command : commands) {
                if (this.sendCommandNotifyMap.get(command.getType()) == null) continue;
                this.sendCommandNotifyMap.get(command.getType()).accept(datanodeDetails, command);
            }
            if (queueReport != null) {
                this.processNodeCommandQueueReport(datanodeDetails, queueReport, summary);
            }
            List<SCMCommand<?>> list = commands;
            return list;
        }
        finally {
            this.writeLock().unlock();
        }
    }

    boolean opStateDiffers(DatanodeDetails dnDetails, NodeStatus nodeStatus) {
        return nodeStatus.getOperationalState() != dnDetails.getPersistedOpState() || nodeStatus.getOpStateExpiryEpochSeconds() != dnDetails.getPersistedOpStateExpiryEpochSec();
    }

    protected void updateDatanodeOpState(DatanodeDetails reportedDn) throws NodeNotFoundException {
        NodeStatus scmStatus = this.getNodeStatus(reportedDn);
        if (this.opStateDiffers(reportedDn, scmStatus)) {
            if (this.scmContext.isLeader()) {
                LOG.info("Scheduling a command to update the operationalState persisted on {} as the reported value ({}, {}) does not match the value stored in SCM ({}, {})", new Object[]{reportedDn, reportedDn.getPersistedOpState(), reportedDn.getPersistedOpStateExpiryEpochSec(), scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds()});
                try {
                    SetNodeOperationalStateCommand command = new SetNodeOperationalStateCommand(Time.monotonicNow(), scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds());
                    command.setTerm(this.scmContext.getTermOfLeader());
                    this.addDatanodeCommand(reportedDn.getUuid(), (SCMCommand<?>)command);
                }
                catch (NotLeaderException nle) {
                    LOG.warn("Skip sending SetNodeOperationalStateCommand, since current SCM is not leader.", (Throwable)nle);
                    return;
                }
            } else {
                LOG.info("Update the operationalState saved in follower SCM for {} as the reported value ({}, {}) does not match the value stored in SCM ({}, {})", new Object[]{reportedDn, reportedDn.getPersistedOpState(), reportedDn.getPersistedOpStateExpiryEpochSec(), scmStatus.getOperationalState(), scmStatus.getOpStateExpiryEpochSeconds()});
                this.setNodeOperationalState(reportedDn, reportedDn.getPersistedOpState(), reportedDn.getPersistedOpStateExpiryEpochSec());
            }
        }
        DatanodeInfo scmDnd = this.nodeStateManager.getNode(reportedDn);
        scmDnd.setPersistedOpStateExpiryEpochSec(reportedDn.getPersistedOpStateExpiryEpochSec());
        scmDnd.setPersistedOpState(reportedDn.getPersistedOpState());
    }

    public Boolean isNodeRegistered(DatanodeDetails datanodeDetails) {
        try {
            this.nodeStateManager.getNode(datanodeDetails);
            return true;
        }
        catch (NodeNotFoundException e) {
            return false;
        }
    }

    @Override
    public void processNodeReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing node report from [datanode={}]", (Object)datanodeDetails.getHostName());
        }
        if (LOG.isTraceEnabled() && nodeReport != null) {
            LOG.trace("HB is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)nodeReport.toString().replaceAll("\n", "\\\\n"));
        }
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            if (nodeReport != null) {
                datanodeInfo.updateStorageReports(nodeReport.getStorageReportList());
                datanodeInfo.updateMetaDataStorageReports(nodeReport.getMetadataStorageReportList());
                this.metrics.incNumNodeReportProcessed();
            }
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumNodeReportProcessingFailed();
            LOG.warn("Got node report from unregistered datanode {}", (Object)datanodeDetails);
        }
    }

    @Override
    public void processLayoutVersionReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutVersionReport) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing Layout Version report from [datanode={}]", (Object)datanodeDetails.getHostName());
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("HB is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)layoutVersionReport.toString().replaceAll("\n", "\\\\n"));
        }
        try {
            this.nodeStateManager.updateLastKnownLayoutVersion(datanodeDetails, layoutVersionReport);
        }
        catch (NodeNotFoundException e) {
            LOG.error("SCM trying to process Layout Version from an unregistered node {}.", (Object)datanodeDetails);
            return;
        }
        this.sendFinalizeToDatanodeIfNeeded(datanodeDetails, layoutVersionReport);
    }

    protected void sendFinalizeToDatanodeIfNeeded(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.LayoutVersionProto layoutVersionReport) {
        int scmMlv;
        int scmSlv = this.scmLayoutVersionManager.getSoftwareLayoutVersion();
        int dnSlv = layoutVersionReport.getSoftwareLayoutVersion();
        int dnMlv = layoutVersionReport.getMetadataLayoutVersion();
        if (dnSlv > scmSlv) {
            LOG.error("Invalid data node in the cluster : {}. DataNode SoftwareLayoutVersion = {}, SCM SoftwareLayoutVersion = {}", new Object[]{datanodeDetails.getHostName(), dnSlv, scmSlv});
        }
        if (FinalizationManager.shouldTellDatanodesToFinalize(this.scmContext.getFinalizationCheckpoint()) && dnMlv < (scmMlv = this.scmLayoutVersionManager.getMetadataLayoutVersion())) {
            LOG.warn("Data node {} can not be used in any pipeline in the cluster. DataNode MetadataLayoutVersion = {}, SCM MetadataLayoutVersion = {}", new Object[]{datanodeDetails.getHostName(), dnMlv, scmMlv});
            FinalizeNewLayoutVersionCommand finalizeCmd = new FinalizeNewLayoutVersionCommand(true, StorageContainerDatanodeProtocolProtos.LayoutVersionProto.newBuilder().setSoftwareLayoutVersion(dnSlv).setMetadataLayoutVersion(dnSlv).build());
            if (this.scmContext.isLeader()) {
                try {
                    finalizeCmd.setTerm(this.scmContext.getTermOfLeader());
                    this.scmNodeEventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND, (Object)new CommandForDatanode(datanodeDetails.getUuid(), (SCMCommand)finalizeCmd));
                }
                catch (NotLeaderException ex) {
                    LOG.warn("Skip sending finalize upgrade command since current SCM is not leader.", (Throwable)ex);
                }
            }
        }
    }

    private void processNodeCommandQueueReport(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.CommandQueueReportProto commandQueueReportProto, Map<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> commandsToBeSent) {
        LOG.debug("Processing Command Queue Report from [datanode={}]", (Object)datanodeDetails.getHostName());
        if (commandQueueReportProto == null) {
            LOG.debug("The Command Queue Report from [datanode={}] is null", (Object)datanodeDetails.getHostName());
            return;
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("Command Queue Report is received from [datanode={}]: <json>{}</json>", (Object)datanodeDetails.getHostName(), (Object)commandQueueReportProto.toString().replaceAll("\n", "\\\\n"));
        }
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            datanodeInfo.setCommandCounts(commandQueueReportProto, commandsToBeSent);
            this.metrics.incNumNodeCommandQueueReportProcessed();
            this.scmNodeEventPublisher.fireEvent(SCMEvents.DATANODE_COMMAND_COUNT_UPDATED, (Object)datanodeDetails);
        }
        catch (NodeNotFoundException e) {
            this.metrics.incNumNodeCommandQueueReportProcessingFailed();
            LOG.warn("Got Command Queue Report from unregistered datanode {}", (Object)datanodeDetails);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getNodeQueuedCommandCount(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type cmdType) throws NodeNotFoundException {
        this.readLock().lock();
        try {
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            int n = datanodeInfo.getCommandCount(cmdType);
            return n;
        }
        finally {
            this.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getCommandQueueCount(UUID dnID, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type cmdType) {
        this.readLock().lock();
        try {
            int n = this.commandQueue.getDatanodeCommandCount(dnID, cmdType);
            return n;
        }
        finally {
            this.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int getTotalDatanodeCommandCount(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type cmdType) throws NodeNotFoundException {
        this.readLock().lock();
        try {
            int dnCount = this.getNodeQueuedCommandCount(datanodeDetails, cmdType);
            if (dnCount == -1) {
                LOG.warn("No command count information for datanode {} and command {}. Assuming zero", (Object)datanodeDetails, (Object)cmdType);
                dnCount = 0;
            }
            int n = this.getCommandQueueCount(datanodeDetails.getUuid(), cmdType) + dnCount;
            return n;
        }
        finally {
            this.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Map<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> getTotalDatanodeCommandCounts(DatanodeDetails datanodeDetails, StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type ... cmdType) throws NodeNotFoundException {
        HashMap<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> counts = new HashMap<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer>();
        this.readLock().lock();
        try {
            for (StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type type : cmdType) {
                counts.put(type, this.getTotalDatanodeCommandCount(datanodeDetails, type));
            }
            HashMap<StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type, Integer> hashMap = counts;
            return hashMap;
        }
        finally {
            this.readLock().unlock();
        }
    }

    @Override
    public SCMNodeStat getStats() {
        long capacity = 0L;
        long used = 0L;
        long remaining = 0L;
        long committed = 0L;
        long freeSpaceToSpare = 0L;
        for (SCMNodeStat stat : this.getNodeStats().values()) {
            capacity += stat.getCapacity().get().longValue();
            used += stat.getScmUsed().get().longValue();
            remaining += stat.getRemaining().get().longValue();
            committed += stat.getCommitted().get().longValue();
            freeSpaceToSpare += stat.getFreeSpaceToSpare().get().longValue();
        }
        return new SCMNodeStat(capacity, used, remaining, committed, freeSpaceToSpare);
    }

    @Override
    public Map<DatanodeDetails, SCMNodeStat> getNodeStats() {
        HashMap<DatanodeDetails, SCMNodeStat> nodeStats = new HashMap<DatanodeDetails, SCMNodeStat>();
        List<DatanodeInfo> healthyNodes = this.nodeStateManager.getNodes(null, HddsProtos.NodeState.HEALTHY);
        List<DatanodeInfo> healthyReadOnlyNodes = this.nodeStateManager.getNodes(null, HddsProtos.NodeState.HEALTHY_READONLY);
        List<DatanodeInfo> staleNodes = this.nodeStateManager.getStaleNodes();
        ArrayList<DatanodeInfo> datanodes = new ArrayList<DatanodeInfo>(healthyNodes);
        datanodes.addAll(healthyReadOnlyNodes);
        datanodes.addAll(staleNodes);
        for (DatanodeInfo dnInfo : datanodes) {
            SCMNodeStat nodeStat = this.getNodeStatInternal(dnInfo);
            if (nodeStat == null) continue;
            nodeStats.put(dnInfo, nodeStat);
        }
        return nodeStats;
    }

    @Override
    public List<DatanodeUsageInfo> getMostOrLeastUsedDatanodes(boolean mostUsed) {
        List<DatanodeDetails> healthyNodes = this.getNodes(HddsProtos.NodeOperationalState.IN_SERVICE, HddsProtos.NodeState.HEALTHY);
        ArrayList<DatanodeUsageInfo> datanodeUsageInfoList = new ArrayList<DatanodeUsageInfo>(healthyNodes.size());
        for (DatanodeDetails node : healthyNodes) {
            DatanodeUsageInfo datanodeUsageInfo = this.getUsageInfo(node);
            datanodeUsageInfoList.add(datanodeUsageInfo);
        }
        if (mostUsed) {
            datanodeUsageInfoList.sort(DatanodeUsageInfo.getMostUtilized().reversed());
        } else {
            datanodeUsageInfoList.sort(DatanodeUsageInfo.getMostUtilized());
        }
        return datanodeUsageInfoList;
    }

    @Override
    public DatanodeUsageInfo getUsageInfo(DatanodeDetails dn) {
        SCMNodeStat stat = this.getNodeStatInternal(dn);
        DatanodeUsageInfo usageInfo = new DatanodeUsageInfo(dn, stat);
        try {
            usageInfo.setContainerCount(this.getContainerCount(dn));
            usageInfo.setPipelineCount(this.getPipeLineCount(dn));
        }
        catch (NodeNotFoundException ex) {
            LOG.error("Unknown datanode {}.", (Object)dn, (Object)ex);
        }
        return usageInfo;
    }

    @Override
    public SCMNodeMetric getNodeStat(DatanodeDetails datanodeDetails) {
        SCMNodeStat nodeStat = this.getNodeStatInternal(datanodeDetails);
        return nodeStat != null ? new SCMNodeMetric(nodeStat) : null;
    }

    private SCMNodeStat getNodeStatInternal(DatanodeDetails datanodeDetails) {
        try {
            long capacity = 0L;
            long used = 0L;
            long remaining = 0L;
            long committed = 0L;
            long freeSpaceToSpare = 0L;
            DatanodeInfo datanodeInfo = this.nodeStateManager.getNode(datanodeDetails);
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = datanodeInfo.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                capacity += reportProto.getCapacity();
                used += reportProto.getScmUsed();
                remaining += reportProto.getRemaining();
                committed += reportProto.getCommitted();
                freeSpaceToSpare += reportProto.getFreeSpaceToSpare();
            }
            return new SCMNodeStat(capacity, used, remaining, committed, freeSpaceToSpare);
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)datanodeDetails.getUuidString());
            return null;
        }
    }

    @Override
    public Map<String, Map<String, Integer>> getNodeCount() {
        HashMap<String, Map<String, Integer>> nodes = new HashMap<String, Map<String, Integer>>();
        for (HddsProtos.NodeOperationalState opState : HddsProtos.NodeOperationalState.values()) {
            HashMap<String, Integer> states = new HashMap<String, Integer>();
            for (HddsProtos.NodeState health : HddsProtos.NodeState.values()) {
                states.put(health.name(), 0);
            }
            nodes.put(opState.name(), states);
        }
        for (DatanodeInfo dni : this.nodeStateManager.getAllNodes()) {
            NodeStatus status = dni.getNodeStatus();
            ((Map)nodes.get(status.getOperationalState().name())).compute(status.getHealth().name(), (k, v) -> v + 1);
        }
        return nodes;
    }

    @Override
    public Map<String, Long> getNodeInfo() {
        HashMap<String, Long> nodeInfo = new HashMap<String, Long>();
        for (UsageStates s : UsageStates.values()) {
            for (UsageMetrics stat : UsageMetrics.values()) {
                nodeInfo.put(s.label + stat.name(), 0L);
            }
        }
        nodeInfo.put("TotalCapacity", 0L);
        nodeInfo.put("TotalUsed", 0L);
        for (DatanodeInfo node : this.nodeStateManager.getAllNodes()) {
            String keyPrefix = "";
            NodeStatus status = node.getNodeStatus();
            if (status.isMaintenance()) {
                keyPrefix = UsageStates.MAINT.getLabel();
            } else if (status.isDecommission()) {
                keyPrefix = UsageStates.DECOM.getLabel();
            } else {
                if (!status.isAlive()) continue;
                keyPrefix = UsageStates.ONLINE.getLabel();
            }
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReportProtos = node.getStorageReports();
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto reportProto : storageReportProtos) {
                if (reportProto.getStorageType() == HddsProtos.StorageTypeProto.DISK) {
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskCapacity.name(), (k, v) -> v + reportProto.getCapacity());
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskRemaining.name(), (k, v) -> v + reportProto.getRemaining());
                    nodeInfo.compute(keyPrefix + UsageMetrics.DiskUsed.name(), (k, v) -> v + reportProto.getScmUsed());
                } else if (reportProto.getStorageType() == HddsProtos.StorageTypeProto.SSD) {
                    nodeInfo.compute(keyPrefix + UsageMetrics.SSDCapacity.name(), (k, v) -> v + reportProto.getCapacity());
                    nodeInfo.compute(keyPrefix + UsageMetrics.SSDRemaining.name(), (k, v) -> v + reportProto.getRemaining());
                    nodeInfo.compute(keyPrefix + UsageMetrics.SSDUsed.name(), (k, v) -> v + reportProto.getScmUsed());
                }
                nodeInfo.compute("TotalCapacity", (k, v) -> v + reportProto.getCapacity());
                nodeInfo.compute("TotalUsed", (k, v) -> v + reportProto.getScmUsed());
            }
        }
        return nodeInfo;
    }

    @Override
    public Map<String, Map<String, String>> getNodeStatusInfo() {
        HashMap<String, Map<String, String>> nodes = new HashMap<String, Map<String, String>>();
        for (DatanodeInfo dni : this.nodeStateManager.getAllNodes()) {
            String hostName = dni.getHostName();
            DatanodeDetails.Port httpPort = dni.getPort(DatanodeDetails.Port.Name.HTTP);
            DatanodeDetails.Port httpsPort = dni.getPort(DatanodeDetails.Port.Name.HTTPS);
            String opstate = "";
            String healthState = "";
            String heartbeatTimeDiff = "";
            if (dni.getNodeStatus() != null) {
                opstate = dni.getNodeStatus().getOperationalState().toString();
                healthState = dni.getNodeStatus().getHealth().toString();
                heartbeatTimeDiff = this.getLastHeartbeatTimeDiff(dni.getLastHeartbeatTime());
            }
            HashMap<String, String> map = new HashMap<String, String>();
            map.put(OPESTATE, opstate);
            map.put(COMSTATE, healthState);
            map.put(LASTHEARTBEAT, heartbeatTimeDiff);
            if (httpPort != null) {
                map.put(httpPort.getName().toString(), httpPort.getValue().toString());
            }
            if (httpsPort != null) {
                map.put(httpsPort.getName().toString(), httpsPort.getValue().toString());
            }
            String capacity = SCMNodeManager.calculateStorageCapacity(dni.getStorageReports());
            map.put(TOTALCAPACITY, capacity);
            String[] storagePercentage = SCMNodeManager.calculateStoragePercentage(dni.getStorageReports());
            String scmUsedPerc = storagePercentage[0];
            String nonScmUsedPerc = storagePercentage[1];
            map.put(USEDSPACEPERCENT, "Ozone: " + scmUsedPerc + "%, other: " + nonScmUsedPerc + "%");
            map.put(DNUUID, dni.getUuidString());
            map.put(VERSION, dni.getVersion());
            nodes.put(hostName, map);
        }
        return nodes;
    }

    public static String calculateStorageCapacity(List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReports) {
        long capacityByte = 0L;
        if (storageReports != null && !storageReports.isEmpty()) {
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto storageReport : storageReports) {
                capacityByte += storageReport.getCapacity();
            }
        }
        return SCMNodeManager.convertUnit(capacityByte);
    }

    private static String convertUnit(double value) {
        StringBuilder unit = new StringBuilder("B");
        if (value > 1024.0) {
            value /= 1024.0;
            unit.replace(0, 1, "KB");
        }
        if (value > 1024.0) {
            value /= 1024.0;
            unit.replace(0, 2, "MB");
        }
        if (value > 1024.0) {
            value /= 1024.0;
            unit.replace(0, 2, "GB");
        }
        if (value > 1024.0) {
            value /= 1024.0;
            unit.replace(0, 2, "TB");
        }
        DecimalFormat decimalFormat = new DecimalFormat("#0.0");
        decimalFormat.setRoundingMode(RoundingMode.HALF_UP);
        String newValue = decimalFormat.format(value);
        return newValue + unit.toString();
    }

    public static String[] calculateStoragePercentage(List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReports) {
        String[] storagePercentage = new String[2];
        String usedPercentage = "N/A";
        String nonUsedPercentage = "N/A";
        if (storageReports != null && !storageReports.isEmpty()) {
            long capacity = 0L;
            long scmUsed = 0L;
            long remaining = 0L;
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto storageReport : storageReports) {
                capacity += storageReport.getCapacity();
                scmUsed += storageReport.getScmUsed();
                remaining += storageReport.getRemaining();
            }
            long scmNonUsed = capacity - scmUsed - remaining;
            DecimalFormat decimalFormat = new DecimalFormat("#0.00");
            decimalFormat.setRoundingMode(RoundingMode.HALF_UP);
            double usedPerc = (double)scmUsed / (double)capacity * 100.0;
            usedPerc = usedPerc > 100.0 ? 100.0 : usedPerc;
            double nonUsedPerc = (double)scmNonUsed / (double)capacity * 100.0;
            nonUsedPerc = nonUsedPerc > 100.0 ? 100.0 : nonUsedPerc;
            usedPercentage = decimalFormat.format(usedPerc);
            nonUsedPercentage = decimalFormat.format(nonUsedPerc);
        }
        storagePercentage[0] = usedPercentage;
        storagePercentage[1] = nonUsedPercentage;
        return storagePercentage;
    }

    @Override
    public Map<String, String> getNodeStatistics() {
        HashMap<String, String> nodeStatistics = new HashMap<String, String>();
        this.nodeUsageStatistics(nodeStatistics);
        this.nodeStateStatistics(nodeStatistics);
        this.nodeSpaceStatistics(nodeStatistics);
        return nodeStatistics;
    }

    private void nodeUsageStatistics(Map<String, String> nodeStatics) {
        if (this.nodeStateManager.getAllNodes().isEmpty()) {
            return;
        }
        float[] usages = new float[this.nodeStateManager.getAllNodes().size()];
        float totalOzoneUsed = 0.0f;
        int i = 0;
        for (DatanodeInfo dni : this.nodeStateManager.getAllNodes()) {
            String[] storagePercentage = SCMNodeManager.calculateStoragePercentage(dni.getStorageReports());
            if (storagePercentage[0].equals("N/A")) {
                usages[i++] = 0.0f;
                continue;
            }
            float storagePerc = Float.parseFloat(storagePercentage[0]);
            usages[i++] = storagePerc;
            totalOzoneUsed += storagePerc;
        }
        totalOzoneUsed /= (float)this.nodeStateManager.getAllNodes().size();
        Arrays.sort(usages);
        float median = usages[usages.length / 2];
        nodeStatics.put(UsageStatics.MEDIAN.getLabel(), String.valueOf(median));
        float max = usages[usages.length - 1];
        nodeStatics.put(UsageStatics.MAX.getLabel(), String.valueOf(max));
        float min = usages[0];
        nodeStatics.put(UsageStatics.MIN.getLabel(), String.valueOf(min));
        float dev = 0.0f;
        for (i = 0; i < usages.length; ++i) {
            dev += (usages[i] - totalOzoneUsed) * (usages[i] - totalOzoneUsed);
        }
        dev = (float)Math.sqrt(dev / (float)usages.length);
        DecimalFormat decimalFormat = new DecimalFormat("#0.00");
        decimalFormat.setRoundingMode(RoundingMode.HALF_UP);
        nodeStatics.put(UsageStatics.STDEV.getLabel(), decimalFormat.format(dev));
    }

    private void nodeStateStatistics(Map<String, String> nodeStatics) {
        int healthyNodeCount = this.nodeStateManager.getHealthyNodeCount();
        int deadNodeCount = this.nodeStateManager.getDeadNodeCount();
        int decommissioningNodeCount = this.nodeStateManager.getDecommissioningNodeCount();
        int enteringMaintenanceNodeCount = this.nodeStateManager.getEnteringMaintenanceNodeCount();
        int volumeFailuresNodeCount = this.nodeStateManager.getVolumeFailuresNodeCount();
        nodeStatics.put(StateStatistics.HEALTHY.getLabel(), String.valueOf(healthyNodeCount));
        nodeStatics.put(StateStatistics.DEAD.getLabel(), String.valueOf(deadNodeCount));
        nodeStatics.put(StateStatistics.DECOMMISSIONING.getLabel(), String.valueOf(decommissioningNodeCount));
        nodeStatics.put(StateStatistics.ENTERING_MAINTENANCE.getLabel(), String.valueOf(enteringMaintenanceNodeCount));
        nodeStatics.put(StateStatistics.VOLUME_FAILURES.getLabel(), String.valueOf(volumeFailuresNodeCount));
    }

    private void nodeSpaceStatistics(Map<String, String> nodeStatics) {
        if (this.nodeStateManager.getAllNodes().isEmpty()) {
            return;
        }
        long capacityByte = 0L;
        long scmUsedByte = 0L;
        long remainingByte = 0L;
        for (DatanodeInfo dni : this.nodeStateManager.getAllNodes()) {
            List<StorageContainerDatanodeProtocolProtos.StorageReportProto> storageReports = dni.getStorageReports();
            if (storageReports == null || storageReports.isEmpty()) continue;
            for (StorageContainerDatanodeProtocolProtos.StorageReportProto storageReport : storageReports) {
                capacityByte += storageReport.getCapacity();
                scmUsedByte += storageReport.getScmUsed();
                remainingByte += storageReport.getRemaining();
            }
        }
        long nonScmUsedByte = capacityByte - scmUsedByte - remainingByte;
        if (nonScmUsedByte < 0L) {
            nonScmUsedByte = 0L;
        }
        String capacity = SCMNodeManager.convertUnit(capacityByte);
        String scmUsed = SCMNodeManager.convertUnit(scmUsedByte);
        String remaining = SCMNodeManager.convertUnit(remainingByte);
        String nonScmUsed = SCMNodeManager.convertUnit(nonScmUsedByte);
        nodeStatics.put(SpaceStatistics.CAPACITY.getLabel(), capacity);
        nodeStatics.put(SpaceStatistics.SCM_USED.getLabel(), scmUsed);
        nodeStatics.put(SpaceStatistics.REMAINING.getLabel(), remaining);
        nodeStatics.put(SpaceStatistics.NON_SCM_USED.getLabel(), nonScmUsed);
    }

    public String getLastHeartbeatTimeDiff(long lastHeartbeatTime) {
        long currentTime = Time.monotonicNow();
        long timeDiff = currentTime - lastHeartbeatTime;
        long seconds = TimeUnit.MILLISECONDS.toSeconds(timeDiff);
        long days = TimeUnit.SECONDS.toDays(seconds);
        long hours = TimeUnit.SECONDS.toHours(seconds -= TimeUnit.DAYS.toSeconds(days));
        long minutes = TimeUnit.SECONDS.toMinutes(seconds -= TimeUnit.HOURS.toSeconds(hours));
        seconds -= TimeUnit.MINUTES.toSeconds(minutes);
        StringBuilder stringBuilder = new StringBuilder();
        if (days > 0L) {
            stringBuilder.append(days).append("d ");
        }
        if (hours > 0L) {
            stringBuilder.append(hours).append("h ");
        }
        if (minutes > 0L) {
            stringBuilder.append(minutes).append("m ");
        }
        if (seconds > 0L) {
            stringBuilder.append(seconds).append("s ");
        }
        String str = stringBuilder.length() == 0 ? "Just now" : "ago";
        stringBuilder.append(str);
        return stringBuilder.toString();
    }

    @Override
    public int minHealthyVolumeNum(List<DatanodeDetails> dnList) {
        ArrayList<Integer> volumeCountList = new ArrayList<Integer>(dnList.size());
        for (DatanodeDetails dn : dnList) {
            try {
                volumeCountList.add(this.nodeStateManager.getNode(dn).getHealthyVolumeCount());
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)dn.getUuid());
            }
        }
        Preconditions.checkArgument((!volumeCountList.isEmpty() ? 1 : 0) != 0);
        return (Integer)Collections.min(volumeCountList);
    }

    @Override
    public int totalHealthyVolumeCount() {
        int sum = 0;
        for (DatanodeInfo dn : this.nodeStateManager.getNodes(HddsProtos.NodeOperationalState.IN_SERVICE, HddsProtos.NodeState.HEALTHY)) {
            sum += dn.getHealthyVolumeCount();
        }
        return sum;
    }

    @Override
    public int pipelineLimit(DatanodeDetails dn) {
        try {
            if (this.heavyNodeCriteria > 0) {
                return this.heavyNodeCriteria;
            }
            if (this.nodeStateManager.getNode(dn).getHealthyVolumeCount() > 0) {
                return this.numPipelinesPerMetadataVolume * this.nodeStateManager.getNode(dn).getMetaDataVolumeCount();
            }
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot generate NodeStat, datanode {} not found.", (Object)dn.getUuid());
        }
        return 0;
    }

    @Override
    public int minPipelineLimit(List<DatanodeDetails> dnList) {
        ArrayList<Integer> pipelineCountList = new ArrayList<Integer>(dnList.size());
        for (DatanodeDetails dn : dnList) {
            pipelineCountList.add(this.pipelineLimit(dn));
        }
        Preconditions.checkArgument((!pipelineCountList.isEmpty() ? 1 : 0) != 0);
        return (Integer)Collections.min(pipelineCountList);
    }

    @Override
    public Collection<DatanodeDetails> getPeerList(DatanodeDetails dn) {
        HashSet<DatanodeDetails> dns = new HashSet<DatanodeDetails>();
        Preconditions.checkNotNull((Object)dn);
        Set<PipelineID> pipelines = this.nodeStateManager.getPipelineByDnID(dn.getUuid());
        PipelineManager pipelineManager = this.scmContext.getScm().getPipelineManager();
        if (!pipelines.isEmpty()) {
            pipelines.forEach(id -> {
                try {
                    Pipeline pipeline = pipelineManager.getPipeline((PipelineID)id);
                    List peers = pipeline.getNodes();
                    dns.addAll(peers);
                }
                catch (PipelineNotFoundException pipelineNotFoundException) {
                    // empty catch block
                }
            });
        }
        dns.remove(dn);
        return dns;
    }

    @Override
    public Set<PipelineID> getPipelines(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelineByDnID(datanodeDetails.getUuid());
    }

    @Override
    public int getPipelinesCount(DatanodeDetails datanodeDetails) {
        return this.nodeStateManager.getPipelinesCount(datanodeDetails);
    }

    @Override
    public void addPipeline(Pipeline pipeline) {
        this.nodeStateManager.addPipeline(pipeline);
    }

    @Override
    public void removePipeline(Pipeline pipeline) {
        this.nodeStateManager.removePipeline(pipeline);
    }

    @Override
    public void addContainer(DatanodeDetails datanodeDetails, ContainerID containerId) throws NodeNotFoundException {
        this.nodeStateManager.addContainer(datanodeDetails.getID(), containerId);
    }

    @Override
    public void removeContainer(DatanodeDetails datanodeDetails, ContainerID containerId) throws NodeNotFoundException {
        this.nodeStateManager.removeContainer(datanodeDetails.getID(), containerId);
    }

    @Override
    public void setContainers(DatanodeDetails datanodeDetails, Set<ContainerID> containerIds) throws NodeNotFoundException {
        this.nodeStateManager.setContainers(datanodeDetails.getID(), containerIds);
    }

    @Override
    public Set<ContainerID> getContainers(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getContainers(datanodeDetails.getID());
    }

    public int getContainerCount(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getContainerCount(datanodeDetails.getID());
    }

    public int getPipeLineCount(DatanodeDetails datanodeDetails) throws NodeNotFoundException {
        return this.nodeStateManager.getPipelinesCount(datanodeDetails);
    }

    @Override
    public void addDatanodeCommand(UUID dnId, SCMCommand<?> command) {
        this.writeLock().lock();
        try {
            this.commandQueue.addCommand(dnId, command);
        }
        finally {
            this.writeLock().unlock();
        }
    }

    @Override
    public void refreshAllHealthyDnUsageInfo() {
        RefreshVolumeUsageCommand refreshVolumeUsageCommand = new RefreshVolumeUsageCommand();
        try {
            refreshVolumeUsageCommand.setTerm(this.scmContext.getTermOfLeader());
        }
        catch (NotLeaderException nle) {
            LOG.warn("Skip sending refreshVolumeUsage command, since current SCM is not leader.", (Throwable)nle);
            return;
        }
        this.getNodes(HddsProtos.NodeOperationalState.IN_SERVICE, HddsProtos.NodeState.HEALTHY).forEach(datanode -> this.addDatanodeCommand(datanode.getUuid(), (SCMCommand<?>)refreshVolumeUsageCommand));
    }

    public void onMessage(CommandForDatanode commandForDatanode, EventPublisher ignored) {
        this.addDatanodeCommand(commandForDatanode.getDatanodeId(), commandForDatanode.getCommand());
    }

    @Override
    public List<SCMCommand<?>> getCommandQueue(UUID dnID) {
        this.writeLock().lock();
        try {
            List<SCMCommand<?>> list = this.commandQueue.getCommand(dnID);
            return list;
        }
        finally {
            this.writeLock().unlock();
        }
    }

    @Override
    public DatanodeDetails getNodeByUuid(String uuid) {
        return uuid != null && !uuid.isEmpty() ? this.getNodeByUuid(UUID.fromString(uuid)) : null;
    }

    @Override
    public DatanodeDetails getNodeByUuid(UUID uuid) {
        if (uuid == null) {
            return null;
        }
        try {
            return this.nodeStateManager.getNode(DatanodeID.of((UUID)uuid));
        }
        catch (NodeNotFoundException e) {
            LOG.warn("Cannot find node for uuid {}", (Object)uuid);
            return null;
        }
    }

    @Override
    public List<DatanodeDetails> getNodesByAddress(String address) {
        List<DatanodeDetails> allNodes = this.getAllNodes();
        LinkedList<DatanodeDetails> results = new LinkedList<DatanodeDetails>();
        if (Strings.isNullOrEmpty((String)address)) {
            LOG.warn("address is null");
            return results;
        }
        Set<DatanodeID> datanodeIDS = this.dnsToDnIdMap.get(address);
        if (datanodeIDS == null) {
            LOG.debug("Cannot find node for address {}", (Object)address);
            return results;
        }
        datanodeIDS.forEach(datanodeID -> {
            try {
                List datanodeDetails = allNodes.stream().filter(node -> node.getID().equals(datanodeID)).collect(Collectors.toList());
                results.addAll(datanodeDetails);
            }
            catch (Exception e) {
                LOG.warn("Error find node for DataNode ID {}", datanodeID);
            }
        });
        return results;
    }

    @Override
    public NetworkTopology getClusterNetworkTopologyMap() {
        return this.clusterMap;
    }

    @Override
    public long getLastHeartbeat(DatanodeDetails datanodeDetails) {
        try {
            DatanodeInfo node = this.nodeStateManager.getNode(datanodeDetails);
            return node.getLastHeartbeatTime();
        }
        catch (NodeNotFoundException e) {
            return -1L;
        }
    }

    @VisibleForTesting
    ScheduledFuture pauseHealthCheck() {
        return this.nodeStateManager.pause();
    }

    @VisibleForTesting
    ScheduledFuture unpauseHealthCheck() {
        return this.nodeStateManager.unpause();
    }

    @VisibleForTesting
    long getSkippedHealthChecks() {
        return this.nodeStateManager.getSkippedHealthChecks();
    }

    @Override
    @VisibleForTesting
    public HDDSLayoutVersionManager getLayoutVersionManager() {
        return this.scmLayoutVersionManager;
    }

    @Override
    @VisibleForTesting
    public void forceNodesToHealthyReadOnly() {
        this.nodeStateManager.forceNodesToHealthyReadOnly();
    }

    private ReentrantReadWriteLock.WriteLock writeLock() {
        return this.lock.writeLock();
    }

    private ReentrantReadWriteLock.ReadLock readLock() {
        return this.lock.readLock();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void removeNode(DatanodeDetails datanodeDetails) throws NodeNotFoundException, IOException {
        this.writeLock().lock();
        try {
            NodeStatus nodeStatus = this.getNodeStatus(datanodeDetails);
            if (datanodeDetails.isDecommissioned() || nodeStatus.isDead()) {
                if (this.clusterMap.contains((Node)datanodeDetails)) {
                    this.clusterMap.remove((Node)datanodeDetails);
                }
                this.nodeStateManager.removeNode(datanodeDetails.getID());
                this.removeFromDnsToDnIdMap(datanodeDetails.getID(), datanodeDetails.getIpAddress());
                List<SCMCommand<?>> cmdList = this.getCommandQueue(datanodeDetails.getUuid());
                LOG.info("Clearing command queue of size {} for DN {}", (Object)cmdList.size(), (Object)datanodeDetails);
            } else {
                LOG.warn("Node not decommissioned or dead, cannot remove: {}", (Object)datanodeDetails);
            }
        }
        finally {
            this.writeLock().unlock();
        }
    }

    private static enum SpaceStatistics {
        CAPACITY("Capacity"),
        SCM_USED("Scmused"),
        NON_SCM_USED("NonScmused"),
        REMAINING("Remaining");

        private String label;

        public String getLabel() {
            return this.label;
        }

        private SpaceStatistics(String label) {
            this.label = label;
        }
    }

    private static enum StateStatistics {
        HEALTHY("Healthy"),
        DEAD("Dead"),
        DECOMMISSIONING("Decommissioning"),
        ENTERING_MAINTENANCE("EnteringMaintenance"),
        VOLUME_FAILURES("VolumeFailures");

        private String label;

        public String getLabel() {
            return this.label;
        }

        private StateStatistics(String label) {
            this.label = label;
        }
    }

    private static enum UsageStatics {
        MIN("Min"),
        MAX("Max"),
        MEDIAN("Median"),
        STDEV("Stdev");

        private String label;

        public String getLabel() {
            return this.label;
        }

        private UsageStatics(String label) {
            this.label = label;
        }
    }

    private static enum UsageStates {
        ONLINE(""),
        MAINT("Maintenance"),
        DECOM("Decommissioned");

        private final String label;

        public String getLabel() {
            return this.label;
        }

        private UsageStates(String label) {
            this.label = label;
        }
    }

    private static enum UsageMetrics {
        DiskCapacity,
        DiskUsed,
        DiskRemaining,
        SSDCapacity,
        SSDUsed,
        SSDRemaining;

    }
}

