/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 */

package org.apache.hop.databases.snowflake;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.junit.jupiter.api.Assertions.fail;

import org.apache.hop.core.HopClientEnvironment;
import org.apache.hop.core.database.DatabaseMeta;
import org.apache.hop.junit.rules.RestoreHopEngineEnvironmentExtension;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

class SnowflakeDatabaseMetaTest {
  @RegisterExtension
  static RestoreHopEngineEnvironmentExtension env = new RestoreHopEngineEnvironmentExtension();

  SnowflakeDatabaseMeta nativeMeta;

  @BeforeEach
  void setupOnce() throws Exception {
    nativeMeta = new SnowflakeDatabaseMeta();
    nativeMeta.setAccessType(DatabaseMeta.TYPE_ACCESS_NATIVE);
    nativeMeta.addDefaultOptions();

    HopClientEnvironment.init();
  }

  @Test
  void testAccessType() {
    int[] aTypes = new int[] {DatabaseMeta.TYPE_ACCESS_NATIVE};
    assertArrayEquals(aTypes, nativeMeta.getAccessTypeList());
  }

  @Test
  void testUrl() {
    assertEquals("net.snowflake.client.jdbc.SnowflakeDriver", nativeMeta.getDriverClass());

    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com:1500/?db=bar",
        nativeMeta.getURL("account.snowflakecomputing.com", "1500", "bar"));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com:1500/?db=bar",
        nativeMeta.getURL("account", "1500", "bar"));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com/?db=bar",
        nativeMeta.getURL("account", "", "bar"));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com", nativeMeta.getURL("account", "", ""));

    nativeMeta.setWarehouse("wh");
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com:1500/?warehouse=wh&db=bar",
        nativeMeta.getURL("account.snowflakecomputing.com", "1500", "bar"));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com:1500/?warehouse=wh&db=bar",
        nativeMeta.getURL("account", "1500", "bar"));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com:1500/?warehouse=wh",
        nativeMeta.getURL("account", "1500", ""));
    assertEquals(
        "jdbc:snowflake://account.snowflakecomputing.com/?warehouse=wh&db=bar",
        nativeMeta.getURL("account", "", "bar"));

    try {
      assertEquals(
          "jdbc:snowflake://account.snowflakecomputing.com:1500/?db=bar",
          nativeMeta.getURL("", "1500", "bar"));
      fail("Should have thrown IllegalArgumentException");
    } catch (IllegalArgumentException dummy) {
      // expected if host is null or empty
    }
  }

  @Test
  void testSupport() {
    assertTrue(nativeMeta.isSupportsSchemas());
    assertTrue(nativeMeta.isSupportsViews());
    assertTrue(nativeMeta.isSupportsSequences());
    assertTrue(nativeMeta.IsSupportsErrorHandlingOnBatchUpdates());
    assertTrue(nativeMeta.isSupportsBooleanDataType());
    assertFalse(nativeMeta.isSupportsBitmapIndex());
    assertFalse(nativeMeta.isSupportsTransactions());
    assertFalse(nativeMeta.isSupportsSynonyms());
  }
}
