/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azure;

import com.microsoft.azure.storage.OperationContext;
import com.microsoft.azure.storage.StorageException;
import com.microsoft.azure.storage.blob.BlobRequestOptions;
import com.microsoft.azure.storage.blob.PageRange;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.azure.PageBlobFormatHelpers;
import org.apache.hadoop.fs.azure.StorageInterface;

final class PageBlobInputStream
extends InputStream {
    private static final Log LOG = LogFactory.getLog(PageBlobInputStream.class);
    private final StorageInterface.CloudPageBlobWrapper blob;
    private final OperationContext opContext;
    private long numberOfPagesRemaining;
    private long currentOffsetInBlob;
    private byte[] currentBuffer;
    private int currentBufferOffset;
    private int currentBufferLength;
    private static final int MAX_PAGES_PER_DOWNLOAD = 8192;
    private boolean closed = false;
    long pageBlobSize = -1L;
    long filePosition = 0L;

    public static long getPageBlobDataSize(StorageInterface.CloudPageBlobWrapper blob, OperationContext opContext) throws IOException, StorageException {
        ArrayList<PageRange> pageRanges = blob.downloadPageRanges(new BlobRequestOptions(), opContext);
        if (pageRanges.size() == 0) {
            return 0L;
        }
        if (pageRanges.get(0).getStartOffset() != 0L) {
            throw PageBlobInputStream.badStartRangeException(blob, pageRanges.get(0));
        }
        long totalRawBlobSize = pageRanges.get(0).getEndOffset() + 1L;
        long lastPageStart = totalRawBlobSize - 512L;
        ByteArrayOutputStream baos = new ByteArrayOutputStream(512);
        blob.downloadRange(lastPageStart, 512L, baos, new BlobRequestOptions(), opContext);
        byte[] lastPage = baos.toByteArray();
        short lastPageSize = PageBlobInputStream.getPageSize(blob, lastPage, 0);
        long totalNumberOfPages = totalRawBlobSize / 512L;
        return (totalNumberOfPages - 1L) * 510L + (long)lastPageSize;
    }

    public PageBlobInputStream(StorageInterface.CloudPageBlobWrapper blob, OperationContext opContext) throws IOException {
        ArrayList<PageRange> allRanges;
        this.blob = blob;
        this.opContext = opContext;
        try {
            allRanges = blob.downloadPageRanges(new BlobRequestOptions(), opContext);
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        if (allRanges.size() > 0) {
            if (allRanges.get(0).getStartOffset() != 0L) {
                throw PageBlobInputStream.badStartRangeException(blob, allRanges.get(0));
            }
            if (allRanges.size() > 1) {
                LOG.warn(String.format("Blob %s has %d page ranges beyond the first range. Only reading the first range.", blob.getUri(), allRanges.size() - 1));
            }
            this.numberOfPagesRemaining = (allRanges.get(0).getEndOffset() + 1L) / 512L;
        } else {
            this.numberOfPagesRemaining = 0L;
        }
    }

    @Override
    public synchronized int available() throws IOException {
        long remaining;
        if (this.closed) {
            throw new IOException("Stream closed");
        }
        if (this.pageBlobSize == -1L) {
            try {
                this.pageBlobSize = PageBlobInputStream.getPageBlobDataSize(this.blob, this.opContext);
            }
            catch (StorageException e) {
                throw new IOException("Unable to get page blob size.", e);
            }
        }
        return (remaining = this.pageBlobSize - this.filePosition) <= Integer.MAX_VALUE ? (int)remaining : Integer.MAX_VALUE;
    }

    @Override
    public synchronized void close() throws IOException {
        this.closed = true;
    }

    private boolean dataAvailableInBuffer() {
        return this.currentBuffer != null && this.currentBufferOffset < this.currentBufferLength;
    }

    private synchronized boolean ensureDataInBuffer() throws IOException {
        if (this.dataAvailableInBuffer()) {
            return true;
        }
        this.currentBuffer = null;
        this.currentBufferOffset = 0;
        this.currentBufferLength = 0;
        if (this.numberOfPagesRemaining == 0L) {
            return false;
        }
        long pagesToRead = Math.min(8192L, this.numberOfPagesRemaining);
        int bufferSize = (int)(pagesToRead * 512L);
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream(bufferSize);
            this.blob.downloadRange(this.currentOffsetInBlob, bufferSize, baos, PageBlobFormatHelpers.withMD5Checking(), this.opContext);
            this.validateDataIntegrity(baos.toByteArray());
        }
        catch (StorageException e) {
            throw new IOException(e);
        }
        this.numberOfPagesRemaining -= pagesToRead;
        this.currentOffsetInBlob += (long)bufferSize;
        return true;
    }

    private void validateDataIntegrity(byte[] buffer) throws IOException {
        if (buffer.length % 512 != 0) {
            throw new AssertionError((Object)("Unexpected buffer size: " + buffer.length));
        }
        int bufferLength = 0;
        int numberOfPages = buffer.length / 512;
        long totalPagesAfterCurrent = this.numberOfPagesRemaining;
        for (int page = 0; page < numberOfPages; ++page) {
            short currentPageSize = PageBlobInputStream.getPageSize(this.blob, buffer, page * 512);
            if (currentPageSize < 510 && --totalPagesAfterCurrent > 0L) {
                throw PageBlobInputStream.fileCorruptException(this.blob, String.format("Page with partial data found in the middle (%d pages from the end) that only has %d bytes of data.", totalPagesAfterCurrent, currentPageSize));
            }
            bufferLength += currentPageSize + 2;
        }
        this.currentBufferOffset = 2;
        this.currentBufferLength = bufferLength;
        this.currentBuffer = buffer;
    }

    private static short getPageSize(StorageInterface.CloudPageBlobWrapper blob, byte[] data, int offset) throws IOException {
        short pageSize = PageBlobFormatHelpers.toShort(data[offset], data[offset + 1]);
        if (pageSize < 0 || pageSize > 510) {
            throw PageBlobInputStream.fileCorruptException(blob, String.format("Unexpected page size in the header: %d.", pageSize));
        }
        return pageSize;
    }

    @Override
    public synchronized int read(byte[] outputBuffer, int offset, int len) throws IOException {
        if (len == 0) {
            return 0;
        }
        int numberOfBytesRead = 0;
        while (len > 0 && this.ensureDataInBuffer()) {
            int bytesRemainingInCurrentPage = this.getBytesRemainingInCurrentPage();
            int numBytesToRead = Math.min(len, bytesRemainingInCurrentPage);
            System.arraycopy(this.currentBuffer, this.currentBufferOffset, outputBuffer, offset, numBytesToRead);
            numberOfBytesRead += numBytesToRead;
            offset += numBytesToRead;
            len -= numBytesToRead;
            if (numBytesToRead == bytesRemainingInCurrentPage) {
                this.advancePagesInBuffer(1);
                continue;
            }
            this.currentBufferOffset += numBytesToRead;
        }
        if (numberOfBytesRead == 0) {
            return -1;
        }
        this.filePosition += (long)numberOfBytesRead;
        return numberOfBytesRead;
    }

    @Override
    public int read() throws IOException {
        byte[] oneByte = new byte[1];
        int result = this.read(oneByte);
        if (result < 0) {
            return result;
        }
        return oneByte[0];
    }

    @Override
    public synchronized long skip(long n) throws IOException {
        long skipped = this.skipImpl(n);
        this.filePosition += skipped;
        return skipped;
    }

    private long skipImpl(long n) throws IOException {
        if (n == 0L) {
            return 0L;
        }
        long skippedWithinBuffer = this.skipWithinBuffer(n);
        if (skippedWithinBuffer > n) {
            throw new AssertionError((Object)String.format("Bug in skipWithinBuffer: it skipped over %d bytes when asked to skip %d bytes.", skippedWithinBuffer, n));
        }
        long skipped = skippedWithinBuffer;
        if ((n -= skippedWithinBuffer) == 0L) {
            return skipped;
        }
        if (this.numberOfPagesRemaining == 0L) {
            throw new EOFException("Attempted to seek or read past the end of the file");
        }
        if (this.numberOfPagesRemaining > 1L) {
            long pagesToSkipOver = Math.min(n / 510L, this.numberOfPagesRemaining - 1L);
            this.numberOfPagesRemaining -= pagesToSkipOver;
            this.currentOffsetInBlob += pagesToSkipOver * 512L;
            skipped += pagesToSkipOver * 510L;
            n -= pagesToSkipOver * 510L;
        }
        if (n == 0L) {
            return skipped;
        }
        if (!this.ensureDataInBuffer()) {
            return skipped;
        }
        return skipped + this.skipWithinBuffer(n);
    }

    private long skipWithinBuffer(long n) throws IOException {
        if (!this.dataAvailableInBuffer()) {
            return 0L;
        }
        long skipped = 0L;
        skipped = this.skipWithinCurrentPage(n);
        if (skipped > n) {
            throw new AssertionError((Object)String.format("Bug in skipWithinCurrentPage: it skipped over %d bytes when asked to skip %d bytes.", skipped, n));
        }
        if ((n -= skipped) == 0L || !this.dataAvailableInBuffer()) {
            return skipped;
        }
        int numberOfPagesInBuffer = this.currentBuffer.length / 512;
        int currentPageIndex = this.currentBufferOffset / 512;
        int wholePagesRemaining = numberOfPagesInBuffer - currentPageIndex - 1;
        if (n < (long)(510 * wholePagesRemaining)) {
            this.advancePagesInBuffer((int)(n / 510L));
            this.currentBufferOffset = (int)((long)this.currentBufferOffset + n % 510L);
            return n + skipped;
        }
        this.advancePagesInBuffer(wholePagesRemaining);
        return this.skipWithinCurrentPage(n -= (long)(wholePagesRemaining * 510)) + (skipped += (long)(wholePagesRemaining * 510));
    }

    private long skipWithinCurrentPage(long n) throws IOException {
        int remainingBytesInCurrentPage = this.getBytesRemainingInCurrentPage();
        if (n <= (long)remainingBytesInCurrentPage) {
            this.currentBufferOffset = (int)((long)this.currentBufferOffset + n);
            return n;
        }
        this.advancePagesInBuffer(1);
        return remainingBytesInCurrentPage;
    }

    private int getBytesRemainingInCurrentPage() throws IOException {
        if (!this.dataAvailableInBuffer()) {
            return 0;
        }
        int currentDataOffsetInPage = this.currentBufferOffset % 512 - 2;
        int pageBoundary = this.getCurrentPageStartInBuffer();
        short sizeOfCurrentPage = PageBlobInputStream.getPageSize(this.blob, this.currentBuffer, pageBoundary);
        return sizeOfCurrentPage - currentDataOffsetInPage;
    }

    private static IOException badStartRangeException(StorageInterface.CloudPageBlobWrapper blob, PageRange startRange) {
        return PageBlobInputStream.fileCorruptException(blob, String.format("Page blobs for ASV should always use a page range starting at byte 0. This starts at byte %d.", startRange.getStartOffset()));
    }

    private void advancePagesInBuffer(int numberOfPages) {
        this.currentBufferOffset = this.getCurrentPageStartInBuffer() + numberOfPages * 512 + 2;
    }

    private int getCurrentPageStartInBuffer() {
        return 512 * (this.currentBufferOffset / 512);
    }

    private static IOException fileCorruptException(StorageInterface.CloudPageBlobWrapper blob, String reason) {
        return new IOException(String.format("The page blob: '%s' is corrupt or has an unexpected format: %s.", blob.getUri(), reason));
    }
}

