/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.bulkwriter;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Range;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.cassandra.bridge.CassandraBridgeFactory;
import org.apache.cassandra.bridge.CassandraVersion;
import org.apache.cassandra.bridge.CassandraVersionFeatures;
import org.apache.cassandra.bridge.SSTableDescriptor;
import org.apache.cassandra.bridge.SSTableWriter;
import org.apache.cassandra.spark.bulkwriter.BulkWriterContext;
import org.apache.cassandra.spark.bulkwriter.SSTableWriterFactory;
import org.apache.cassandra.spark.bulkwriter.SchemaInfo;
import org.apache.cassandra.spark.bulkwriter.TableSchema;
import org.apache.cassandra.spark.common.Digest;
import org.apache.cassandra.spark.common.SSTables;
import org.apache.cassandra.spark.data.FileType;
import org.apache.cassandra.spark.data.LocalDataLayer;
import org.apache.cassandra.spark.data.partitioner.Partitioner;
import org.apache.cassandra.spark.reader.StreamScanner;
import org.apache.cassandra.spark.sparksql.filters.SSTableTimeRangeFilter;
import org.apache.cassandra.spark.utils.DigestAlgorithm;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortedSSTableWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(SortedSSTableWriter.class);
    public static final String CASSANDRA_VERSION_PREFIX = "cassandra-";
    private final Path outDir;
    private final SSTableWriter cqlSSTableWriter;
    private final int partitionId;
    private final DigestAlgorithm digestAlgorithm;
    private BigInteger minToken = null;
    private BigInteger maxToken = null;
    private long rowCount = 0L;
    private final Map<Path, Digest> overallFileDigests = new HashMap<Path, Digest>();
    private boolean isClosed = false;
    private int sstableCount = 0;
    private long bytesWritten = 0L;

    public SortedSSTableWriter(SSTableWriter tableWriter, Path outDir, DigestAlgorithm digestAlgorithm, int partitionId) {
        this.cqlSSTableWriter = tableWriter;
        this.outDir = outDir;
        this.digestAlgorithm = digestAlgorithm;
        this.partitionId = partitionId;
    }

    public SortedSSTableWriter(BulkWriterContext writerContext, Path outDir, DigestAlgorithm digestAlgorithm, int partitionId) {
        this.outDir = outDir;
        this.digestAlgorithm = digestAlgorithm;
        this.partitionId = partitionId;
        String lowestCassandraVersion = writerContext.cluster().getLowestCassandraVersion();
        String packageVersion = this.getPackageVersion(lowestCassandraVersion);
        LOGGER.info("Running with version {}", (Object)packageVersion);
        SchemaInfo schema = writerContext.schema();
        TableSchema tableSchema = schema.getTableSchema();
        this.cqlSSTableWriter = SSTableWriterFactory.getSSTableWriter(CassandraVersionFeatures.cassandraVersionFeaturesFromCassandraVersion((String)packageVersion), this.outDir.toString(), writerContext.cluster().getPartitioner().toString(), tableSchema.createStatement, tableSchema.modificationStatement, schema.getUserDefinedTypeStatements(), writerContext.job().sstableDataSizeInMiB());
    }

    @NotNull
    public String getPackageVersion(String lowestCassandraVersion) {
        return CASSANDRA_VERSION_PREFIX + lowestCassandraVersion;
    }

    public void addRow(BigInteger token, Map<String, Object> boundValues) throws IOException {
        if (this.rowCount == 0L) {
            this.minToken = token;
        }
        this.maxToken = token;
        this.cqlSSTableWriter.addRow(boundValues);
        ++this.rowCount;
    }

    public void setSSTablesProducedListener(Consumer<Set<SSTableDescriptor>> listener) {
        this.cqlSSTableWriter.setSSTablesProducedListener(listener);
    }

    public long rowCount() {
        return this.rowCount;
    }

    public long bytesWritten() {
        return this.bytesWritten;
    }

    public int sstableCount() {
        return this.sstableCount;
    }

    public synchronized Map<Path, Digest> prepareSStablesToSend(@NotNull BulkWriterContext writerContext, Set<SSTableDescriptor> sstables) throws IOException {
        if (this.isClosed) {
            LOGGER.debug("Writer is already closed, returning empty digest map. Remaining SSTables will be handled by sendRemainingSSTables()");
            return Collections.emptyMap();
        }
        DirectoryStream.Filter<Path> sstableFilter = path -> {
            SSTableDescriptor baseName = SSTables.getSSTableDescriptor((Path)path);
            return sstables.contains(baseName) && !this.overallFileDigests.containsKey(path);
        };
        HashSet<Path> dataFilePaths = new HashSet<Path>();
        HashMap<Path, Digest> fileDigests = new HashMap<Path, Digest>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.getOutDir(), sstableFilter);){
            for (Path path2 : stream) {
                if (path2.getFileName().toString().endsWith("-" + FileType.DATA.getFileSuffix())) {
                    dataFilePaths.add(path2);
                    ++this.sstableCount;
                }
                Digest digest = this.digestAlgorithm.calculateFileDigest(path2);
                fileDigests.put(path2, digest);
                LOGGER.debug("Calculated digest={} for path={}", (Object)digest, (Object)path2);
            }
        }
        this.bytesWritten += this.calculatedTotalSize(fileDigests.keySet());
        this.overallFileDigests.putAll(fileDigests);
        this.validateSSTables(writerContext, this.getOutDir(), dataFilePaths);
        return fileDigests;
    }

    public synchronized void close(BulkWriterContext writerContext) throws IOException {
        if (this.isClosed) {
            LOGGER.info("Already closed");
            return;
        }
        this.isClosed = true;
        this.cqlSSTableWriter.close();
        HashSet<Path> hashedFiles = new HashSet<Path>(this.overallFileDigests.keySet());
        HashSet<Path> newlyHashedFiles = new HashSet<Path>();
        DirectoryStream.Filter<Path> unhashedFilter = path -> !hashedFiles.contains(path);
        for (Path dataFile : this.getDataFileStream(unhashedFilter)) {
            Map<Path, Digest> newFileDigests = this.calculateFileDigestMap(dataFile);
            this.overallFileDigests.putAll(newFileDigests);
            newlyHashedFiles.addAll(newFileDigests.keySet());
            ++this.sstableCount;
        }
        this.bytesWritten += this.calculatedTotalSize(newlyHashedFiles);
        this.validateSSTables(writerContext);
    }

    @VisibleForTesting
    public void validateSSTables(@NotNull BulkWriterContext writerContext) {
        this.validateSSTables(writerContext, this.getOutDir(), null);
    }

    @VisibleForTesting
    public void validateSSTables(@NotNull BulkWriterContext writerContext, @NotNull Path outputDirectory, @Nullable Set<Path> dataFilePaths) {
        try {
            CassandraVersion version = CassandraBridgeFactory.getCassandraVersion((String)writerContext.cluster().getLowestCassandraVersion());
            String keyspace = writerContext.job().qualifiedTableName().keyspace();
            String schema = writerContext.schema().getTableSchema().createStatement;
            Partitioner partitioner = writerContext.cluster().getPartitioner();
            Set<String> udtStatements = writerContext.schema().getUserDefinedTypeStatements();
            LocalDataLayer layer = new LocalDataLayer(version, partitioner, keyspace, schema, udtStatements, Collections.emptyList(), false, null, SSTableTimeRangeFilter.ALL, outputDirectory.toString());
            if (dataFilePaths != null) {
                layer.setDataFilePaths(dataFilePaths);
            }
            try (StreamScanner scanner = layer.openCompactionScanner(this.partitionId, Collections.emptyList(), null);){
                while (scanner.next()) {
                    scanner.advanceToNextColumn();
                }
            }
        }
        catch (IOException exception) {
            LOGGER.error("[{}]: Unexpected exception while validating SSTables {}", (Object)this.partitionId, (Object)this.getOutDir());
            throw new RuntimeException(exception);
        }
    }

    private DirectoryStream<Path> getDataFileStream(DirectoryStream.Filter<Path> filter) throws IOException {
        DirectoryStream.Filter<Path> combinedFilter = path -> {
            String fileName = path.getFileName().toString();
            return fileName.endsWith("Data.db") && filter.accept((Path)path);
        };
        return Files.newDirectoryStream(this.getOutDir(), combinedFilter);
    }

    private Map<Path, Digest> calculateFileDigestMap(Path dataFile) throws IOException {
        HashMap<Path, Digest> fileHashes = new HashMap<Path, Digest>();
        try (DirectoryStream<Path> filesToHash = Files.newDirectoryStream(dataFile.getParent(), SSTables.getSSTableBaseName((Path)dataFile) + "*");){
            for (Path path : filesToHash) {
                Digest digest = this.digestAlgorithm.calculateFileDigest(path);
                fileHashes.put(path, digest);
                LOGGER.debug("Calculated digest={} for path={}", (Object)digest, (Object)path);
            }
        }
        return fileHashes;
    }

    private long calculatedTotalSize(Collection<Path> paths) throws IOException {
        long totalSize = 0L;
        for (Path path : paths) {
            totalSize += Files.size(path);
        }
        return totalSize;
    }

    public Range<BigInteger> getTokenRange() {
        return Range.closed((Comparable)this.minToken, (Comparable)this.maxToken);
    }

    public Path getOutDir() {
        return this.outDir;
    }

    public Map<Path, Digest> fileDigestMap() {
        return Collections.unmodifiableMap(this.overallFileDigests);
    }
}

