# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
from __future__ import annotations

import msgspec
from flask import request
from flask.sessions import SecureCookieSessionInterface
from flask_babel import LazyString
from flask_session.sqlalchemy import SqlAlchemySessionInterface


class _LazySafeSerializer:
    def dumps(self, session_dict):
        encoder = msgspec.msgpack.Encoder(
            enc_hook=lambda obj: str(obj) if isinstance(obj, LazyString) else obj
        )
        return encoder.encode(dict(session_dict))

    def loads(self, data):
        decoder = msgspec.msgpack.Decoder()
        return decoder.decode(data)

    # optional old API
    encode = dumps
    decode = loads


class SessionExemptMixin:
    """Exempt certain blueprints/paths from autogenerated sessions."""

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.serializer = _LazySafeSerializer()

    def save_session(self, *args, **kwargs):
        """Prevent creating session from REST API and health requests."""
        if request.path == "/health":
            return None
        return super().save_session(*args, **kwargs)


class AirflowDatabaseSessionInterface(SessionExemptMixin, SqlAlchemySessionInterface):
    """Session interface that exempts some routes and stores session data in the database."""


class AirflowSecureCookieSessionInterface(SessionExemptMixin, SecureCookieSessionInterface):
    """Session interface that exempts some routes and stores session data in a signed cookie."""
