# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module TestMetricsExporter
      module Utils
        # Instance of metrics exporter configuration
        #
        # @return [Config]
        def config
          Config.configuration
        end

        # Configured logger instance
        #
        # @return [Logger]
        def logger
          config.logger
        end

        # Configured clickhouse client
        #
        # @return [ClickHouse::Client]
        def clickhouse_client
          ClickHouse::Client.new(
            url: config.clickhouse_config.url,
            database: config.clickhouse_config.database,
            username: config.clickhouse_config.username,
            password: config.clickhouse_config.password,
            logger: logger
          )
        end

        # GCS client instance
        #
        # @return [Fog::Storage::Google, GcsTools::GCSMockClient]
        def gcs_client
          GcsTools.gcs_client(
            project_id: config.gcs_config.project_id,
            credentials: config.gcs_config.credentials,
            dry_run: config.gcs_config.dry_run
          )
        end

        # Create table for metrics export using current ClickHouse configuration
        #
        # This method is mostly for schema documentation but it can be used together with initial_run! method in
        #
        # @return [void]
        def create_clickhouse_metrics_table
          table_name = config.clickhouse_config.table_name

          clickhouse_client.query(<<~SQL)
            CREATE TABLE IF NOT EXISTS #{table_name}
            (
                timestamp DateTime64(6, 'UTC'),
                id String,
                name String,
                hash String,
                file_path String,
                status LowCardinality(String),
                run_time UInt32,
                location String,
                quarantined Bool,
                test_retried Bool,
                feature_category LowCardinality(String) DEFAULT 'unknown',
                run_type LowCardinality(String) DEFAULT 'unknown',
                spec_file_path_prefix LowCardinality(String) DEFAULT '',
                ci_project_id UInt32,
                ci_job_name LowCardinality(String),
                ci_job_id UInt64,
                ci_pipeline_id UInt64,
                ci_merge_request_iid UInt32 DEFAULT 0,
                ci_project_path LowCardinality(String),
                ci_branch String,
                ci_target_branch LowCardinality(String),
                ci_server_url LowCardinality(String) DEFAULT 'https://gitlab.com',
                exception_class String DEFAULT '',
                exception_classes Array(String) DEFAULT [],
                failure_exception String DEFAULT ''
            )
            ENGINE = MergeTree()
            PARTITION BY toYYYYMM(timestamp)
            ORDER BY (ci_project_path, status, run_type, feature_category, file_path, timestamp, ci_pipeline_id)
            SETTINGS index_granularity = 8192;
          SQL
          return if config.extra_metadata_columns.empty?

          clickhouse_client.query(
            "ALTER TABLE #{table_name} #{config.extra_metadata_columns.map { |column| "ADD COLUMN IF NOT EXISTS #{column}" }.join(', ')};"
          )
        end

        module_function :config, :logger, :clickhouse_client
      end
    end
  end
end
