# frozen_string_literal: true

require "rspec/core/formatters/base_formatter"

module GitlabQuality
  module TestTooling
    module TestMetricsExporter
      class Formatter < RSpec::Core::Formatters::BaseFormatter
        include Utils

        RSpec::Core::Formatters.register(self, :start, :stop)

        LOG_PREFIX = "[MetricsExporter]"

        def start(_notification)
          return unless config.initial_run

          logger.info("#{LOG_PREFIX} Running initial setup for metrics export")
          raise "Initial setup is enabled, but clickhouse configuration is missing!" unless clickhouse_config

          create_clickhouse_metrics_table
        rescue StandardError => e
          logger.error("#{LOG_PREFIX} Error occurred during initial setup: #{e.message}")
        end

        def stop(notification)
          logger.debug("#{LOG_PREFIX} Starting test metrics export")
          data = notification.examples.filter_map do |example|
            next if config.skip_record_proc.call(example)

            TestMetrics.new(example, time).data
          end
          return logger.warn("#{LOG_PREFIX} No test execution records found, metrics will not be exported!") if data.empty?

          push_to_gcs(data)
          push_to_clickhouse(data)
        end

        private

        delegate :gcs_config, :clickhouse_config, to: :config

        # Single common timestamp for all exported example metrics to keep data points consistently grouped
        #
        # @return [String]
        def time
          return @time if @time

          ci_created_at = Runtime::Env.ci_pipeline_created_at
          @time = (ci_created_at ? Time.strptime(ci_created_at, '%Y-%m-%dT%H:%M:%S%z') : Time.now.utc).strftime('%Y-%m-%dT%H:%M:%S.%6N')
        end

        # Push data to gcs
        #
        # @param data [Array]
        # @return [void]
        def push_to_gcs(data)
          return logger.debug("#{LOG_PREFIX} GCS configuration missing, skipping gcs export!") unless gcs_config

          gcs_client.put_object(gcs_config.bucket_name, gcs_config.metrics_file_name, data.to_json)
          logger.info("#{LOG_PREFIX} Successfully pushed #{data.size} entries to GCS bucket!")
        rescue StandardError => e
          logger.error("#{LOG_PREFIX} Error occurred while pushing metrics to GCS: #{e.message}")
        end

        # Push data to clickhouse
        #
        # @param data [Array<Hash>]
        # @return [void]
        def push_to_clickhouse(data)
          return logger.debug("ClickHouse configuration missing, skipping ClickHouse export!") unless clickhouse_config

          clickhouse_client.insert_json_data(clickhouse_config.table_name, data)
          logger.info("#{LOG_PREFIX} Successfully pushed #{data.size} entries to ClickHouse!")
        rescue StandardError => e
          logger.error("#{LOG_PREFIX} Error occurred while pushing metrics to ClickHouse: #{e.message}")
        end
      end
    end
  end
end
