# frozen_string_literal: true

module Gitlab
  # Module containing GitLab's application theme definitions and helper methods
  # for accessing them.
  module Themes
    extend self

    # Theme ID used when no `default_theme` configuration setting is provided.
    APPLICATION_DEFAULT = 3

    # Theme IDs previously used
    DEPRECATED_THEME_IDS = [
      6, # Light indigo
      7, # Light blue
      8, # Light green
      10, # Light red
      11 # Dark mode theme
    ].freeze

    # Struct class representing a single Theme
    Theme = Struct.new(:id, :name, :css_class, :primary_color, :primary_color_dark)

    # All available Themes
    def available_themes
      [
        Theme.new(3, s_('NavigationTheme|Default'), 'ui-neutral', '#F1F0F6', '#232128'),
        Theme.new(1, s_('NavigationTheme|Indigo'), 'ui-indigo', '#F4F0FF', '#25202F'),
        Theme.new(4, s_('NavigationTheme|Blue'), 'ui-blue', '#E9F3FC', '#1B202F'),
        Theme.new(5, s_('NavigationTheme|Green'), 'ui-green', '#ECF4EE', '#1C2527'),
        Theme.new(9, s_('NavigationTheme|Red'), 'ui-red', '#FCF1EF', '#2C1D1F'),
        Theme.new(2, s_('NavigationTheme|Gray'), 'ui-gray', '#ECECEF', '#232227')
      ]
    end

    # Convenience method to get a space-separated String of all the theme
    # classes that might be applied to the `body` element
    #
    # Returns a String
    def body_classes
      available_themes.collect(&:css_class).uniq.join(' ')
    end

    # Maps deprecated light themes to their default counterpart
    def map_deprecated_themes
      {
        # Light indigo to indigo
        6 => 1,
        # Light blue to blue
        7 => 4,
        # Light green to green
        8 => 5,
        # Light red to red
        10 => 9
      }
    end

    # Get a Theme by its ID
    #
    # If the ID is invalid, returns the default Theme.
    #
    # id - Integer ID
    #
    # Returns a Theme
    def by_id(id)
      # Map deprecated IDs to new values
      mapped_id = map_deprecated_themes[id] || id

      available_themes.detect { |t| t.id == mapped_id } || default
    end

    # Returns the number of defined Themes
    def count
      available_themes.size
    end

    # Get the default Theme
    #
    # Returns a Theme
    def default
      by_id(default_id)
    end

    # Iterate through each Theme
    #
    # Yields the Theme object
    def each(&block)
      available_themes.each(&block)
    end

    # Get the Theme for the specified user, or the default
    #
    # user - User record
    #
    # Returns a Theme
    def for_user(user)
      if user
        by_id(user.theme_id)
      else
        default
      end
    end

    def self.valid_ids
      available_themes.map(&:id) + DEPRECATED_THEME_IDS
    end

    private

    def default_id
      @default_id ||= begin
        id = Gitlab.config.gitlab.default_theme.to_i
        theme_ids = available_themes.map(&:id)

        theme_ids.include?(id) ? id : APPLICATION_DEFAULT
      end
    end
  end
end
