/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.program.util;

import ghidra.framework.model.DomainObjectChangeRecord;
import ghidra.program.model.address.Address;

/**
 * Event data for a DomainObjectChangeEvent generated by a Program.
 */
public class ProgramChangeRecord extends DomainObjectChangeRecord {
	private Address start;
	private Address end;
	private Object affectedObj;

	/**
	 * Construct a new ProgramChangeRecord; any of the Address or
	 * Object params may be null, depending on what the type param is.
	 * @param eventType event type
	 * @param start starting address that is affected by the event
	 * @param end ending address that is affected by the event
	 * @param affected the object that was affected by this change, if applicable
	 * @param oldValue original value
	 * @param newValue new value
	 */
	public ProgramChangeRecord(ProgramEvent eventType, Address start, Address end, Object affected,
			Object oldValue, Object newValue) {
		super(eventType, oldValue, newValue);
		this.start = start;
		this.end = end;
		this.affectedObj = affected;
	}

	public ProgramChangeRecord(ProgramEvent eventType, Object oldValue, Object newValue) {
		this(eventType, null, null, null, oldValue, newValue);
	}

	/**
	 * Get the start address of the affected addresses of this change or null if not applicable. 
	 * @return the start address of the effected address of this change
	 */
	public Address getStart() {
		return start;
	}

	/**
	 * Get the end address of the affected addresses of this change or null if not applicable. 
	 * @return the end address of the effected address of this change
	 */
	public Address getEnd() {
		return end;
	}

	/**
	 * Return the object that is the subject of this change record.
	 * @return the object affected or null if not applicable
	 */
	public Object getObject() {
		return affectedObj;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder(super.toString());
		if (affectedObj != null) {
			builder.append(", affected = " + affectedObj);
		}
		if (start != null) {
			builder.append(", start = ");
			builder.append(start);
		}
		if (end != null) {
			builder.append(", end = ");
			builder.append(end);

		}
		return builder.toString();
	}

}
