/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { collectSortedFragments } from './measure-element.js';
export const processLayout = (node, callbacks) => {
    const { initialState, onNewline, onSpace, onText } = callbacks;
    if ((node.nodeName !== 'ink-box' &&
        node.nodeName !== 'ink-root' &&
        node.nodeName !== 'ink-text' &&
        node.nodeName !== 'ink-virtual-text') ||
        !node.yogaNode) {
        return { state: initialState(), lineBottom: 0 };
    }
    const { fragments } = collectSortedFragments(node);
    const state = {
        ...initialState(),
        currentX: 0,
        lineBottom: 0,
        yieldedContent: false,
    };
    for (const fragment of fragments) {
        if (fragment.y >= state.lineBottom) {
            const gap = fragment.y - state.lineBottom;
            const newlines = state.yieldedContent ? 1 + gap : gap;
            if (newlines > 0) {
                onNewline(newlines, state);
                state.currentX = 0;
            }
            state.lineBottom = fragment.y;
        }
        if (fragment.x > state.currentX) {
            const spaces = fragment.x - state.currentX;
            onSpace(spaces, state);
            state.currentX = fragment.x;
        }
        if (fragment.text.length > 0) {
            state.yieldedContent = true;
            onText(fragment, state);
        }
        const newlinesInText = (fragment.text.match(/\n/g) ?? []).length;
        if (newlinesInText > 0) {
            const lastNewlineIndex = fragment.text.lastIndexOf('\n');
            state.currentX = fragment.text.length - lastNewlineIndex - 1;
        }
        else {
            state.currentX += fragment.text.length;
        }
        state.lineBottom = Math.max(state.lineBottom, fragment.y + fragment.height);
    }
    return { state, lineBottom: state.lineBottom };
};
//# sourceMappingURL=layout.js.map