/*
 * Decompiled with CFR 0.152.
 */
package org.hibernate.engine.jdbc.connections.internal;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.HibernateException;
import org.hibernate.engine.jdbc.connections.internal.ConnectionCreator;
import org.hibernate.engine.jdbc.connections.internal.ConnectionValidator;
import org.hibernate.internal.log.ConnectionInfoLogger;

class PooledConnections {
    private final ConcurrentLinkedQueue<Connection> allConnections = new ConcurrentLinkedQueue();
    private final ConcurrentLinkedQueue<Connection> availableConnections = new ConcurrentLinkedQueue();
    private final ConnectionCreator connectionCreator;
    private final ConnectionValidator connectionValidator;
    private final boolean autoCommit;
    private final int minSize;
    private final int maxSize;
    private volatile boolean primed;

    private PooledConnections(Builder builder) {
        ConnectionInfoLogger.INSTANCE.debugf("Initializing Connection pool with %s Connections", builder.initialSize);
        this.connectionCreator = builder.connectionCreator;
        this.connectionValidator = builder.connectionValidator == null ? ConnectionValidator.ALWAYS_VALID : builder.connectionValidator;
        this.autoCommit = builder.autoCommit;
        this.maxSize = builder.maxSize;
        this.minSize = builder.minSize;
        this.addConnections(builder.initialSize);
    }

    void validate() {
        int size = this.size();
        if (!this.primed && size >= this.minSize) {
            ConnectionInfoLogger.INSTANCE.debug("Connection pool now considered primed; min-size will be maintained");
            this.primed = true;
        }
        if (size < this.minSize && this.primed) {
            int numberToBeAdded = this.minSize - size;
            ConnectionInfoLogger.INSTANCE.debugf("Adding %s Connections to the pool", numberToBeAdded);
            this.addConnections(numberToBeAdded);
        } else if (size > this.maxSize) {
            int numberToBeRemoved = size - this.maxSize;
            ConnectionInfoLogger.INSTANCE.debugf("Removing %s Connections from the pool", numberToBeRemoved);
            this.removeConnections(numberToBeRemoved);
        }
    }

    void add(Connection conn) {
        Connection connection = this.releaseConnection(conn);
        if (connection != null) {
            this.availableConnections.offer(connection);
        }
    }

    private Connection releaseConnection(Connection conn) {
        SQLException t = null;
        try {
            conn.setAutoCommit(true);
            conn.clearWarnings();
            if (this.connectionValidator.isValid(conn)) {
                return conn;
            }
        }
        catch (SQLException ex) {
            t = ex;
        }
        this.closeConnection(conn, t);
        ConnectionInfoLogger.INSTANCE.debug("Connection release failed. Closing pooled connection", t);
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Connection poll() {
        Connection conn;
        do {
            if ((conn = this.availableConnections.poll()) != null) continue;
            ConcurrentLinkedQueue<Connection> concurrentLinkedQueue = this.allConnections;
            synchronized (concurrentLinkedQueue) {
                if (this.allConnections.size() < this.maxSize) {
                    this.addConnections(1);
                    return this.poll();
                }
            }
            throw new HibernateException("The internal connection pool has reached its maximum size and no connection is currently available");
        } while ((conn = this.prepareConnection(conn)) == null);
        return conn;
    }

    protected Connection prepareConnection(Connection conn) {
        SQLException t = null;
        try {
            conn.setAutoCommit(this.autoCommit);
            if (this.connectionValidator.isValid(conn)) {
                return conn;
            }
        }
        catch (SQLException ex) {
            t = ex;
        }
        this.closeConnection(conn, t);
        ConnectionInfoLogger.INSTANCE.debug("Connection preparation failed. Closing pooled connection", t);
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void closeConnection(Connection conn, Throwable t) {
        try {
            conn.close();
        }
        catch (SQLException ex) {
            ConnectionInfoLogger.INSTANCE.unableToClosePooledConnection(ex);
            if (t != null) {
                t.addSuppressed(ex);
            }
        }
        finally {
            if (!this.allConnections.remove(conn)) {
                ConnectionInfoLogger.INSTANCE.debug("Connection remove failed.");
            }
        }
    }

    public void close() throws SQLException {
        try {
            int allocationCount = this.allConnections.size() - this.availableConnections.size();
            if (allocationCount > 0) {
                ConnectionInfoLogger.INSTANCE.error("Connection leak detected: there are " + allocationCount + " unclosed connections upon shutting down pool " + this.getUrl());
            }
        }
        finally {
            this.removeConnections(Integer.MAX_VALUE);
        }
    }

    public int size() {
        return this.allConnections.size();
    }

    protected void removeConnections(int numberToBeRemoved) {
        Connection connection;
        for (int i = 0; i < numberToBeRemoved && (connection = this.availableConnections.poll()) != null; ++i) {
            this.closeConnection(connection, null);
        }
    }

    protected void addConnections(int numberOfConnections) {
        for (int i = 0; i < numberOfConnections; ++i) {
            Connection connection = this.connectionCreator.createConnection();
            this.allConnections.add(connection);
            this.availableConnections.add(connection);
        }
    }

    public String getUrl() {
        return this.connectionCreator.getUrl();
    }

    int getOpenConnectionCount() {
        return this.allConnections.size() - this.availableConnections.size();
    }

    public Iterable<Connection> getAllConnections() {
        return this.allConnections;
    }

    static class Builder {
        private final ConnectionCreator connectionCreator;
        private ConnectionValidator connectionValidator;
        private boolean autoCommit;
        private int initialSize = 1;
        private int minSize = 1;
        private int maxSize = 20;

        Builder(ConnectionCreator connectionCreator) {
            this.connectionCreator = connectionCreator;
        }

        Builder autoCommit(boolean autoCommit) {
            this.autoCommit = autoCommit;
            return this;
        }

        Builder initialSize(int initialSize) {
            this.initialSize = initialSize;
            return this;
        }

        Builder minSize(int minSize) {
            this.minSize = minSize;
            return this;
        }

        Builder maxSize(int maxSize) {
            this.maxSize = maxSize;
            return this;
        }

        Builder validator(ConnectionValidator connectionValidator) {
            this.connectionValidator = connectionValidator;
            return this;
        }

        PooledConnections build() {
            return new PooledConnections(this);
        }
    }
}

