import { jsxs, Fragment, jsx } from 'react/jsx-runtime';
import { css } from '@emotion/css';
import { useState, useMemo, useEffect } from 'react';
import { useAsyncFn, useDebounce } from 'react-use';
import { getValueFormat, formattedValueToString } from '@grafana/data';
import { useTheme2, Spinner, Icon } from '@grafana/ui';
import '../types.js';

function QueryValidator({ db, query, onValidate, range }) {
  var _a;
  const [validationResult, setValidationResult] = useState();
  const theme = useTheme2();
  const valueFormatter = useMemo(() => getValueFormat("bytes"), []);
  const styles = useMemo(() => {
    return {
      error: css`
        color: ${theme.colors.error.text};
        font-size: ${theme.typography.bodySmall.fontSize};
        font-family: ${theme.typography.fontFamilyMonospace};
      `,
      valid: css`
        color: ${theme.colors.success.text};
      `,
      info: css`
        color: ${theme.colors.text.secondary};
      `
    };
  }, [theme]);
  const [state, validateQuery] = useAsyncFn(
    async (q) => {
      var _a2;
      if (((_a2 = q.rawSql) == null ? undefined : _a2.trim()) === "") {
        return null;
      }
      return await db.validateQuery(q, range);
    },
    [db]
  );
  useDebounce(
    async () => {
      const result = await validateQuery(query);
      if (result) {
        setValidationResult(result);
      }
      return null;
    },
    1e3,
    [query, validateQuery]
  );
  useEffect(() => {
    if (validationResult == null ? undefined : validationResult.isError) {
      onValidate(false);
    }
    if (validationResult == null ? undefined : validationResult.isValid) {
      onValidate(true);
    }
  }, [validationResult, onValidate]);
  if (!state.value && !state.loading) {
    return null;
  }
  const error = ((_a = state.value) == null ? undefined : _a.error) ? processErrorMessage(state.value.error) : "";
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    state.loading && /* @__PURE__ */ jsxs("div", { className: styles.info, children: [
      /* @__PURE__ */ jsx(Spinner, { inline: true, size: 12 }),
      " Validating query..."
    ] }),
    !state.loading && state.value && /* @__PURE__ */ jsxs(Fragment, { children: [
      /* @__PURE__ */ jsx(Fragment, { children: state.value.isValid && state.value.statistics && /* @__PURE__ */ jsxs("div", { className: styles.valid, children: [
        /* @__PURE__ */ jsx(Icon, { name: "check" }),
        " This query will process",
        " ",
        /* @__PURE__ */ jsx("strong", { children: formattedValueToString(valueFormatter(state.value.statistics.TotalBytesProcessed)) }),
        " ",
        "when run."
      ] }) }),
      /* @__PURE__ */ jsx(Fragment, { children: state.value.isError && /* @__PURE__ */ jsx("div", { className: styles.error, children: error }) })
    ] })
  ] });
}
function processErrorMessage(error) {
  const splat = error.split(":");
  if (splat.length > 2) {
    return splat.slice(2).join(":");
  }
  return error;
}

export { QueryValidator };
//# sourceMappingURL=QueryValidator.js.map
