# frozen_string_literal: true

require 'pp'
require 'stringio'

module GitlabQuality
  module TestTooling
    module Report
      module FeatureReadiness
        class ReportOnEpic
          FEATURE_READINESS_REPORT_COMMENT_ID = '<!-- FEATURE READINESS REPORT COMMENT -->'

          class << self
            include GitlabQuality::TestTooling::FeatureReadiness::Concerns::WorkItemConcern

            def report(analyzed_epic, work_item_client)
              must_haves_report_rows = generate_report_rows(analyzed_epic, :must_haves)
              should_haves_report_rows = generate_report_rows(analyzed_epic, :should_haves)

              existing_note = existing_note_containing_text(FEATURE_READINESS_REPORT_COMMENT_ID, analyzed_epic[:epic_iid], work_item_client)

              if existing_note
                work_item_client.update_note(note_id: existing_note[:id],
                  body: comment({ must_haves: must_haves_report_rows, should_haves: should_haves_report_rows }, analyzed_epic).tr('"', "'"))
              else
                work_item_client.create_discussion(id: analyzed_epic[:epic_id],
                  note: comment({ must_haves: must_haves_report_rows, should_haves: should_haves_report_rows }, analyzed_epic).tr('"', "'"))
              end
            end

            private

            def generate_report_rows(epic, type)
              status_checks = check_statuses(epic)
              create_rows(epic, type, status_checks)
            end

            def create_rows(epic, type, status_checks)
              if type == :must_haves
                [
                  create_documentation_row(epic, status_checks),
                  create_feature_flag_row(epic, status_checks),
                  create_unit_tests_coverage_row(status_checks)

                ]
              else
                [
                  create_feature_tests_row(epic, status_checks),
                  create_e2e_tests_row(epic, status_checks)
                ]
              end
            end

            def create_documentation_row(epic, status_checks)
              ["Documentation added?", status_icon(status_checks[:has_docs]),
                prepend_text('Added in:', format_links(epic[:doc_mrs]))]
            end

            def create_feature_flag_row(epic, status_checks)
              ["Feature Flag added?", status_icon(status_checks[:feature_flag_added]),
                prepend_text('Added in:', format_links(epic[:feature_flag_mrs]))]
            end

            def create_feature_tests_row(epic, status_checks)
              ["Feature tests added?", status_icon(status_checks[:has_feature_specs]),
                format_links(epic[:feature_spec_mrs])]
            end

            def create_e2e_tests_row(epic, status_checks)
              ["End-to-end tests added?", status_icon(status_checks[:has_e2e_specs]),
                format_links(epic[:e2e_spec_mrs])]
            end

            def create_unit_tests_coverage_row(status_checks)
              ["Unit tests coverage complete?", status_icon(status_checks[:has_complete_unit_tests]),
                prepend_text('Coverage missing for:', format_links(status_checks[:missing_specs]))]
            end

            def prepend_text(prepend_text, text)
              return "#{prepend_text} #{text}" unless text.empty?

              text
            end

            def check_statuses(epic)
              {
                has_docs: epic[:doc_mrs].any?,
                feature_flag_added: epic[:feature_flag_mrs].any?,
                has_feature_specs: epic[:feature_spec_mrs].any?,
                has_e2e_specs: epic[:e2e_spec_mrs].any?,
                missing_specs: missing_spec_mrs(epic),
                has_complete_unit_tests: missing_spec_mrs(epic).empty?
              }
            end

            def comment(rows, epic)
              # Generate markdown table
              must_haves_table_rows = rows[:must_haves].map do |description, status, links|
                "| #{description} | #{status} | #{links} |"
              end.join("\n")

              should_haves_table_rows = rows[:should_haves].map do |description, status, links|
                "| #{description} | #{status} | #{links} |"
              end.join("\n")

              <<~COMMENT
                #{FEATURE_READINESS_REPORT_COMMENT_ID}

                # :vertical_traffic_light: Feature Readiness Evaluation Report

                ### :octagonal_sign: Must haves

                | Evaluation | Result | Notes |
                |------------|--------|-------|
                #{must_haves_table_rows}

                ### :warning: Should haves

                | Evaluation | Result | Notes |
                |------------|--------|-------|
                #{should_haves_table_rows}

                #{data(epic)}

                ---

                _Please note that this automation is under testing. Please add any feedback on [this issue](https://gitlab.com/gitlab-org/quality/quality-engineering/team-tasks/-/issues/3587)._

              COMMENT
            end

            def status_icon(condition)
              condition ? ':white_check_mark:' : ':x:'
            end

            def format_links(data)
              return '' if data.empty?

              data.map do |item|
                item.map { |key, url| "[#{key}](#{url})" }.first
              end.join(", ")
            end

            def missing_spec_mrs(epic)
              epic[:issues].flat_map do |issue|
                issue[:merge_requests].flat_map do |mr|
                  next [] unless mr[:files_with_missing_specs]&.any?

                  mr[:files_with_missing_specs].map do |file|
                    { file => mr[:merge_request_web_url] }
                  end
                end.compact
              end
            end

            def data(epic)
              output = StringIO.new
              PP.pp(epic, output)
              <<~DATA
                <details><summary>Expand for data</summary>

                ```ruby
                #{output.string}
                ```

                </details>
              DATA
            end
          end
        end
      end
    end
  end
end
