# frozen_string_literal: true

require 'rest-client'
require 'json'

module GitlabQuality
  module TestTooling
    module GitlabClient
      class GitlabGraphqlClient
        def initialize(token:, project:, group:, endpoint: nil)
          @token = token
          @project = project
          @group = group
          @endpoint = endpoint || Runtime::Env.gitlab_graphql_api_base
        end

        def post(payload)
          payload = { query: payload } if payload.is_a?(String)
          request_args = {
            method: :post,
            url: endpoint,
            payload: payload,
            headers: { 'Authorization' => "Bearer #{token}" },
            verify_ssl: false
          }
          extract_graphql_body(RestClient::Request.execute(request_args))
        rescue StandardError => e
          return_response_or_raise(e)
        end

        private

        attr_reader :project, :token, :endpoint, :group

        def return_response_or_raise(error)
          return error.response if error.respond_to?(:response) && error.response

          raise error
        end

        def parse_body(response)
          JSON.parse(response.body, symbolize_names: true)
        end

        def extract_graphql_body(graphql_response)
          parsed_body = parse_body(graphql_response)

          data = parsed_body[:data].to_h
          data.values[0].to_h
        end
      end
    end
  end
end
