#!/usr/bin/env python
import logging
import operator
import statistics

from dataclasses import dataclass
from enum import Enum
from typing import List
from typing import Optional

import pandas as pd
import numpy as np

from nsys_cpu_stats.report_exporter import ReportExporter, DatabaseIds
from nsys_cpu_stats.trace_processor import TraceProcessor, TraceLoaderType
from nsys_cpu_stats.trace_loader import TraceLoaderRegions, TraceLoaderSupport, TraceLoaderGPUMetrics, TraceLoaderEvents
from nsys_cpu_stats.trace_utils import FrameDurations, TimeSlice, CallStack
from nsys_cpu_stats.trace_progress_indicator import g_progress_indicator as progress
import nsys_cpu_stats.trace_utils as tu
from server.app.models.hotspot_analysis.region import RegionCreate, RegionCustomFieldsCreate, RegionPresentThreadsCreate, RegionSubmitThreadsCreate
from server.app.models.hotspot_analysis.region_app_dxg_krnl_profile_ranges import RegionAppDxgKernelProfileRangesCreate
from server.app.models.hotspot_analysis.region_etw_events import RegionEtwEventsCreate
from server.app.models.hotspot_analysis.region_issues import RegionIssuesCreate
from server.app.models.hotspot_analysis.region_metrics import RegionMetricsCreate
from server.app.models.hotspot_analysis.region_thread import RegionThreadCreate
from server.app.models.hotspot_analysis.thread_callstacks import ThreadCallstackCreate
from server.app.models.hotspot_analysis.thread_hotspots import ThreadHotspotCollectionCreate, ThreadHotspotRecordCreate

logger = logging.getLogger(__name__)

class AnnotationMixin:
    @classmethod
    def annotations(cls):
        return {c.name.lower(): c for c in cls}

    @classmethod
    def search(cls, m):
        for name, metric_type in HotspotReportThreading.annotations().items():
            if m in name:
                return metric_type
        return None


class HotspotRange(AnnotationMixin, Enum):
    NONE = 'Full App'
    SLOWEST = 'Slow Frames'
    PCIE_BAR1_READS = 'Bar1 Reads'
    PCIE_BAR1_WRITES = 'Bar1 Writes'
    GR_IDLE = 'GR Idle'
    PERIODIC = 'Periodic Frames'
    CUDNN_KERNEL_LAUNCHES = 'CUDNN CPU Kernel Launch Regions'
    CUDNN_GPU_KERNELS = 'CUDNN GPU Kernel Execution Regions'

    def __str__(self):
        return self.value


class HotspotReportThreading(AnnotationMixin, Enum):
    ALL = 'All Threads'
    BUSY_THREAD = 'Busiest Threads'

    def __str__(self):
        return self.value


class HotspotMetricType(AnnotationMixin, Enum):
    # TODO: We implicitly rely on string interning to save memory for us when we use string Enums
    NONE = 'None'
    MODULE_NAME = 'Module'
    SYMBOL_NAME = 'Symbols'
    KNOWN_SYMBOL_NAMES = 'Known Symbols'
    DX12_API = 'DX12 API'
    ETW_EVENTS = 'ETW Events'
    PIX_MARKERS = 'Pix Markers'
    NVTX_MARKERS = 'NVTX Markers'
    NVTX_GPU_MARKERS = 'NVTX Projected GPU Markers'
    MPI_MARKERS = 'MPI Markers'
    CUDA_API = 'CUDA API'
    CUDA_GPU_KERNELS = 'CUDA GPU Kernels'
    DXG_KRNL_PROFILE_RANGES = 'DxgKrnl Profile Ranges'
    CSWITCH_SYMBOLS = 'Context Switch Symbols'
    DX12_GPU_WORKLOAD = 'DX12 GPU Workloads'

    def __str__(self):
        return self.value


@dataclass
class EventMarker:
    id: int = -1
    start: int = -1
    end: int = -1
    correlation_id: int = -1
    name_id: int = -1
    tid: int = -1
    pid: int = -1


@dataclass
class HotspotRecord:
    sample_count: int = 0
    duration: float = 0.0
    start_time: float = 0.0
    value: float = 0.0
    name: str = ""
    extra_1: str = None
    extra_2: str = None


class HotspotCollection:
    start_time_ns: int = 0
    end_time_ns: int = 0
    raw_total_samples: int = None
    raw_total_duration: float = None
    sample_string: str = ""
    duration_string: str = ""
    unit_string: str = ""
    name_string: str = ""
    start_time_string: str = ""
    isDuration: bool = False
    type: HotspotMetricType = HotspotMetricType.NONE
    hotspot_list: list[HotspotRecord] = []
    flamegraph_string: str = None
    extra_1_string: str = None
    extra_2_string: str = None

    def trim(self, min_sample_count=None, min_duration=None, min_value=None):
        for ii, hs in enumerate(self.hotspot_list):
            discard = False
            if min_sample_count and hs.sample_count < min_sample_count:
                discard = True
            if min_duration and hs.duration < min_duration:
                discard = True
            if min_value and hs.value < min_value:
                discard = True
            if discard:
                self.hotspot_list[ii] = None
        self.hotspot_list = [hs for hs in self.hotspot_list if hs is not None]

    def getHotspotSampleCount(self, name):
        for hs in self.hotspot_list:
            if hs.name == name:
                return hs.sample_count
        return None

    def getHotspotDuration(self, name, duration_threshold: float = 0.0):
        for hs in self.hotspot_list:
            if hs.name == name:
                if hs.duration > duration_threshold:
                    return hs.duration
        return None

    def getHotspotMultiSampleCount(self, name_starts_with):
        results = {}
        for hs in self.hotspot_list:
            if hs.name.startswith(name_starts_with):
                results[hs.name] = hs.sample_count
        return results


@dataclass
class Flamegraph:
    callstack_string: str = ""
    name: str = ""
    input_filename: str = ""
    input_filename_ext: str = ".cs.fg"


# This should be sorted in order of importance
# CPU bound should be first
class FrameHealthCategory(Enum):
    CPU_BOUND = 'CPU Bound'
    GPU_BOUND = 'GPU Bound'
    TOOLS_OVERHEAD = 'Tools Overhead'
    COMPILATION = 'Shader Compilations'
    CR = 'Create Resource Calls'
    VRAM_ALLOC = 'VRAM Allocations'
    VRAM_PAGING = 'VRAM Paging'
    VRAM_MAPPING = 'VRAM Map/Unmap'
    BLOCKED_THREAD = 'Graphics Kernel Blocking Threads'
    CSWITCH_THREAD = 'Context Switch'
    SYNC_FROM_GPU = 'Sync with GPU'
    SYNC_FROM_CPU = 'Sync with CPU'
    INTERRUPTS = 'DPC/Interrupts'
    SUBMITTING_WORK = 'Submit Work'
    OS_SCHEDULING = 'OS Scheduling'
    BAR1_READ = 'BAR1 Read Activity'

    def __str__(self):
        return self.value


@dataclass
class FrameHealthItem:
    category: FrameHealthCategory = FrameHealthCategory.CPU_BOUND
    health_string: str = ""
    health_value: float = 0.0
    sub_category: str = None  # Only used for sub categories
    isPriority: bool = False


class FrameHealth:
    cpu_bound: bool = True
    frame_health_list: List[FrameHealthItem] = []

    def get_dict(self):
        d = {}
        # Prioritise durations over sample counts
        d_priority = {}
        d_sample = {}

        self.frame_health_list.sort(key=lambda obj: obj.health_value, reverse=True)

        for frame_health in self.frame_health_list:  # pylint: disable=too-many-nested-blocks
            health_issue_string = str(frame_health.category)

            # Process priority issues first
            if frame_health.isPriority:
                health_string = "<li><b>" + frame_health.health_string + "</b></li>"

                # Check the sub category to see if there are any non priority issues
                if frame_health.sub_category:
                    # Group all of the sub categories into a new list
                    sub_frame_health_list = [fh for fh in self.frame_health_list if fh.sub_category == frame_health.sub_category and fh.isPriority is False]
                    if sub_frame_health_list:
                        # If there is only 1 sub item and it has the same values as the main item, replace the main item with this one.
                        if len(sub_frame_health_list) == 1 and round(sub_frame_health_list[0].health_value, 1) == round(frame_health.health_value, 1):
                            health_string = "<li><b>" + sub_frame_health_list[0].health_string + "</b></li>"
                        else:
                            # add all sub categories to the string as a new sub bullet point
                            health_string += " <ul> "
                            for sub_frame_health in sub_frame_health_list:
                                health_string += " <li>" + sub_frame_health.health_string + "</li>"
                            health_string += " </ul> "

                d_priority[health_issue_string] = d_priority.get(health_issue_string, "") + " " + health_string
            else:
                # Anything with a named function should have been handled above, so skip
                if not frame_health.sub_category:
                    d_sample[health_issue_string] = d_sample.get(health_issue_string, "") + " <li>" + frame_health.health_string + "</li>"

        # Add the issues, based on the order of the enums
        for health_issue in FrameHealthCategory:
            health_issue_string = str(health_issue)
            if health_issue_string in d_priority:
                d[health_issue_string] = d_priority[health_issue_string]
            if health_issue_string in d_sample:
                d[health_issue_string] = d.get(health_issue_string, "") + " " + d_sample[health_issue_string]

        # Add each item as a list
        d_html = {}
        for key, item in d.items():
            d_html[key] = f"<ul>{item}</ul>"

        return d_html

    def get_issue_list(self) -> List[FrameHealthCategory]:
        issue_list = []
        for frame_health in self.frame_health_list:
            if frame_health.category not in issue_list:
                issue_list.append(frame_health.category)
        return issue_list

    def reset(self):
        self.cpu_bound = True
        self.frame_health_list: list[FrameHealthItem] = []

    def ignore_cpu_bound_issues(self, reportRange: HotspotRange) -> bool:
        return reportRange in (HotspotRange.PCIE_BAR1_READS, HotspotRange.PCIE_BAR1_WRITES, HotspotRange.GR_IDLE)


# Remove duplicates and prioritise based on enum ordering
def filter_and_prioritise_categorys(health_summary: List[FrameHealthCategory]) -> List[FrameHealthCategory]:
    health = list(set(health_summary))
    prioritised_health = []
    for health_issue in FrameHealthCategory:
        if health_issue in health:
            prioritised_health.append(health_issue)
    return prioritised_health


class CPUFrameInfo:
    report_range: HotspotRange
    report_threading: HotspotReportThreading
    report_metric: HotspotMetricType
    report_exporter: Optional[ReportExporter] = None

    def __init__(self):
        self.tp = TraceProcessor()
        self.flamegraph_list: List[Flamegraph] = []
        self.process_name = ""
        self.quiet = True
        self.frame_health: FrameHealth = FrameHealth()
        self.app_health_summary: List[FrameHealthCategory] = []
        self.app_health_gpu_bound = True
        self.frame_health_summary = {}
        self.present_tid_list = []
        self.submit_tid_list = []

    def init_nsys_database(self, file_in):
        self.tp.init_loader(TraceLoaderType.NSysRep_Loader).init_database(file_in)
        self.tp.set_current_loader(TraceLoaderType.NSysRep_Loader)
        self.tp.init_common()

    def close_nsys_database(self):
        if self.tp.current_loader == TraceLoaderType.NSysRep_Loader:
            self.tp.get_loader(TraceLoaderType.NSysRep_Loader).close_database()

    @staticmethod
    def __get_kernel_module_list() -> List[str]:
        return ['ntdll.dll', 'KernelBase.dll', 'kernel32.dll', 'ntoskrnl.exe', 'dxgkrnl.sys']

    @staticmethod
    def __get_symbol_blacklist() -> List[str]:
        return [
            # 'CallAndLogImpl', 'EtwWriteTransfer' # Disable for now
        ]

    # These need sorting on priority. Highest priority first
    @staticmethod
    def __get_symbol_whitelist() -> List[str]:
        return [
            'ReadFromSubresource',
            'WriteToSubresource',
            'CDevice::DestroyHeapAndResource',
            'CDevice::CreateCommittedResource',
            'CDevice::CreateHeap',
            'CDevice::CreatePlacedResource',
            'CDevice::CreateGraphicsPipelineState',
            'CDevice::CreateComputePipelineState',
            'CDevice::CreatePipelineState',
            'ResourceBarrier',
            'CopyDescriptors'
            'BuildRaytracingAccelerationStructure',
            'Raytracing',
            'RtlEnterCriticalSection',
            'RtlLeaveCriticalSection',
            'KiInterruptDispatch',
            'KiApcInterrupt',
            'KiPageFault',
            'CDevice::UpdateTileMapping',
            'ExecuteCommandLists',
            'vkCreateGraphicsPipeline',
            'vkCreateComputePipeline',
            'vkQueueSubmit'
        ]

    @staticmethod
    def __get_module_whitelist() -> List[str]:
        return [
            'nvwgf2umx.dll',
            'nvoglv64.dll',
            'cudart.dll',
            'cuda.dll',
            'libcudart.so',
            'libcuda.so',
            'amdxc64.dll',
            'amdxx64.dll',
            'amdvlk64.dll',
            'nvlddmkm.sys',
            'nvrtum64.dll',
            'nvgpucomp64.dll'
            #            'D3D12Core.dll'
        ]

    def generate_hotspot_dfs(self, name, hotspots: HotspotCollection):
        if len(hotspots.hotspot_list) == 0:
            return
        self.tp.add_dataframe_from_dict(name + "_" + hotspots.name_string, self.create_hotspot_dict(hotspots), None, False)

    @staticmethod
    def create_hotspot_dict(hotspots: HotspotCollection, name_string: Optional[str] = "Name"):
        if not hotspots or not hotspots.hotspot_list:
            return None

        hs_dict = {}
        hs_dict[hotspots.unit_string] = [hs.value for hs in hotspots.hotspot_list]
        if hotspots.isDuration:
            hs_dict[hotspots.duration_string] = [hs.duration for hs in hotspots.hotspot_list]
        if hotspots.sample_string:
            hs_dict[hotspots.sample_string] = [hs.sample_count for hs in hotspots.hotspot_list]
        if hotspots.start_time_string:
            hs_dict[hotspots.start_time_string] = [hs.start_time for hs in hotspots.hotspot_list]
        hs_dict[name_string] = [hs.name for hs in hotspots.hotspot_list]
        if hotspots.extra_1_string:
            hs_dict[hotspots.extra_1_string] = [hs.extra_1 for hs in hotspots.hotspot_list]
        if hotspots.extra_2_string:
            hs_dict[hotspots.extra_2_string] = [hs.extra_2 for hs in hotspots.hotspot_list]

        return hs_dict

    @staticmethod
    def is_substring_in_string(string_var: str, substring_list: List[str]):
        """Are any of the substrings in the list in the given string?"""
        return string_var in substring_list or any(substring in string_var for substring in substring_list)

    @staticmethod
    def __create_html_tooltip(text: str, tooltip: str):
        return f'<a title="{tooltip}">{text}</a>'

    ####################################################
    #
    # Get hot symbols from the given list of callchains
    #
    ####################################################
    def __get_hot_symbols(self,
                          callstack_sample_list: List[CallStack],
                          symbol_whitelist: bool,
                          create_tooltip: bool,
                          callstack_type: Optional[tu.CallStackType] = None) -> HotspotCollection:
        sample_count_dict = {}

        white_list = None
        blacklist_list = None
        if symbol_whitelist:
            white_list = self.__get_symbol_whitelist()
        blacklist_list = self.__get_symbol_blacklist()

        total_sample_count = 0
        for cs in callstack_sample_list:
            if callstack_type and callstack_type != cs.stack_type:
                continue

            total_sample_count += 1
            local_symbol_list = []
            for stack_frame in cs.stack:
                module_name = self.tp.get_module_string(stack_frame.module)
                symbol_name = self.tp.get_string(stack_frame.function)

                #  Ignore any blacklisted symbols
                if blacklist_list and self.is_substring_in_string(symbol_name, blacklist_list):
                    continue

                if white_list and not self.is_substring_in_string(symbol_name, white_list):
                    continue

                name = module_name + '@' + symbol_name
                if create_tooltip:
                    name = self.__create_html_tooltip(name, self.__get_compact_callstack_string(cs))

                if name in local_symbol_list:
                    continue

                local_symbol_list.append(name)

                sample_count_dict[name] = sample_count_dict.get(name, 0) + 1

        threshold = 0.5
        if symbol_whitelist:
            threshold = 0.0

        hotspots = HotspotCollection()
        if symbol_whitelist:
            hotspots.type = HotspotMetricType.KNOWN_SYMBOL_NAMES
        else:
            hotspots.type = HotspotMetricType.SYMBOL_NAME

        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = -1
        hotspots.end_time_ns = -1
        hotspots.raw_total_samples = total_sample_count
        hotspots.sample_string = "Callstack Count"
        hotspots.unit_string = "% of Total Sampled Callstacks"
        hotspots.hotspot_list = []

        sorted_symbols = sorted(sample_count_dict, key=lambda k: sample_count_dict[k], reverse=True)

        for name in sorted_symbols:
            sample_percent = 0
            if total_sample_count:
                sample_percent = sample_count_dict[name] / total_sample_count * 100

            if sample_percent < threshold:
                continue

            hs_rec = HotspotRecord()
            hs_rec.sample_count = sample_count_dict[name]
            hs_rec.value = sample_percent
            hs_rec.name = name
            hotspots.hotspot_list.append(hs_rec)

        return hotspots

    ####################################################
    #
    # Generate collapsed callstacks for flamegraph.pl
    #
    ####################################################
    def get_callstacks_for_flamegraph(self,
                                      callstack_sample_list: List[CallStack],
                                      target_pid) -> (str, List[ThreadCallstackCreate]):
        thread_callstack: List[ThreadCallstackCreate] = []
        # all of the callstacks for a frame
        collapsed_cs_frame = ""

        module_black_list = self.__get_kernel_module_list()
        module_white_list = self.__get_module_whitelist()
        white_list = self.__get_symbol_whitelist()
        blacklist_list = self.__get_symbol_blacklist()
        process_name = None
        if target_pid:
            process_name = self.tp.get_process_name(target_pid).split("\\")[-1].split("/")[-1]

        # Need the format to be symbol0; symbol1; symbol2 sampleCount

        index = 0
        for cs in callstack_sample_list:
            collapsed_cs = ""
            for stack_frame in cs.stack:
                module_name = self.tp.get_module_string(stack_frame.module)
                symbol_name = self.tp.get_string(stack_frame.function)

                #  Ignore any blacklisted symbols
                if blacklist_list and self.is_substring_in_string(symbol_name, blacklist_list):
                    continue

                if collapsed_cs:
                    collapsed_cs += ";"
                collapsed_cs += module_name + "@" + symbol_name

                # These annotations are used in the viewer for colour coding
                if self.is_substring_in_string(symbol_name, white_list):
                    collapsed_cs += "_[i]"
                elif self.is_substring_in_string(module_name, module_black_list):
                    collapsed_cs += "_[k]"
                elif self.is_substring_in_string(module_name, module_white_list):
                    collapsed_cs += "_[d]"
                elif process_name in module_name:
                    collapsed_cs += "_[a]"

            call_count = 1
            callstack_type = "Event Callstacks"
            if cs.stack_type == tu.CallStackType.SAMPLED:
                callstack_type = "Sampled Callstacks"
            elif cs.stack_type == tu.CallStackType.CSWITCH:
                callstack_type = "CSwitch Callstacks"
            collapsed_cs = f"{callstack_type};{collapsed_cs} {call_count}"

            collapsed_cs_frame += collapsed_cs + "\n"
            thread_callstack.append(ThreadCallstackCreate(
                callstack_index=index,
                count=call_count,
                type=callstack_type,
                thread_trace=collapsed_cs,
            ))
            index = index + 1

        if collapsed_cs_frame == "":
            return None, []
        return collapsed_cs_frame, thread_callstack

    ####################################################
    #
    # Get hot modules from the given list of callchains
    #
    ####################################################
    def __get_hot_modules(self,
                          callstack_sample_list: List[CallStack],
                          callstack_type: tu.CallStackType) -> HotspotCollection:
        sample_count_dict = {}

        # Walk the callstacks and for each stack, walk all frames looking for unique modules and count them
        for cs in callstack_sample_list:
            local_module_list = []
            if callstack_type != cs.stack_type:
                continue
            for stack_frame in cs.stack:
                module = stack_frame.module
                if module in local_module_list:
                    continue
                local_module_list.append(module)

                # Inc sample
                sample_count_dict[module] = sample_count_dict.get(module, 0) + 1

        hotspots = HotspotCollection()
        hotspots.type = HotspotMetricType.MODULE_NAME

        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = -1
        hotspots.end_time_ns = -1
        hotspots.raw_total_samples = len(callstack_sample_list)
        hotspots.sample_string = "Sample Count"
        hotspots.unit_string = "%"
        hotspots.hotspot_list = []

        sorted_modules = sorted(sample_count_dict, key=lambda k: sample_count_dict[k], reverse=True)
        total_sample_count = len(callstack_sample_list)

        for module in sorted_modules:
            hs_rec = HotspotRecord()
            hs_rec.sample_count = sample_count_dict[module]
            hs_rec.value = hs_rec.sample_count / total_sample_count * 100
            hs_rec.name = self.tp.get_module_string(module)
            hotspots.hotspot_list.append(hs_rec)

        return hotspots

    @staticmethod
    def __apply_nesting(input_string, nest_count):
        return "&#8594; " * nest_count + input_string

    ####################################################
    #
    # Get markers from a table for the given time period and tid/pid
    #
    ####################################################
    def __get_combined_hotspot_events(self,
                                      start_time_ns: int,
                                      end_time_ns: int,
                                      target_pid: int,
                                      target_tid: int,
                                      metric_type: HotspotMetricType) -> (HotspotCollection, HotspotCollection):
        loader_event_type = None
        if metric_type is HotspotMetricType.DX12_API:
            loader_event_type = TraceLoaderEvents.DX12_API_CALLS
        elif metric_type is HotspotMetricType.CUDA_API:
            loader_event_type = TraceLoaderEvents.CUDA_API_CALLS
        elif metric_type is HotspotMetricType.CUDA_GPU_KERNELS:
            loader_event_type = TraceLoaderEvents.CUDA_GPU_KERNELS
        elif metric_type is HotspotMetricType.NVTX_MARKERS:
            loader_event_type = TraceLoaderEvents.NVTX_MARKERS
        elif metric_type is HotspotMetricType.MPI_MARKERS:
            loader_event_type = TraceLoaderEvents.MPI_MARKERS
        elif metric_type is HotspotMetricType.NVTX_GPU_MARKERS:
            loader_event_type = TraceLoaderEvents.NVTX_GPU_MARKERS
        elif metric_type is HotspotMetricType.DX12_GPU_WORKLOAD:
            loader_event_type = TraceLoaderEvents.DX12_GPU_WORKLOAD

        tooltip_dict = {}
        if loader_event_type:
            api_duration_list = self.tp.get_ordered_events(event_type=loader_event_type, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
            if api_duration_list:
                api_duration_dict, api_count_dict, tooltip_dict = self.__get_hotspots_from_ordered_list(api_duration_list)
            else:
                api_duration_dict, api_count_dict = self.tp.get_events(event_type=loader_event_type, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        else:
            logger.warning(f"Hotspot metric type {metric_type} not supported in this function.")
            return None, None

        if not api_duration_dict or not api_count_dict:
            return None, None

        frame_duration = end_time_ns - start_time_ns

        hotspots = HotspotCollection()
        hotspots.type = metric_type
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = frame_duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.sample_string = "Count"
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        for api_call in sorted(api_duration_dict, key=api_duration_dict.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.duration = api_duration_dict[api_call] / 1000000.0
            hs_rec.sample_count = api_count_dict[api_call]
            hs_rec.value = api_duration_dict[api_call] / frame_duration * 100.0
            hs_rec.name = api_call
            if api_call in tooltip_dict:
                hs_rec.name = self.__create_html_tooltip(hs_rec.name, tooltip_dict[api_call])

            hotspots.hotspot_list.append(hs_rec)

        # Ordered hotspots
        ordered_hotspots = None
        if api_duration_list:
            ordered_hotspots = HotspotCollection()
            ordered_hotspots.type = metric_type
            ordered_hotspots.name_string = "Ordered " + str(ordered_hotspots.type)
            ordered_hotspots.start_time_ns = start_time_ns
            ordered_hotspots.end_time_ns = end_time_ns
            ordered_hotspots.raw_total_duration = frame_duration
            ordered_hotspots.duration_string = "Duration (ms)"
            ordered_hotspots.sample_string = ""
            ordered_hotspots.start_time_string = "Start Time (ms)"
            ordered_hotspots.unit_string = "% of Frame Duration"
            ordered_hotspots.isDuration = True
            ordered_hotspots.hotspot_list = []

            if hotspots.type == HotspotMetricType.NVTX_GPU_MARKERS:
                ordered_hotspots.extra_1_string = "Kernel Name"

            nested_list = []

            for api_call, start_time, duration, tooltip in api_duration_list:
                hs_rec = HotspotRecord()
                hs_rec.duration = duration / 1000000.0
                hs_rec.start_time = start_time / 1000000.0
                hs_rec.value = duration / frame_duration * 100.0

                # Pop the stale nested regions
                while nested_list and nested_list[-1] <= start_time:
                    nested_list.pop()

                nested_length = len(nested_list)
                # Add an arrow for nested indentation
                hs_rec.name = self.__apply_nesting(api_call, nested_length)
                if tooltip:
                    hs_rec.name = self.__create_html_tooltip(hs_rec.name, tooltip)
                    if hotspots.type == HotspotMetricType.NVTX_GPU_MARKERS:
                        hs_rec.extra_1 = tooltip

                ordered_hotspots.hotspot_list.append(hs_rec)

                if duration:
                    nested_list.append(start_time + duration)

        return hotspots, ordered_hotspots

    ####################################################
    #
    # Get markers from a table for the given time period and tid/pid
    #
    ####################################################
    def __get_hotspot_events(self,
                             start_time_ns: int,
                             end_time_ns: int,
                             target_pid: int,
                             target_tid: int,
                             metric_type: HotspotMetricType) -> (HotspotCollection, HotspotCollection):
        loader_event_type = None
        if metric_type is HotspotMetricType.DX12_API:
            loader_event_type = TraceLoaderEvents.DX12_API_CALLS
        elif metric_type is HotspotMetricType.CUDA_API:
            loader_event_type = TraceLoaderEvents.CUDA_API_CALLS
        elif metric_type is HotspotMetricType.CUDA_GPU_KERNELS:
            loader_event_type = TraceLoaderEvents.CUDA_GPU_KERNELS
        elif metric_type is HotspotMetricType.NVTX_MARKERS:
            loader_event_type = TraceLoaderEvents.NVTX_MARKERS
        elif metric_type is HotspotMetricType.MPI_MARKERS:
            loader_event_type = TraceLoaderEvents.MPI_MARKERS

        if loader_event_type:
            api_duration_dict, api_count_dict = self.tp.get_events(event_type=loader_event_type, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        else:
            logger.warning(f"Hotspot metric type {metric_type} not supported in this function.")
            return None

        if not api_duration_dict or not api_count_dict:
            return None

        frame_duration = end_time_ns - start_time_ns

        hotspots = HotspotCollection()
        hotspots.type = metric_type
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = frame_duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.sample_string = "Count"
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        for api_call in sorted(api_duration_dict, key=api_duration_dict.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.duration = api_duration_dict[api_call] / 1000000.0
            hs_rec.sample_count = api_count_dict[api_call]
            hs_rec.value = api_duration_dict[api_call] / frame_duration * 100.0
            hs_rec.name = api_call
            hotspots.hotspot_list.append(hs_rec)

        return hotspots

    ####################################################
    #
    # Get ordered events from a table for the given time period and tid/pid
    #
    ####################################################
    def __get_ordered_events(self,
                             start_time_ns: int,
                             end_time_ns: int,
                             target_pid: int,
                             target_tid: int,
                             metric_type: HotspotMetricType) -> Optional[HotspotCollection]:
        loader_event_type = None
        if metric_type is HotspotMetricType.NVTX_MARKERS:
            loader_event_type = TraceLoaderEvents.NVTX_MARKERS
        elif metric_type is HotspotMetricType.NVTX_GPU_MARKERS:
            loader_event_type = TraceLoaderEvents.NVTX_GPU_MARKERS
        elif metric_type is HotspotMetricType.PIX_MARKERS:
            loader_event_type = TraceLoaderEvents.PIX_MARKERS
        elif metric_type is HotspotMetricType.DX12_GPU_WORKLOAD:
            loader_event_type = TraceLoaderEvents.DX12_GPU_WORKLOAD
        elif metric_type is HotspotMetricType.CUDA_GPU_KERNELS:
            loader_event_type = TraceLoaderEvents.CUDA_GPU_KERNELS
        if loader_event_type:
            api_duration_list = self.tp.get_ordered_events(event_type=loader_event_type, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        else:
            logger.warning(f"Hotspot metric type {metric_type} not supported in this function.")
            return None
        if not api_duration_list:
            return None

        frame_duration = end_time_ns - start_time_ns

        hotspots = HotspotCollection()
        hotspots.type = metric_type
        hotspots.name_string = "Ordered " + str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = frame_duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.start_time_string = "Start Time (ms)"
        hotspots.sample_string = ""
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        nested_list = []

        for api_call, start_time, duration, tooltip in api_duration_list:
            hs_rec = HotspotRecord()
            hs_rec.duration = duration / 1000000.0
            hs_rec.start_time = start_time / 1000000.0
            hs_rec.value = duration / frame_duration * 100.0

            # Pop the stale nested regions
            while nested_list and nested_list[-1] <= start_time:
                nested_list.pop()

            nested_length = len(nested_list)
            # Add an arrow for nested indentation
            hs_rec.name = "&#8594; " * nested_length + api_call
            if tooltip:
                hs_rec.name = self.__create_html_tooltip(hs_rec.name, tooltip)
            hotspots.hotspot_list.append(hs_rec)

            if duration:
                nested_list.append(start_time + duration)

        return hotspots

    ####################################################
    #
    # Summarise the durations and counts from an ordered list
    #
    ####################################################
    def __get_hotspots_from_ordered_list(self, ordered_duration_list: List):
        duration_dict = {}
        count_dict = {}
        tooltip_dict = {}

        for api_call, _, api_duration, tooltip in ordered_duration_list:
            duration = duration_dict.get(api_call, 0)
            duration_dict[api_call] = duration + api_duration
            count = count_dict.get(api_call, 0)
            count_dict[api_call] = count + 1
            if api_call in tooltip_dict:
                if tooltip != tooltip_dict[api_call]:
                    tooltip_dict[api_call] = "Non matching kernels"
            else:
                tooltip_dict[api_call] = tooltip

        return duration_dict, count_dict, tooltip_dict

    ####################################################
    #
    # Get hot DX12 API calls for the given time period and tid/pid
    #
    ####################################################
    def __get_hot_dx12_api_calls(self,
                                 target_pid: int,
                                 target_tid: int,
                                 start_time_ns: int,
                                 end_time_ns: int) -> HotspotCollection:
        return self.__get_hotspot_events(start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid, metric_type=HotspotMetricType.DX12_API)

    ####################################################
    #
    # Get hot CUDA API calls for the given time period and tid/pid
    #
    ####################################################
    def __get_hot_cuda_api_calls(self,
                                 target_pid: int,
                                 target_tid: int,
                                 start_time_ns: int,
                                 end_time_ns: int) -> HotspotCollection:
        return self.__get_hotspot_events(start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid, metric_type=HotspotMetricType.CUDA_API)

    ####################################################
    #
    # Get hot ETW events for the given time period and tid/pid
    #
    ####################################################
    def __get_hot_etw_events(self,
                             target_pid: int,
                             target_tid: int,
                             start_time_ns: int,
                             end_time_ns: int) -> HotspotCollection:
        etw_sample_dict, etw_task_names = self.tp.get_events(event_type=TraceLoaderEvents.ETW_EVENTS, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        if not etw_sample_dict:
            return None

        total_sample_count = sum(etw_sample_dict.values())

        hotspots = HotspotCollection()
        hotspots.type = HotspotMetricType.ETW_EVENTS
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_samples = total_sample_count
        hotspots.sample_string = "Event Count"
        hotspots.unit_string = "%"
        hotspots.hotspot_list = []

        for etw_event in sorted(etw_sample_dict, key=etw_sample_dict.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.sample_count = etw_sample_dict[etw_event]
            hs_rec.value = etw_sample_dict[etw_event] / total_sample_count * 100
            hs_rec.name = etw_task_names.get(etw_event, etw_event)  # If the name is not in the dict, just use the name
            hotspots.hotspot_list.append(hs_rec)
            if target_tid and hs_rec.name == 'Present':
                if target_tid not in self.present_tid_list:
                    self.present_tid_list.append(target_tid)

        return hotspots

    ####################################################
    #
    # Get hot dxg kernel ranges from ETW events for the given time period and tid/pid
    #
    ####################################################
    def __get_hot_dxgkrnl_profile_ranges(self,
                                         target_pid: int,
                                         target_tid: int,
                                         start_time_ns: int,
                                         end_time_ns: int) -> HotspotCollection:
        region_durations_dict, region_counts_dict = self.tp.get_events(event_type=TraceLoaderEvents.DXGKRNL_PROFILE_RANGE, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        if not region_durations_dict or not region_counts_dict:
            return None

        frame_duration = end_time_ns - start_time_ns

        hotspots = HotspotCollection()
        hotspots.type = HotspotMetricType.DXG_KRNL_PROFILE_RANGES
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = frame_duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.sample_string = "Range Count"
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        for region in sorted(region_durations_dict, key=region_durations_dict.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.duration = region_durations_dict[region] / 1000000.0
            hs_rec.sample_count = region_counts_dict[region]
            hs_rec.value = region_durations_dict[region] / frame_duration * 100.0
            hs_rec.name = region
            hotspots.hotspot_list.append(hs_rec)

            if target_tid:
                if hs_rec.name == 'DxgkPresent':
                    if target_tid not in self.present_tid_list:
                        self.present_tid_list.append(target_tid)
                if hs_rec.name == 'DxgkSubmitCommandToHwQueue':
                    if target_tid not in self.submit_tid_list:
                        self.submit_tid_list.append(target_tid)

        return hotspots

    ####################################################
    #
    # Get hot DX12 PIX Markers given time period and tid/pid
    #
    ####################################################
    def __get_hot_pix_markers(self,
                              target_pid: int,
                              target_tid: int,
                              start_time_ns: int,
                              end_time_ns: int) -> HotspotCollection:
        pix_marker_durations, pix_marker_counts = self.tp.get_events(event_type=TraceLoaderEvents.PIX_MARKERS, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid)
        if not pix_marker_durations or not pix_marker_counts:
            return None

        duration = end_time_ns - start_time_ns

        hotspots = HotspotCollection()
        hotspots.type = HotspotMetricType.PIX_MARKERS
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.sample_string = "Count"
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        for pm in sorted(pix_marker_durations, key=pix_marker_durations.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.duration = pix_marker_durations[pm] / 1000000.0
            hs_rec.sample_count = pix_marker_counts[pm]
            hs_rec.value = pix_marker_durations[pm] / duration * 100
            hs_rec.name = pm
            hotspots.hotspot_list.append(hs_rec)

        return hotspots

    ####################################################
    #
    # Get hot MPI markers for the given time period and tid/pid
    #
    ####################################################
    def __get_hot_mpi_markers(self,
                              target_pid: int,
                              target_tid: int,
                              start_time_ns: int,
                              end_time_ns: int) -> HotspotCollection:
        return self.__get_hotspot_events(start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid, target_tid=target_tid, metric_type=HotspotMetricType.MPI_MARKERS)

    ####################################################
    #
    # Get the full callstack in a compact form
    #
    ####################################################
    def __get_compact_callstack_string(self, callstack: CallStack) -> str:
        callstack_string = ""
        dup_list = []
        for stack_frame in callstack.stack:
            function = self.tp.get_string(stack_frame.function)
            module = self.tp.get_module_string(stack_frame.module)
            if self.is_substring_in_string(function, ['0x7f', '0xff']):
                function = "Unknown"
            if (function, module) in dup_list:
                dup_list.append((function, module))
                continue
            if dup_list:
                func, mod = dup_list[0]
                callstack_string += f"{mod}@{func}"
                if len(dup_list) > 1:
                    callstack_string += f"[{len(dup_list)}x]"
                callstack_string += "->"

            dup_list = []
            dup_list.append((function, module))

        if dup_list:
            func, mod = dup_list[0]
            callstack_string += f"{mod}@{func}"
            if len(dup_list) > 1:
                callstack_string += f"[{len(dup_list)}x]"

        return callstack_string

    ####################################################
    #
    # Try to find the sensible cswitch symbol
    #
    ####################################################
    def __find_best_context_switch_symbol(self, callstack: CallStack):
        last_symbol = None
        # Lets find the last usermode symbol
        for stack_frame in reversed(callstack.stack):
            # ignore kernel symbols - find the first non kernel symbol
            if self.is_substring_in_string(self.tp.get_module_string(stack_frame.module), self.__get_kernel_module_list()):
                last_symbol = stack_frame.function
                continue

            symbol = stack_frame.function
            if self.is_substring_in_string(self.tp.get_string(stack_frame.function), ['0x7f', '0xff', 'Unknown', 'PDB not found']):
                symbol = last_symbol
            return symbol
        return None

    ####################################################
    #
    # Get cswitch durations
    #
    ####################################################
    def __get_hot_context_switch_symbols(self,
                                         target_pid: int,
                                         target_tid: int,
                                         start_time_ns: int,
                                         end_time_ns: int,
                                         timeslice_list: List[TimeSlice],
                                         callstack_list: List[CallStack]
                                         ) -> Optional[HotspotCollection]:
        frame_duration = end_time_ns - start_time_ns
        cswitch_symbol_durations = {}
        cswitch_symbol_counts = {}

        ts_list = self.tp.filter_timeslices(timeslice_list, target_pid=target_pid, target_tid=target_tid)

        # Walking the callstacks can be super slow, so we trim first
        trimmed_cs = [cs for cs in callstack_list if cs.time >= start_time_ns and cs.time <= end_time_ns]
        if target_pid:
            trimmed_cs = [cs for cs in trimmed_cs if cs.pid == target_pid]
        if self.tp.is_supported(TraceLoaderSupport.CSWITCH_CALLSTACK_SEPARATE):
            trimmed_cs = [cs for cs in trimmed_cs if cs.stack_type == tu.CallStackType.CSWITCH]

        # Walk the timeslices
        if self.tp.is_supported(TraceLoaderSupport.CSWITCH_CALLSTACK_FRONT):
            found_tid = []

            tid_prev_end = {}
            for ts in ts_list:
                tid = tu.get_tid(ts.gtid)

                # Capture any threads that start mid-frame
                # This is essentially the duration we are blocked
                if tid not in found_tid:
                    found_tid.append(tid)
                    if ts.start > start_time_ns:
                        tid_prev_end[tid] = start_time_ns

                # If we already have a timeslice stored, then lets calc the duration of its callstack,
                # otherwise store its end and continue
                if tid not in tid_prev_end:
                    tid_prev_end[tid] = ts.end
                    continue

                # symbol is the blocking symbol
                prev_end = tid_prev_end[tid]
                duration = ts.start - prev_end
                tid_prev_end[tid] = ts.end

                # find the callstack at the START of this timeslice - should only be 1
                cswitch_cs = [cs for cs in trimmed_cs if cs.tid == tid and cs.time == ts.start]

                cswitch_symbol = None
                if cswitch_cs:
                    cswitch_symbol = self.__find_best_context_switch_symbol(cswitch_cs[0])
                    # Add a tooltip of the full callstack
                    cswitch_symbol = self.__create_html_tooltip(self.tp.get_string(cswitch_symbol), self.__get_compact_callstack_string(cswitch_cs[0]))

                cswitch_symbol_durations[cswitch_symbol] = cswitch_symbol_durations.get(cswitch_symbol, 0) + duration
                cswitch_symbol_counts[cswitch_symbol] = cswitch_symbol_counts.get(cswitch_symbol, 0) + 1

        elif self.tp.is_supported(TraceLoaderSupport.CSWITCH_CALLSTACK_BACK):
            # Walk the timeslices
            cswitch_state_tid_dict = {}
            found_tid = []

            for ts in ts_list:
                tid = tu.get_tid(ts.gtid)

                # Capture any threads that start mid-frame
                if tid not in found_tid:
                    found_tid.append(tid)
                    if ts.start > start_time_ns:
                        cswitch_state_tid_dict[tid] = (start_time_ns, None)

                # If we already have a timeslice stored, then lets calc the duration of its callstack
                if tid in cswitch_state_tid_dict:
                    prev_end, cswitch_symbol = cswitch_state_tid_dict[tid]
                    duration = ts.start - prev_end

                    cswitch_symbol_durations[cswitch_symbol] = cswitch_symbol_durations.get(cswitch_symbol, 0) + duration
                    cswitch_symbol_counts[cswitch_symbol] = cswitch_symbol_counts.get(cswitch_symbol, 0) + 1
                    cswitch_state_tid_dict.pop(tid)

                # Now lets deal with this timeslice
                cswitch_cs = [cs for cs in trimmed_cs if cs.tid == tid and cs.time == ts.end]

                cswitch_symbol = None
                if cswitch_cs:
                    cswitch_symbol = self.__find_best_context_switch_symbol(cswitch_cs[0])
                    # Add a tooltip of the full callstack
                    cswitch_symbol = f'<a title="{self.__get_compact_callstack_string(cswitch_cs[0])}">{self.tp.get_string(cswitch_symbol)}</a>'
                cswitch_state_tid_dict[tid] = (ts.end, cswitch_symbol)

            for tid, (prev_end, symbol) in cswitch_state_tid_dict.items():
                if prev_end > end_time_ns:
                    duration = end_time_ns - prev_end
                    cswitch_symbol_durations[symbol] = cswitch_symbol_durations.get(symbol, 0) + duration
                    cswitch_symbol_counts[symbol] = cswitch_symbol_counts.get(symbol, 0) + 1
        else:
            logger.warning("Neither CSWITCH_CALLSTACK_BACK nor CSWITCH_CALLSTACK_FRONT support for cswitch hotspots")
            return None

        hotspots = HotspotCollection()
        hotspots.type = HotspotMetricType.CSWITCH_SYMBOLS
        hotspots.name_string = str(hotspots.type)
        hotspots.start_time_ns = start_time_ns
        hotspots.end_time_ns = end_time_ns
        hotspots.raw_total_duration = frame_duration
        hotspots.duration_string = "Duration (ms)"
        hotspots.sample_string = "Callstack Count"
        hotspots.unit_string = "% of Frame Duration"
        hotspots.isDuration = True
        hotspots.hotspot_list = []

        for cswitch_symbol in sorted(cswitch_symbol_durations, key=cswitch_symbol_durations.get, reverse=True):
            hs_rec = HotspotRecord()
            hs_rec.duration = cswitch_symbol_durations[cswitch_symbol] / 1000000.0
            hs_rec.sample_count = cswitch_symbol_counts[cswitch_symbol]
            hs_rec.value = cswitch_symbol_durations[cswitch_symbol] / frame_duration * 100
            hs_rec.name = cswitch_symbol
            hotspots.hotspot_list.append(hs_rec)

        return hotspots

    ####################################################
    #
    # Get a dataframe name based on the range
    #
    ####################################################
    def __get_per_range_df_name(self, hotspot_range: HotspotRange, data_name: str) -> str:
        region_overview_name = f'{hotspot_range}_{data_name}'
        region_overview_name = self.tp.get_dataframe_key(region_overview_name)
        return region_overview_name

    def get_per_range_df_name(self, hotspot_range: HotspotRange, data_name: str):
        return self.__get_per_range_df_name(hotspot_range, data_name)

    ####################################################
    #
    # Create the thread utilisation dict based on real thread names
    #
    ####################################################
    def __create_named_thread_utilisation_dict(self, thread_utilisation_dict, sorted_thread_keys, target_pid):
        named_thread_util_dict = {}
        # Walk the sorted threads inserting the info
        for key in sorted_thread_keys:
            pid, tid = tu.convert_global_tid(key)
            if pid != target_pid:
                continue

            util = thread_utilisation_dict[key]

            name = self.tp.get_thread_name(key)
            named_thread_util_dict[name] = util

        return named_thread_util_dict

    ####################################################
    #
    # Filter the timeslices based on time and pid.
    # If input times are None, calculates good times based on the timeslices
    # If pid is None, finds the busiest pid and assumes that
    #
    ####################################################
    def get_filtered_timeslices(self,
                                start_time_ns: Optional[float] = None,
                                end_time_ns: Optional[float] = None,
                                target_pid: Optional[int] = None,
                                quiet: Optional[bool] = False):

        all_timeslice_list = self.tp.get_all_timeslices(quiet=self.quiet)

        if start_time_ns is None and end_time_ns is None:
            start_time_ns, end_time_ns = self.tp.get_safe_timerange_from_timeslices(all_timeslice_list)

        timeslice_list = self.tp.filter_timeslices(all_timeslice_list, start_time_ns, end_time_ns, target_pid)

        if target_pid is None:
            # Process the timeslices to generate thread utilisation.
            # This is only needed to determine the main PID
            # Could be optimised
            total_time, thread_utilisation_dict = self.tp.get_thread_utilisation(timeslice_list, True)

            target_pid, process_util_dict = self.tp.find_target_pid(thread_utilisation_dict, start_time_ns, end_time_ns, quiet)

            timeslice_list = self.tp.filter_timeslices(timeslice_list, start_time_ns, end_time_ns, target_pid)

        return timeslice_list, start_time_ns, end_time_ns, target_pid

    ####################################################
    #
    # Helper for common dxgkrnl profile health issues
    #
    ####################################################
    def __report_dxgkrnl_profile_health_issues(self, hs_app, target_tid_name, duration_threshold, is_priority):
        if target_tid_name:
            thread_string = f"thread '{target_tid_name}'"
            thread_string2 = ""
        else:
            thread_string = "application threads"
            thread_string2 = "a total of "

        dxgkUpdateGpuVirtualAddress = hs_app.getHotspotDuration('DxgkUpdateGpuVirtualAddress', duration_threshold)
        ddiNotifyDpc = hs_app.getHotspotDuration('DdiNotifyDpc', duration_threshold)
        dpiFdoMessageInterruptRoutine = hs_app.getHotspotDuration('DpiFdoMessageInterruptRoutine', duration_threshold)
        dxgkSubmitCommandToHwQueue = hs_app.getHotspotDuration('DxgkSubmitCommandToHwQueue', duration_threshold)
        dxgkCreateAllocation = hs_app.getHotspotDuration('DxgkCreateAllocation', duration_threshold)
        dxgkDestroyAllocation = hs_app.getHotspotDuration('DxgkDestroyAllocation2', duration_threshold)
        #                    dxgkSubmitWaitForSyncObjectsToHwQueue = hs_app.getHotspotDuration('DxgkSubmitWaitForSyncObjectsToHwQueue', duration_threshold)
        dxgkWaitForSynchronizationObject = hs_app.getHotspotDuration('DxgkWaitForSynchronizationObject', duration_threshold)

        if dxgkUpdateGpuVirtualAddress:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC,
                                                                       health_string=f"[{dxgkUpdateGpuVirtualAddress:.1f} ms] : Graphics Kernel is blocking {thread_string} whilst updating GPU virtual addresses for {thread_string2}{dxgkUpdateGpuVirtualAddress:.1f} ms.",
                                                                       health_value=dxgkUpdateGpuVirtualAddress,
                                                                       sub_category="dxgkUpdateGpuVirtualAddress",
                                                                       isPriority=is_priority))
        if ddiNotifyDpc:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.INTERRUPTS,
                                                                       health_string=f"[{ddiNotifyDpc:.1f} ms] : Graphics Kernel is blocking {thread_string} by handling DPC/Interrupts with DdiNotifyDpc for {thread_string2}{ddiNotifyDpc:.1f} ms.",
                                                                       health_value=ddiNotifyDpc,
                                                                       sub_category="ddiNotifyDpc",
                                                                       isPriority=is_priority))
        if dpiFdoMessageInterruptRoutine:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.INTERRUPTS,
                                                                       health_string=f"[{dpiFdoMessageInterruptRoutine:.1f} ms] : Graphics Kernel is blocking {thread_string} by handling DPC/Interrupts with DpiFdoMessageInterruptRoutine for {thread_string2}{dpiFdoMessageInterruptRoutine:.1f} ms.",
                                                                       health_value=dpiFdoMessageInterruptRoutine,
                                                                       sub_category="dpiFdoMessageInterruptRoutine",
                                                                       isPriority=is_priority))
        if dxgkSubmitCommandToHwQueue:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SUBMITTING_WORK,
                                                                       health_string=f"[{dxgkSubmitCommandToHwQueue:.1f} ms] : Graphics Kernel is blocking {thread_string} whilst submitting commands to the GPU for {thread_string2}{dxgkSubmitCommandToHwQueue:.1f} ms.",
                                                                       health_value=dxgkSubmitCommandToHwQueue,
                                                                       sub_category="dxgkSubmitCommandToHwQueue",
                                                                       isPriority=is_priority))
        if dxgkCreateAllocation:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC,
                                                                       health_string=f"[{dxgkCreateAllocation:.1f} ms] : Graphics Kernel is blocking {thread_string} whilst allocating VRAM for {thread_string2}{dxgkCreateAllocation:.1f} ms.",
                                                                       health_value=dxgkCreateAllocation,
                                                                       sub_category="dxgkCreateAllocation",
                                                                       isPriority=is_priority))
        if dxgkDestroyAllocation:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC,
                                                                       health_string=f"[{dxgkDestroyAllocation:.1f} ms] : Graphics Kernel is blocking {thread_string} whilst deallocating VRAM for {thread_string2}{dxgkDestroyAllocation:.1f} ms.",
                                                                       health_value=dxgkDestroyAllocation,
                                                                       sub_category="dxgkDestroyAllocation",
                                                                       isPriority=is_priority))
        if dxgkWaitForSynchronizationObject:
            self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SYNC_FROM_GPU,
                                                                       health_string=f"[{dxgkWaitForSynchronizationObject:.1f} ms] : Graphics Kernel is blocking {thread_string} waiting for sync objects for {thread_string2}{dxgkWaitForSynchronizationObject:.1f} ms.",
                                                                       health_value=dxgkWaitForSynchronizationObject,
                                                                       sub_category="dxgkWaitForSynchronizationObject",
                                                                       isPriority=is_priority))


    @dataclass
    class ProcessHotspotFrameParams:
        filtered_timeslice_list: List[TimeSlice]
        frame_string: str
        short_frame_string: str
        frame_index: int
        frame_index_global: Optional[int]
        frame_duration: FrameDurations
        main_pid: int
        report_thread_count: int
        report_range: HotspotRange
        report_threading: HotspotReportThreading
        report_metric: HotspotMetricType
        cpu_config: tu.CPUConfig = None
        extra_frame_info_dict: Optional[dict] = None
        database_hotspot_analysis_ids: Optional[DatabaseIds] = None

    ####################################################
    #
    # Main logic for the stutter analysis
    #
    ####################################################
    def process_hotspot_frame(self, params: ProcessHotspotFrameParams):

        # Reset the frame health
        self.frame_health.reset()
        median = True
        self.present_tid_list = []
        self.submit_tid_list = []

        df_name = params.short_frame_string
        if "Median" not in params.short_frame_string:
            df_name = params.short_frame_string + "_" + str(params.frame_index)
            median = False

        # Add the threading to the key
        df_name = f'{df_name}_{params.report_threading}'
        df_name = df_name.replace(" ", "_")

        progress.StartSubstep(str(params.frame_index), params.frame_string)
        logger.info(f'Hotspot Analysis for {params.frame_string} : {params.frame_index} : [{params.frame_duration.duration / 1000000:2.2f}ms @ {params.frame_duration.start / 1000000000:.4f}s]')
        logger.info(df_name)
        logger.info("=" * 50)

        frame_info_dict = {}
        frame_thread_info_dict = {
            "Name": [],
            "TID": [],
            "Utilisation (%)": [],
        }
        frame_info_dict["<b>Duration (ms)</b>"] = params.frame_duration.duration / 1000000
        frame_info_dict["Start Time (s)"] = params.frame_duration.start / 1000000000

        if params.frame_duration.total_value > 0:
            frame_info_dict["Summed Metric Value"] = params.frame_duration.total_value

        if params.extra_frame_info_dict:
            frame_info_dict.update(params.extra_frame_info_dict)

        ###############################################
        #
        # Filter the timeslices for the given frames
        #
        ###############################################
        frame_timeslice_list = self.tp.filter_timeslices(params.filtered_timeslice_list, params.frame_duration.start, params.frame_duration.start + params.frame_duration.duration, target_pid=params.main_pid)

        # Find the busiest threads
        total_time, thread_utilisation_dict, cpu_thread_utilisation_dict = self.tp.get_thread_utilisation_per_core(frame_timeslice_list, True)
        if total_time <= 0:
            if params.report_range in (HotspotRange.NONE, HotspotRange.CUDNN_GPU_KERNELS):
                # Add a fake thread for GPU ranges
                thread_utilisation_dict[0] = 0
            else:
                logger.warning("NO CPU activity found for the given metrics")
                return

        sorted_thread_keys = sorted(thread_utilisation_dict, key=thread_utilisation_dict.get, reverse=True)

        # Add a thread utilisation df for the html graph
        named_thread_time_dict = self.__create_named_thread_utilisation_dict(thread_utilisation_dict, sorted_thread_keys, params.main_pid)
        frame_duration_ms = params.frame_duration.duration / 1000000
        named_thread_time_dict.update({k: frame_duration_ms * i for k, i in named_thread_time_dict.items()})

        self.tp.add_dataframe_from_dict(
            df_name + "_thread_time",
            {
                "Threads": named_thread_time_dict.keys(),
                "Time (ms)": named_thread_time_dict.values()
            },
            None,
            False
        )

        # If the cpu config isn't passed in, try to get it from the data file
        if not params.cpu_config:
            params.cpu_config = self.tp.get_cpu_config()

        cpu_named_thread_dict = None
        if params.cpu_config:
            # Work out P/E cores if applicable
            p_core_list = range(params.cpu_config.p_core_starting_index, params.cpu_config.p_core_starting_index + params.cpu_config.logical_p_core_count)
            cpu_named_thread_dict = {}

            # WAR - the viewer doesn't seem to cope with work ONLY on an E-Core, so pre-populate the dict.
            for key_tid in sorted_thread_keys:
                tid_name = self.tp.get_thread_name(key_tid)
                cpu_named_thread_dict[(tid_name, "P Core")] = 0.0
                cpu_named_thread_dict[(tid_name, "E Core")] = 0.0

            # split into P/E cores
            for (thread, cpu), thread_value in cpu_thread_utilisation_dict.items():
                tid_name = self.tp.get_thread_name(thread)
                cpu_type = "P Core" if cpu in p_core_list else "E Core"
                p_key = (tid_name, cpu_type)
                if p_key in cpu_named_thread_dict:
                    cpu_named_thread_dict[p_key] += thread_value * frame_duration_ms
                else:
                    cpu_named_thread_dict[p_key] = thread_value * frame_duration_ms

            sorted_cpu_named_thread_dict = {}
            for ii, key_tid in enumerate(sorted_thread_keys):
                tid_name = self.tp.get_thread_name(key_tid)
                if (tid_name, "P Core") in cpu_named_thread_dict:
                    sorted_cpu_named_thread_dict[(tid_name, "P Core")] = cpu_named_thread_dict[(tid_name, "P Core")]
                if (tid_name, "E Core") in cpu_named_thread_dict:
                    e_time = cpu_named_thread_dict[(tid_name, "E Core")]
                    sorted_cpu_named_thread_dict[(tid_name, "E Core")] = e_time
                    # Top 3 threads
                    if ii < 3 and e_time > frame_duration_ms * 0.05:  # 5% of a frame
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.OS_SCHEDULING, health_string=f"[{e_time:.1f}ms] : Thread '{tid_name}' spent {e_time:.1f}ms running on an E-core.", health_value=e_time, isPriority=True))

            # Convert into a series
            df = pd.Series(sorted_cpu_named_thread_dict).reset_index()
            df.columns = ['Threads', 'P/E Core', 'Time (ms)']
            self.tp.df_dict[df_name + "_cpu_thread_time"] = df

        # Find the busiest threads
        total_time, thread_utilisation_dict = self.tp.get_thread_utilisation(frame_timeslice_list, True)
        if total_time <= 0:
            if params.report_range in (HotspotRange.NONE, HotspotRange.CUDNN_GPU_KERNELS):
                # Add a fake thread for GPU ranges
                thread_utilisation_dict[0] = 0
            else:
                logger.warning("NO CPU activity found for the given metrics")
                return

        sorted_thread_keys = sorted(thread_utilisation_dict, key=thread_utilisation_dict.get, reverse=True)
        max_thread_utilisation = thread_utilisation_dict[sorted_thread_keys[0]] if len(thread_utilisation_dict) > 0 else 0

        # Add a thread utilisation df for the html graph
        named_thread_time_dict = self.__create_named_thread_utilisation_dict(thread_utilisation_dict, sorted_thread_keys, params.main_pid)
        frame_duration_ms = params.frame_duration.duration / 1000000
        named_thread_time_dict.update({k: frame_duration_ms * i for k, i in named_thread_time_dict.items()})

        self.tp.add_dataframe_from_dict(
            df_name + "_thread_time",
            {
                "Threads": named_thread_time_dict.keys(),
                "Time (ms)": named_thread_time_dict.values()
            },
            None,
            False
        )

        ###############################################
        #
        # Find which threads to look at
        #
        ###############################################
        target_gtid_list = []

        # Generate a list of threads to look at
        if params.report_threading == HotspotReportThreading.ALL:
            target_gtid_list.append(None)

        elif params.report_threading == HotspotReportThreading.BUSY_THREAD:
            for ii in sorted_thread_keys:
                if thread_utilisation_dict[ii] > 0.01:
                    target_gtid_list.append(ii)

        tid_count = len(target_gtid_list)

        # Display all of the threads
        if params.report_thread_count and tid_count > params.report_thread_count:
            tid_count = params.report_thread_count

        # Don't report the threads if we're tracking all of them
        if params.report_threading != HotspotReportThreading.ALL:
            frame_info_dict["Threads"] = tid_count

        if median:
            region_name = "Median"
        elif params.report_range == HotspotRange.SLOWEST:
            region_name = f"Slow Frame {params.frame_index + 1}"
        elif params.report_range == HotspotRange.PERIODIC:
            region_name = f"Periodic Frame {params.frame_index + 1}"
        else:
            region_name = f"Slow Region {params.frame_index + 1}"

        hotspot_region = RegionCreate(
            hotspot_analysis_id=0,
            region_name=region_name,  # DONE
            region_type=params.frame_string,  # DONE
            region_index=params.frame_index,
            is_median=bool(median),
            region_index_global=params.frame_index_global,
            duration_ms=params.frame_duration.duration / 1000000,  # DONE
            start_time_s=params.frame_duration.start / 1000000000,  # DONE
            summed_metric_value=params.frame_duration.total_value if params.frame_duration.total_value > 0 else None,
            threads_count=tid_count,
            custom_fields=[]
        )

        if params.extra_frame_info_dict is not None:
            for k, v in params.extra_frame_info_dict.items():
                hotspot_region.custom_fields.append(RegionCustomFieldsCreate(name=k, value=str(v)))

        hotspot_region_patch = RegionCreate.model_as_partial()(
            submit_threads=[],
            present_threads=[],
            region_app_dxg_kernel_profile_ranges=[],
            region_etw_events=[],
            region_issues=[],
            is_cpu_bound=None,
            is_median=bool(median),
            region_metrics=[]
        )

        database_region_ids: Optional[DatabaseIds] = self.report_exporter.create_remote_hotspot_region(hotspot_region, hotspot_analysis_ids=params.database_hotspot_analysis_ids)

        threads_create_dict: dict[int, RegionThreadCreate] = {}
        for thread_id in sorted_thread_keys:
            thread_name = self.tp.get_thread_name(thread_id)
            utilization_pct = thread_utilisation_dict.get(thread_id, 0)
            time_ms = named_thread_time_dict.get(thread_name, 0)
            time_pcore_ms: Optional[float] = None
            time_ecore_ms: Optional[float] = None
            utilization_pcore_pct: Optional[float] = None
            utilization_ecore_pct: Optional[float] = None
            if cpu_named_thread_dict is not None:
                time_pcore_ms = cpu_named_thread_dict.get((thread_name, "P Core"), 0)
                time_ecore_ms = cpu_named_thread_dict.get((thread_name, "E Core"), 0)
                if time_ms:
                    utilization_pcore_pct = utilization_pct * time_pcore_ms / time_ms
                    utilization_ecore_pct = utilization_pct * time_ecore_ms / time_ms

            threads_create_dict[thread_id] = RegionThreadCreate(
                region_id=0,
                name=thread_name,
                thread_id=thread_id,
                utilization_pct=utilization_pct,
                time_ms=time_ms,
                has_info=thread_id in target_gtid_list,

                time_pcore_ms=time_pcore_ms,
                time_ecore_ms=time_ecore_ms,
                utilization_pcore_pct=utilization_pcore_pct,
                utilization_ecore_pct=utilization_ecore_pct,
                thread_callstack=[],  # DONE
                thread_hotspot_collections=[]
            )

        ###############################################
        #
        # Walk the threads to process them
        #
        ###############################################
        target_time_start = params.frame_duration.start
        target_time_end = params.frame_duration.start + params.frame_duration.duration

        # Get all callstacks for this pid for this time period
        cs_list_all_threads = self.tp.get_callstacks(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid)

        for i in range(0, tid_count):  # pylint: disable=too-many-nested-blocks
            hotspot_list: list[HotspotCollection] = []

            if target_gtid_list[i] is not None:
                target_pid, target_tid = tu.convert_global_tid(target_gtid_list[i])
                # Filter for this thread, if there is a thread
                cs_list = self.tp.filter_callstacks(target_tid, cs_list_all_threads)
                target_tid_name = self.tp.get_thread_name(target_gtid_list[i])
            else:
                target_tid = None
                target_tid_name = None
                cs_list = cs_list_all_threads

            thread_create: RegionThreadCreate

            if params.report_threading != HotspotReportThreading.ALL:
                frame_thread_info_dict["Name"].append(self.tp.get_thread_name(target_gtid_list[i]))
                frame_thread_info_dict["Utilisation (%)"].append(thread_utilisation_dict[target_gtid_list[i]] * 100)
                frame_thread_info_dict["TID"].append(target_tid)
                df_tid_name = df_name + "_Thread_" + str(target_tid)
                thread_create = threads_create_dict[target_gtid_list[i]]
            else:
                frame_thread_info_dict["Name"].append('All')
                frame_thread_info_dict["Utilisation (%)"].append(max_thread_utilisation * 100)
                frame_thread_info_dict["TID"].append(0)
                df_tid_name = df_name + "_Thread_0"
                thread_create =  RegionThreadCreate(
                    region_id=0,
                    name='All',
                    thread_id=0,
                    utilization_pct=max_thread_utilisation * 100,
                    time_ms=0,  # TODO: can calculate?
                    has_info=True,

                    time_pcore_ms=None,  # TODO: can calculate?
                    time_ecore_ms=None,  # TODO: can calculate?
                    utilization_pcore_pct=None,  # TODO: can calculate?
                    utilization_ecore_pct=None,  # TODO: can calculate?
                    thread_callstack=[],
                    thread_hotspot_collections=[]
                )
                threads_create_dict[0] = thread_create

            # Walk over the modules
            if HotspotMetricType.MODULE_NAME in params.report_metric:
                hot_modules = self.__get_hot_modules(cs_list, tu.CallStackType.SAMPLED)
                hotspot_list.append(hot_modules)

            if HotspotMetricType.SYMBOL_NAME in params.report_metric:
                hs = HotspotCollection()
                hs.type = HotspotMetricType.SYMBOL_NAME
                hs.name_string = str(hs.type)
                hs.flamegraph_string, thread_callstack = self.get_callstacks_for_flamegraph(cs_list, params.main_pid)
                if hs.flamegraph_string:
                    name = df_tid_name + "_" + hs.name_string
                    fg_input_file = name.replace(" ", "_")
                    self.flamegraph_list.append(Flamegraph(hs.flamegraph_string, fg_input_file))
                    hotspot_list.append(hs)
                    thread_create.thread_callstack = thread_callstack

            if HotspotMetricType.KNOWN_SYMBOL_NAMES in params.report_metric:
                # First, just get the sampled hotspots
                hs = self.__get_hot_symbols(cs_list, True, True, tu.CallStackType.SAMPLED)
                if hs:
                    hotspot_list.append(hs)

                # Now get the hotspots based on ALL callstack types as this is used for the frame health
                hs = self.__get_hot_symbols(cs_list, True, False)
                if hs:
                    CPSO = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::CreatePipelineState')
                    CGPSO = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::CreateGraphicsPipelineState')
                    CCPSO = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::CreateComputePipelineState')
                    CGPSOvk = hs.getHotspotSampleCount('nvoglv64.dll@vkCreateGraphicsPipeline')
                    CCPSOvk = hs.getHotspotSampleCount('nvoglv64.dll@vkCreateComputePipeline')
                    CPR = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::CreatePlacedResource')
                    CCR = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::CreateCommittedResource')
                    UTM = hs.getHotspotSampleCount('D3D12Core.dll@CDevice::UpdateTileMapping')
                    ECL = hs.getHotspotSampleCount('D3D12Core.dll@CCommandQueue<0>::ExecuteCommandLists')
                    ECLvk = hs.getHotspotSampleCount('nvoglv64.dll@vkQueueSubmit')
                    # PageFault = hs.getHotspotSampleCount('ntoskrnl.exe@KiPageFault')

                    total_PSO = 0
                    for PSO in (CPSO, CGPSO, CCPSO, CGPSOvk, CCPSOvk):
                        if PSO:
                            total_PSO += PSO

                    total_CR = 0
                    for CR in (CPR, CCR):
                        if CR:
                            total_CR += CR

                    total_samples = hs.raw_total_samples

                    if total_PSO > 0:  # Any PSOs should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.COMPILATION, health_string=f"{total_PSO} callstacks containing Pipeline State Object related compile functions in thread '{target_tid_name}'.", health_value=total_PSO, isPriority=False))
                    if total_CR > 0:  # Any CRs should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC, health_string=f"{total_CR} callstacks containing CreateResource related functions in thread '{target_tid_name}'.", health_value=total_CR, isPriority=False))
                    if UTM and UTM > 0:  # Any UTMs should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_PAGING, health_string=f"{UTM} callstacks containing UpdateTileMapping in thread '{target_tid_name}'.", health_value=UTM, isPriority=False))
                    if ECL and ECL > total_samples * 0.05:  # > 5% of samples?
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SUBMITTING_WORK, health_string=f"{ECL} callstacks containing ExecuteCommandLists in thread '{target_tid_name}'.", health_value=ECL, isPriority=False))
                    if ECLvk and ECLvk > total_samples * 0.05:  # > 5% of samples?
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SUBMITTING_WORK, health_string=f"{ECLvk} callstacks containing vkQueueSubmit in thread '{target_tid_name}'.", health_value=ECLvk, isPriority=False))
            #                    if PageFault and PageFault > 0:
            #                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.INTERRUPTS, health_string=f"{PageFault} callstacks containing KiPageFault in thread '{target_tid_name}'."))

            if HotspotMetricType.DX12_API in params.report_metric:
                # Get the DX12 API calls
                hs = self.__get_hot_dx12_api_calls(params.main_pid, target_tid, target_time_start, target_time_end)
                if hs:
                    hotspot_list.append(hs)
                    PSO = hs.getHotspotDuration('ID3D12Device2::CreatePipelineState')
                    Map = hs.getHotspotSampleCount('ID3D12Resource::Map')  # Sample count is more useful than duration of the actual call
                    Unmap = hs.getHotspotSampleCount('ID3D12Resource::Unmap')
                    if PSO:  # Any PSOs should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.COMPILATION, health_string=f"[{PSO:.1f} ms] : CreatePipelineState() was executing for {PSO:.1f} ms in thread '{target_tid_name}'.", health_value=PSO, isPriority=True))
                    if Map:  # Any Map/Unmap should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_MAPPING, health_string=f"ID3D12Resource::Map() was called {Map} times in thread '{target_tid_name}'.", health_value=Map, isPriority=False))
                    if Unmap:  # Any Map/Unmap should be flagged
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_MAPPING, health_string=f"ID3D12Resource::Unmap() was called {Unmap} times in thread '{target_tid_name}'.", health_value=Unmap, isPriority=False))

            if HotspotMetricType.CUDA_API in params.report_metric:
                # Get the CUDA API calls
                hotspot_list.append(self.__get_hot_cuda_api_calls(params.main_pid, target_tid, target_time_start, target_time_end))

            if params.report_range in (HotspotRange.NONE, HotspotRange.CUDNN_GPU_KERNELS) and HotspotMetricType.CUDA_GPU_KERNELS in params.report_metric:
                # Get the CUDA GPU calls
                hotspots, ordered_list = self.__get_combined_hotspot_events(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid, target_tid=target_tid, metric_type=HotspotMetricType.CUDA_GPU_KERNELS)
                hotspot_list.append(hotspots)
                hotspot_list.append(ordered_list)

            if HotspotMetricType.ETW_EVENTS in params.report_metric:
                # Get the ETW events
                hot_etw_events: Optional[HotspotCollection] = self.__get_hot_etw_events(params.main_pid, target_tid, target_time_start, target_time_end)
                hotspot_list.append(hot_etw_events)

            if HotspotMetricType.PIX_MARKERS in params.report_metric:
                # Get the PIX markers
                hotspot_list.append(self.__get_hot_pix_markers(params.main_pid, target_tid, target_time_start, target_time_end))
                hotspot_list.append(self.__get_ordered_events(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid, target_tid=target_tid, metric_type=HotspotMetricType.PIX_MARKERS))

            if HotspotMetricType.NVTX_MARKERS in params.report_metric:
                # Get the NVTX markers
                hotspots, ordered_list = self.__get_combined_hotspot_events(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid, target_tid=target_tid, metric_type=HotspotMetricType.NVTX_MARKERS)
                hotspot_list.append(hotspots)
                hotspot_list.append(ordered_list)

            if params.report_range in (HotspotRange.NONE, HotspotRange.CUDNN_GPU_KERNELS) and HotspotMetricType.NVTX_GPU_MARKERS in params.report_metric:
                # Get the NVTX markers
                hotspots, ordered_list = self.__get_combined_hotspot_events(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid, target_tid=target_tid, metric_type=HotspotMetricType.NVTX_GPU_MARKERS)
                hotspot_list.append(hotspots)
                hotspot_list.append(ordered_list)

            if HotspotMetricType.MPI_MARKERS in params.report_metric:
                # Get the MPI markers
                hotspot_list.append(self.__get_hot_mpi_markers(params.main_pid, target_tid, target_time_start, target_time_end))

            if HotspotMetricType.DX12_GPU_WORKLOAD in params.report_metric:
                hotspots, ordered_list = self.__get_combined_hotspot_events(start_time_ns=target_time_start, end_time_ns=target_time_end, target_pid=params.main_pid, target_tid=target_tid, metric_type=HotspotMetricType.DX12_GPU_WORKLOAD)
                hotspot_list.append(hotspots)
                hotspot_list.append(ordered_list)

            if HotspotMetricType.DXG_KRNL_PROFILE_RANGES in params.report_metric:
                # Get the Kernel Profile Ranges for the entire application
                hs = self.__get_hot_dxgkrnl_profile_ranges(params.main_pid, target_tid, target_time_start, target_time_end)
                hotspot_list.append(hs)
                if hs:
                    # Ignore anything less than 5% of the frame
                    self.__report_dxgkrnl_profile_health_issues(hs, target_tid_name, frame_duration_ms * 0.05, is_priority=False)

            if HotspotMetricType.CSWITCH_SYMBOLS in params.report_metric:
                # Get the dxgKrnl Profile Ranges
                hs = self.__get_hot_context_switch_symbols(params.main_pid, target_tid, target_time_start, target_time_end, frame_timeslice_list, cs_list)
                hotspot_list.append(hs)
                if i < 2 or target_tid in self.present_tid_list or target_tid in self.submit_tid_list:  # Look at top 2 threads and present/submit threads
                    for h in hs.hotspot_list:
                        # Ignore these for now
                        if h.name is None:
                            continue
                        if h.duration > max(frame_duration_ms * 0.1, 1.0):
                            category = FrameHealthCategory.CSWITCH_THREAD
                            self.frame_health.frame_health_list.append(FrameHealthItem(category=category, health_string=f"[{h.duration:.1f} ms] : Thread '{target_tid_name}' switched out for {h.duration:.1f} ms from function {h.name}.", health_value=h.duration, isPriority=True))
                            if self.is_substring_in_string(h.name, ["EtwWriteTransfer"]):
                                category = FrameHealthCategory.TOOLS_OVERHEAD
                                self.frame_health.frame_health_list.append(FrameHealthItem(category=category, health_string=f"[{h.duration:.1f} ms] : Thread '{target_tid_name}' switched out for {h.duration:.1f} ms from function {h.name}.", health_value=h.duration, isPriority=True))
                            if self.is_substring_in_string(h.name, ["KiDpcInterrupt"]):
                                category = FrameHealthCategory.INTERRUPTS
                                self.frame_health.frame_health_list.append(FrameHealthItem(category=category, health_string=f"[{h.duration:.1f} ms] : Thread '{target_tid_name}' switched out for {h.duration:.1f} ms from function {h.name}.", health_value=h.duration, isPriority=True))

            # Filter out the empty hs lists - perhaps because thw metric didn't exist
            hotspot_list: list[HotspotCollection] = list(filter(None, hotspot_list))

            # Add the dictionaries to the data frames
            for hs in hotspot_list:
                name = self.tp.get_dataframe_key(df_tid_name + "_" + hs.name_string)
                hs_dict = self.create_hotspot_dict(hs)
                if hs_dict is None:
                    continue
                # logger.info("Adding HS : ", name)
                self.tp.add_dataframe_from_dict(name, hs_dict, None, False)
                collection_name = hs.type.value
                view_type = 'table'
                if hs.type == HotspotMetricType.ETW_EVENTS:
                    view_type = 'pie'
                elif hs.type == HotspotMetricType.MODULE_NAME:
                    view_type = 'bar'
                    collection_name = 'Modules in Sampled Callstacks'
                elif hs.type == HotspotMetricType.KNOWN_SYMBOL_NAMES:
                    collection_name = 'Known Symbols From Sampled Callstacks'
                elif hs.type == HotspotMetricType.CSWITCH_SYMBOLS:
                    collection_name = 'Context Switch Callstacks'
                hotspot_collection_create = ThreadHotspotCollectionCreate(
                    type=hs.type.value,
                    name=collection_name,
                    start_time_ns=hs.start_time_ns,
                    end_time_ns=hs.end_time_ns,
                    raw_total_samples=hs.raw_total_samples,
                    raw_total_duration=hs.raw_total_duration,
                    sample_string=hs.sample_string,
                    start_time_string=hs.start_time_string,
                    duration_string=hs.duration_string,
                    unit_string=hs.unit_string,
                    name_string=hs.name_string,
                    is_duration=hs.isDuration,
                    extra_1_string=hs.extra_1_string,
                    extra_2_string=hs.extra_2_string,
                    view_type=view_type,

                    thread_hotspot_records=[],
                )
                for hotspot in hs.hotspot_list:
                    hotspot_collection_create.thread_hotspot_records.append(ThreadHotspotRecordCreate(
                        name=hotspot.name,
                        value=hotspot.value,
                        duration=hotspot.duration,
                        sample_count=hotspot.sample_count,
                        start_time=hotspot.start_time,
                        extra_1=hotspot.extra_1,
                        extra_2=hotspot.extra_2,
                    ))
                thread_create.thread_hotspot_collections.append(hotspot_collection_create)

        # Get the GPU Metrics
        gpu_metrics, gpu_metric_sample_count = self.tp.get_all_average_gpu_metrics(target_time_start, target_time_end)
        if gpu_metrics:
            self.tp.add_dataframe_from_dict(df_name + "_gpu_metrics", {"GPU Metrics": gpu_metrics.keys(), "Average Values": gpu_metrics.values()}, None, False)
            key = self.tp.get_gpu_metric_name(TraceLoaderGPUMetrics.GPU_UTILISATION)
            if key and key in gpu_metrics:
                self.frame_health.cpu_bound = gpu_metrics[key] < 95  # 95%?
                if not self.frame_health.ignore_cpu_bound_issues(params.report_range):
                    if self.frame_health.cpu_bound:
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.CPU_BOUND, health_string=f"CPU Bound - GPU Activity is {gpu_metrics[key]:.1f}%", health_value=gpu_metrics[key], isPriority=True))
                    else:
                        self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.GPU_BOUND, health_string=f"GPU Bound - GPU Activity is {gpu_metrics[key]:.1f}%, any following issues may not be impacting the performance of this frame/region and are informational only.", health_value=gpu_metrics[key], isPriority=True))

            key = self.tp.get_gpu_metric_name(TraceLoaderGPUMetrics.PCIE_BAR1_READS)
            if gpu_metrics.get(key, 0) > 50:  # Is this a good number?
                self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.BAR1_READ, health_string=f"High number of read requests to Bar1 - {gpu_metrics[key]}", health_value=gpu_metrics[key], isPriority=False))
            # TODO: new metrics will not be saved
            for key, value in gpu_metrics.items():
                hotspot_region_patch.region_metrics.append(RegionMetricsCreate(
                    name=key,
                    value=tu.safe_float(value),
                    unit=tu.get_metric_unit(key),
                ))

        # Get more 'per frame' data
        if HotspotMetricType.ETW_EVENTS in params.report_metric:
            # Get the System (pid=4) ETW events
            hs_system = self.__get_hot_etw_events(4, None, target_time_start, target_time_end)
            hs_dict = self.create_hotspot_dict(hs_system)
            if hs_dict:
                self.tp.add_dataframe_from_dict(df_name + "_System_" + hs_system.name_string, hs_dict, None, False)

                allocationFault = hs_system.getHotspotSampleCount('AllocationFault')
                commitVirtualAddress = hs_system.getHotspotSampleCount('CommitVirtualAddress')
                gpuVirtualAddressRangeMapping = hs_system.getHotspotSampleCount('GpuVirtualAddressRangeMapping')
                pagingOpUpdatePageTable = hs_system.getHotspotSampleCount('PagingOpUpdatePageTable')

                if allocationFault and allocationFault > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC, health_string=f"<b>System process generated {allocationFault} VRAM allocation fault events (paging).</b>", health_value=allocationFault, isPriority=False))
                if commitVirtualAddress and commitVirtualAddress > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC, health_string=f"<b>System process generated {commitVirtualAddress} VRAM virtual address events.</b>", health_value=commitVirtualAddress, isPriority=False))
                if gpuVirtualAddressRangeMapping and gpuVirtualAddressRangeMapping > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_MAPPING, health_string=f"<b>System process generated {gpuVirtualAddressRangeMapping} VRAM virtual address mapping events.</b>", health_value=gpuVirtualAddressRangeMapping, isPriority=False))
                if pagingOpUpdatePageTable and pagingOpUpdatePageTable > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_PAGING, health_string=f"<b>System process generated {pagingOpUpdatePageTable} VRAM page table update events.</b>", health_value=pagingOpUpdatePageTable, isPriority=False))
                for hotspot in hs_system.hotspot_list:
                    hotspot_region_patch.region_etw_events.append(RegionEtwEventsCreate(
                        name=hotspot.name,
                        count=hotspot.sample_count,
                        pct=hotspot.value,
                        type='system'
                    ))

            # Get the ETW events for the entire application
            hs_pid = self.__get_hot_etw_events(params.main_pid, None, target_time_start, target_time_end)
            hs_dict = self.create_hotspot_dict(hs_pid)
            if hs_dict:
                self.tp.add_dataframe_from_dict(df_name + "_App_" + hs_pid.name_string, hs_dict, None, False)

                adapterAllocation = hs_pid.getHotspotSampleCount('AdapterAllocation')
                updateGpuVirtualAddress = hs_pid.getHotspotSampleCount('UpdateGpuVirtualAddress')
                vidMmMakeResident = hs_pid.getHotspotSampleCount('VidMmMakeResident')
                waitForSynchronizationObjectFromGpu = hs_pid.getHotspotSampleCount('WaitForSynchronizationObjectFromGpu')
                waitForSynchronizationObjectFromCpu = hs_pid.getHotspotSampleCount('WaitForSynchronizationObjectFromCpu')
                blockThreads_dict = hs_pid.getHotspotMultiSampleCount('BlockThread:')

                if adapterAllocation and adapterAllocation > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC, health_string=f"<b>Application adapter has made {adapterAllocation} VRAM allocations.</b>", health_value=adapterAllocation, isPriority=False))
                if updateGpuVirtualAddress and updateGpuVirtualAddress > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_ALLOC, health_string=f"<b>Application has generated {updateGpuVirtualAddress} VRAM virtual address update events.</b>", health_value=updateGpuVirtualAddress, isPriority=False))
                if vidMmMakeResident and vidMmMakeResident > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.VRAM_PAGING, health_string=f"<b>Application has generated {vidMmMakeResident} VRAM make resident events.</b>", health_value=vidMmMakeResident, isPriority=False))
                if waitForSynchronizationObjectFromGpu and waitForSynchronizationObjectFromGpu > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SYNC_FROM_GPU, health_string=f"Application is stalling waiting for a GPU fence {waitForSynchronizationObjectFromGpu} times.", health_value=waitForSynchronizationObjectFromGpu, isPriority=False))
                if waitForSynchronizationObjectFromCpu and waitForSynchronizationObjectFromCpu > 0:
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.SYNC_FROM_CPU, health_string=f"Application is stalling waiting for a CPU fence {waitForSynchronizationObjectFromCpu} times (likely from the graphics kernel).", health_value=waitForSynchronizationObjectFromCpu, isPriority=False))
                for key, item in blockThreads_dict.items():
                    self.frame_health.frame_health_list.append(FrameHealthItem(category=FrameHealthCategory.BLOCKED_THREAD, health_string=f"Application threads were blocked by DxgKrnl process due to {str(key)}, {item} times.", health_value=item, isPriority=False))
                for hotspot in hs_pid.hotspot_list:
                    hotspot_region_patch.region_etw_events.append(RegionEtwEventsCreate(
                        name=hotspot.name,
                        count=hotspot.sample_count,
                        pct=hotspot.value,
                        type='app'
                    ))

        if HotspotMetricType.DXG_KRNL_PROFILE_RANGES in params.report_metric:
            # Get the Kernel Profile Ranges for the entire application
            hs_app = self.__get_hot_dxgkrnl_profile_ranges(params.main_pid, None, target_time_start, target_time_end)
            if hs_app:
                hs_app.trim(min_value=5.0)  # Ignore anything less than 5% of the frame
                hs_dict = self.create_hotspot_dict(hs_app, "Top DxgKrnl Profile Event Ranges")
                if hs_dict:
                    self.tp.add_dataframe_from_dict(df_name + "_App_" + hs_app.name_string, hs_dict, None, False)

                    self.__report_dxgkrnl_profile_health_issues(hs_app, None, frame_duration_ms * 0.05, is_priority=True)
                    for hotspot in hs_app.hotspot_list:
                        hotspot_region_patch.region_app_dxg_kernel_profile_ranges.append(RegionAppDxgKernelProfileRangesCreate(
                            frame_duration_pct=hotspot.value,
                            duration_ms=hotspot.duration,
                            range_count=hotspot.sample_count,
                            top_dxg_kernel_profile_event_ranges=hotspot.name
                        ))

        health_issues_dict = self.frame_health.get_dict()
        region_str = "Region"
        if params.report_range in (HotspotRange.PERIODIC, HotspotRange.SLOWEST):
            region_str = "Frame"

        if not median:
            if self.frame_health.ignore_cpu_bound_issues(params.report_range) or self.frame_health.cpu_bound:
                self.app_health_gpu_bound = False

            self.app_health_summary += self.frame_health.get_issue_list()
            self.frame_health_summary[f"{region_str} {str(params.frame_index + 1)}"] = filter_and_prioritise_categorys(self.frame_health.get_issue_list())
        else:
            self.frame_health_summary[f"Median {region_str}"] = filter_and_prioritise_categorys(self.frame_health.get_issue_list())

        if self.present_tid_list:
            frame_info_dict['Present Thread'] = ", ".join([str(item) for item in self.present_tid_list])
            hotspot_region_patch.present_threads = [RegionPresentThreadsCreate(thread_id=tid) for tid in self.present_tid_list]
        if self.submit_tid_list:
            frame_info_dict['Submit Thread'] = ", ".join([str(item) for item in self.submit_tid_list])
            hotspot_region_patch.submit_threads = [RegionSubmitThreadsCreate(thread_id=tid) for tid in self.submit_tid_list]

        # Add the frame_info
        self.tp.add_dataframe_from_dict(df_name + "_issues", {"Performance Issues": health_issues_dict.keys(), "Issue Indicators": health_issues_dict.values()}, None, False)
        self.tp.add_dataframe_from_dict(df_name + "_info", {"Frame Info": frame_info_dict.keys(), "Values": frame_info_dict.values()}, None, False)
        self.tp.add_dataframe_from_dict(df_name + "_thread_info", frame_thread_info_dict)

        for health_issue in self.frame_health.frame_health_list:
            hotspot_region_patch.region_issues.append(RegionIssuesCreate(
                category=health_issue.category.value,
                health_string=health_issue.health_string,
                health_value=health_issue.health_value,
                is_priority=health_issue.isPriority,
                sub_category=health_issue.sub_category
            ))
        hotspot_region_patch.is_cpu_bound = self.frame_health.cpu_bound

        self.report_exporter.patch_remote_hotspot_region(database_region_ids, hotspot_region_patch)
        for thread_create in threads_create_dict.values():
            self.report_exporter.create_remote_hotspot_thread(thread_create, database_region_ids)

    ####################################################
    #
    # Main logic for the stutter analysis
    #
    ####################################################
    @tu.timeit
    def process_stutter_analysis(self,
                                 filtered_timeslice_list: Optional[List[TimeSlice]],
                                 start_time_ns: float,
                                 end_time_ns: float,
                                 target_pid: int,
                                 report_range: HotspotRange,
                                 report_threading: HotspotReportThreading,
                                 report_metric: HotspotMetricType,
                                 report_frame_count: int,
                                 report_thread_count: Optional[int],
                                 report_median_frame: bool,
                                 report_info: Optional[bool] = True,
                                 cpu_config: Optional[tu.CPUConfig] = None,
                                 quiet: Optional[bool] = False,
                                 report_exporter: Optional[ReportExporter] = None) -> None:

        capture_info_dict = {}
        region_overview_dict = {}
        self.app_health_summary = []
        self.frame_health_summary = {}

        self.report_exporter = report_exporter

        self.report_range = report_range
        self.report_threading = report_threading
        self.report_metric = report_metric

        if filtered_timeslice_list is None:
            filtered_timeslice_list, start_time_ns, end_time_ns, target_pid = self.get_filtered_timeslices(start_time_ns, end_time_ns, target_pid, quiet)
        duration = end_time_ns - start_time_ns

        self.process_name = self.tp.get_process_name(target_pid)
        self.quiet = quiet

        if report_info:
            capture_info_dict["Process Name"] = self.process_name
            capture_info_dict["PID"] = target_pid
            capture_info_dict["Start Time (s)"] = start_time_ns / 1000000000
            capture_info_dict["End Time (s)"] = end_time_ns / 1000000000
            capture_info_dict["Duration (s)"] = duration / 1000000000
            self.tp.add_dataframe_from_dict("Capture_Info", {"Capture Info": capture_info_dict.keys(), "Values": capture_info_dict.values()}, None, False)

        ###############################################
        #
        # Find some 'frames'
        #
        ###############################################
        sort_on_duration = True
        short_frame_descriptor_string = str(report_range)  # "PCIe_Read"
        frame_descriptor_string = None
        frametime_list_cpu: list[FrameDurations] = []

        if report_range == HotspotRange.NONE:
            frame_descriptor_string = "Full App Duration"
            frametime_list_cpu.append(FrameDurations(start_time_ns, end_time_ns - start_time_ns))

        elif report_range in (HotspotRange.SLOWEST, HotspotRange.PERIODIC,
                              HotspotRange.CUDNN_KERNEL_LAUNCHES, HotspotRange.CUDNN_GPU_KERNELS):
            frame_descriptor_string = str(report_range)
            if report_range in (HotspotRange.SLOWEST, HotspotRange.PERIODIC):
                _, frametime_list_cpu = self.tp.get_region_durations(region_type=TraceLoaderRegions.CPU_FRAMETIMES, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid)
            elif report_range == HotspotRange.CUDNN_KERNEL_LAUNCHES:
                _, frametime_list_cpu = self.tp.get_region_durations(region_type=TraceLoaderRegions.CUDNN_KERNEL_LAUNCHES, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid)
            elif report_range == HotspotRange.CUDNN_GPU_KERNELS:
                _, frametime_list_cpu = self.tp.get_region_durations(region_type=TraceLoaderRegions.CUDNN_GPU_KERNELS, start_time_ns=start_time_ns, end_time_ns=end_time_ns, target_pid=target_pid)
            else:
                logger.error(f"Invalid hotspot range {report_range}. Aborting ")
                return

            # Soft check - the markers may not be there
            if report_range in (HotspotRange.CUDNN_KERNEL_LAUNCHES,
                                HotspotRange.CUDNN_GPU_KERNELS):
                if frametime_list_cpu is None or len(frametime_list_cpu) == 0:
                    logger.error(f"No frametimes found for {frame_descriptor_string}. Aborting.")
                    return

            if frametime_list_cpu:
                # Store the frametimes if we are investigating slow frames
                frametimes = {
                    "duration_ms": [],
                    "start_ns": []
                }
                for x in frametime_list_cpu:
                    frametimes["duration_ms"].append(x.duration / 1000000)
                    frametimes["start_ns"].append(x.start)
                self.tp.add_dataframe_from_dict(self.__get_per_range_df_name(report_range, "Frametimes"), frametimes)

                # GPU frametimes don't always make sense
                if report_range not in (HotspotRange.CUDNN_KERNEL_LAUNCHES,
                                        HotspotRange.CUDNN_GPU_KERNELS):
                    frametime_list_gpu = self.tp.get_derived_region_durations(region_type=TraceLoaderRegions.GPU_FRAMETIMES, base_durations=frametime_list_cpu, start_time_ns=start_time_ns, end_time_ns=end_time_ns)
                    if frametime_list_gpu:
                        frametime_gpu_ms = [x.duration / 1000000 for x in frametime_list_gpu]
                        self.tp.add_dataframe_from_list(self.__get_per_range_df_name(report_range, "GPU_Frametimes"), frametime_gpu_ms, None)

        elif self.tp.is_supported(TraceLoaderSupport.GPU_METRICS) and report_range in (HotspotRange.PCIE_BAR1_READS, HotspotRange.PCIE_BAR1_WRITES, HotspotRange.GR_IDLE):
            gpu_metric_min = None
            gpu_metric_max = None
            gpu_metric_min_percent = None
            gpu_metric_max_percent = None

            gpu_metric_type = None
            if report_range == HotspotRange.PCIE_BAR1_READS:
                frame_descriptor_string = "PCIe Read Requests to BAR1 - Largest Summed Regions"
                gpu_metric_type = TraceLoaderGPUMetrics.PCIE_BAR1_READS
                gpu_metric_min = 1
                sort_on_duration = False
            if report_range == HotspotRange.PCIE_BAR1_WRITES:
                frame_descriptor_string = "PCIe Write Requests to BAR1 - Largest Summed Regions"
                gpu_metric_type = TraceLoaderGPUMetrics.PCIE_BAR1_WRITES
                gpu_metric_min = 1
                sort_on_duration = False
            if report_range == HotspotRange.GR_IDLE:
                frame_descriptor_string = "GR Idle - Longest Regions"
                gpu_metric_type = TraceLoaderGPUMetrics.GPU_UTILISATION
                gpu_metric_min = 0
                gpu_metric_max = 0
                sort_on_duration = True

            gpu_metric_list, frametime_list_cpu = self.tp.get_gpu_metric_frame_list(gpu_metric_type,
                                                                                    gpu_metric_min,
                                                                                    gpu_metric_max,
                                                                                    gpu_metric_min_percent,
                                                                                    gpu_metric_max_percent,
                                                                                    start_time_ns, end_time_ns)
            if frametime_list_cpu is None or len(frametime_list_cpu) == 0:
                logger.error(f"No data found for GPU metric within the required range {gpu_metric_min} -> {gpu_metric_max}. Aborting hotspot report {report_range}.")
                return

            gpu_metrics_name = self.__get_per_range_df_name(report_range, "GPUMetrics")
            self.tp.add_dataframe_from_list(gpu_metrics_name, gpu_metric_list, None)

        else:
            logger.error(f"Unsupported report type : {report_range}.")
            return

        ###############################################
        #
        # Sort them to find the slow/long frames
        #
        ###############################################
        if frametime_list_cpu is None or len(frametime_list_cpu) == 0:
            logger.error("No frame data found. Aborting.")
            return

        if sort_on_duration:
            sorted_frametime_list_cpu: list[FrameDurations] = sorted(frametime_list_cpu, key=operator.attrgetter('duration'), reverse=True)
            region_mean = statistics.mean([f.duration for f in frametime_list_cpu])
        else:
            sorted_frametime_list_cpu: list[FrameDurations] = sorted(frametime_list_cpu, key=operator.attrgetter('total_value'), reverse=True)
            region_mean = statistics.mean([f.total_value for f in frametime_list_cpu])

        sorted_gpumetric_regions_durations_name = self.__get_per_range_df_name(report_range, "sorted_gpumetric_regions_durations")
        sorted_gpumetric_regions_values_name = self.__get_per_range_df_name(report_range, "sorted_gpumetric_regions_values")
        if report_range in (HotspotRange.PCIE_BAR1_READS, HotspotRange.PCIE_BAR1_WRITES, HotspotRange.GR_IDLE):
            if sort_on_duration:
                sorted_frametime_ms = [x.duration / 1000000 for x in sorted_frametime_list_cpu]
                self.tp.add_dataframe_from_list(sorted_gpumetric_regions_durations_name, sorted_frametime_ms, None)
            else:
                sorted_frame_value = [x.total_value for x in sorted_frametime_list_cpu]
                self.tp.add_dataframe_from_list(sorted_gpumetric_regions_values_name, sorted_frame_value, None)

        frame_count = len(frametime_list_cpu)
        median_frame_index = int(frame_count / 2)

        region_overview_dict["Region Description"] = frame_descriptor_string
        if report_range in (HotspotRange.SLOWEST,
                            HotspotRange.PERIODIC,
                            HotspotRange.CUDNN_KERNEL_LAUNCHES,
                            HotspotRange.CUDNN_GPU_KERNELS):
            region_overview_dict["Number of regions"] = frame_count
            region_overview_dict["Fastest (ms)"] = sorted_frametime_list_cpu[frame_count - 1].duration / 1000000
            region_overview_dict["Slowest (ms)"] = sorted_frametime_list_cpu[0].duration / 1000000
            region_overview_dict["Median (ms)"] = sorted_frametime_list_cpu[median_frame_index].duration / 1000000
            region_overview_dict["Mean (ms)"] = region_mean / 1000000

            # Look for frame to frame variance and stutter
            # Take the 2nd descrete difference, so the rate of change over 3 frames
            # or spikiness
            if frame_count > 3:
                frametime_variance = np.fabs(np.diff(frametimes["duration_ms"], n=2))

                # first, find the top 95% - these are outliers for the frame to frame variance, so we ignore
                threshold_high = np.percentile(frametime_variance, 95)
                clipped_frametime_variance = np.clip(frametime_variance, 0, threshold_high)

                region_overview_dict["Average Micro-stutters (ms)"] = np.mean(clipped_frametime_variance) / 2
        #            region_overview_dict["Max Stutter Variance (ms)"] = np.max(frametime_variance)

        elif report_range in (HotspotRange.PCIE_BAR1_READS, HotspotRange.PCIE_BAR1_WRITES, HotspotRange.GR_IDLE):
            if sort_on_duration:
                region_overview_dict["Smallest time period (ms)"] = sorted_frametime_list_cpu[frame_count - 1].duration / 1000000
                region_overview_dict["Largest time period (ms)"] = sorted_frametime_list_cpu[0].duration / 1000000
                region_overview_dict["Median time period (ms)"] = sorted_frametime_list_cpu[median_frame_index].duration / 1000000
                region_overview_dict["Mean time period (ms)"] = region_mean / 1000000
            else:
                region_overview_dict["Smallest summed value"] = sorted_frametime_list_cpu[frame_count - 1].total_value
                region_overview_dict["Largest summed value"] = sorted_frametime_list_cpu[0].total_value
                region_overview_dict["Median summed value"] = sorted_frametime_list_cpu[median_frame_index].total_value
                region_overview_dict["Mean summed value"] = region_mean

        # logger.info(region_overview_dict)

        ####################################################
        #
        # If we've found some 'frames', start to process them
        #
        ####################################################
        if frame_count == 0:
            logger.error("NO frames found. Aborting.")
            return

        report_frame_count = min(report_frame_count, frame_count)

        region_overview_dict["Region Count"] = report_frame_count

        # Get the slow frame indices for the html graphs
        slow_frames_dict = {}
        if report_range in (HotspotRange.SLOWEST,
                            HotspotRange.CUDNN_KERNEL_LAUNCHES,
                            HotspotRange.CUDNN_GPU_KERNELS):
            slow_region_str = "Slow Region"
            if report_range == HotspotRange.SLOWEST:
                slow_region_str = "Slow Frame"
            si = []
            st = []
            for slow_frame_index in range(0, report_frame_count):
                for frame_index, frame_duration in enumerate(frametime_list_cpu):
                    if frame_duration.duration == sorted_frametime_list_cpu[slow_frame_index].duration:
                        si.append(frame_index)
                        st.append(frame_duration.duration / 1000000)
                        break
            slow_frames_dict["Frame Index"] = si
            slow_frames_dict["Frame Duration"] = st
            slow_frames_dict["Slow Frame Index"] = [slow_region_str + " " + str(index + 1) for index in range(0, report_frame_count)]
            self.tp.add_dataframe_from_dict(self.__get_per_range_df_name(report_range, "slow_frames"), slow_frames_dict, None, False)

        periodic_frames_dict = {}
        if report_range == HotspotRange.PERIODIC:
            periodic_frame_inc = frame_count / report_frame_count
            start_index = int(periodic_frame_inc * 0.5)
            fi = []
            ft = []
            for ii in range(0, report_frame_count):
                frame_index = start_index + int(periodic_frame_inc * ii)
                fi.append(frame_index)
                ft.append(frametime_list_cpu[frame_index].duration / 1000000)

            periodic_frames_dict["Frame Index"] = fi
            periodic_frames_dict["Frame Duration"] = ft
            periodic_frames_dict["Periodic Frame Index"] = ["Periodic Frame " + str(index + 1) for index in range(0, report_frame_count)]
            self.tp.add_dataframe_from_dict(self.__get_per_range_df_name(report_range, "periodic_frames"), periodic_frames_dict, None, False)

        def get_range_dataframe(name: str) -> Optional[pd.DataFrame]:
            range_name = self.__get_per_range_df_name(self.report_range, name)
            return self.tp.df_dict.get(range_name)

        database_hotspot_analysis_ids = self.report_exporter.create_remote_hotspot_analysis(
            region_overview=region_overview_dict,
            capture_info=capture_info_dict,
            cpu_frametimes=get_range_dataframe("Frametimes"),
            gpu_frametimes=get_range_dataframe("GPU_Frametimes"),
            sorted_gpumetric_regions_values=get_range_dataframe("sorted_gpumetric_regions_values"),
            sorted_gpumetric_regions_durations=get_range_dataframe("sorted_gpumetric_regions_durations"),
            app_health_gpu_bound=self.app_health_gpu_bound,
            report_range=self.report_range.value,
            report_threading=self.report_threading.value)

        if report_median_frame:
            extra_frame_info_dict = {}

            self.process_hotspot_frame(self.ProcessHotspotFrameParams(
                filtered_timeslice_list=filtered_timeslice_list,
                frame_string="Median Frame",
                short_frame_string=self.__get_per_range_df_name(report_range, "Median"),
                frame_index=median_frame_index,
                frame_index_global=median_frame_index,
                frame_duration=sorted_frametime_list_cpu[median_frame_index],
                main_pid=target_pid,
                report_thread_count=report_thread_count,
                report_range=report_range,
                report_threading=report_threading,
                report_metric=report_metric,
                cpu_config=cpu_config,
                extra_frame_info_dict=extra_frame_info_dict,
                database_hotspot_analysis_ids=database_hotspot_analysis_ids
            ))

        # Periodic frames aren't pulled from the sorted frame list
        if report_range == HotspotRange.PERIODIC:
            for frame_index in range(0, report_frame_count):
                self.process_hotspot_frame(self.ProcessHotspotFrameParams(
                    filtered_timeslice_list=filtered_timeslice_list,
                    frame_string=frame_descriptor_string,
                    short_frame_string=short_frame_descriptor_string,
                    frame_index=frame_index,
                    frame_index_global=tu.safe_list_get(periodic_frames_dict.get("Frame Index", []), frame_index, None),
                    frame_duration=frametime_list_cpu[periodic_frames_dict.get("Frame Index", [])[frame_index]],
                    main_pid=target_pid,
                    report_thread_count=report_thread_count,
                    report_range=report_range,
                    report_threading=report_threading,
                    report_metric=report_metric,
                    cpu_config=cpu_config,
                    extra_frame_info_dict=None,
                    database_hotspot_analysis_ids=database_hotspot_analysis_ids
                ))
        else:
            for slow_frame_index in range(0, report_frame_count):
                extra_frame_info_dict = {}
                self.process_hotspot_frame(
                    self.ProcessHotspotFrameParams(
                        filtered_timeslice_list=filtered_timeslice_list,
                        frame_string=frame_descriptor_string,
                        short_frame_string=short_frame_descriptor_string,
                        frame_index=slow_frame_index,
                        frame_index_global=tu.safe_list_get(slow_frames_dict.get("Frame Index", []), slow_frame_index, None),
                        frame_duration=sorted_frametime_list_cpu[slow_frame_index],
                        main_pid=target_pid,
                        report_thread_count=report_thread_count,
                        report_range=report_range,
                        report_threading=report_threading,
                        report_metric=report_metric,
                        cpu_config=cpu_config,
                        extra_frame_info_dict=extra_frame_info_dict,
                        database_hotspot_analysis_ids=database_hotspot_analysis_ids
                    ))

        region_issues_dict = {}
        # Add the frame health info at the end
        if len(self.app_health_summary) > 0:
            summary = ""
            if self.app_health_gpu_bound:
                summary = "GPU Bound. The following issues are informative only and may not be affecting performance."
            else:
                prioritised_app_health = filter_and_prioritise_categorys(self.app_health_summary)
                summary = ", ".join([str(item) for item in prioritised_app_health])
            region_issues_dict["Summary of Issues Detected"] = f"<b>{summary}</b>"

            for key, health_summary in self.frame_health_summary.items():
                region_issues_dict[f"- {key}"] = ", ".join([str(item) for item in health_summary])

        else:
            region_issues_dict["Summary of Issues Detected"] = "No issues detected."

        # This is threading type specific
        region_issues_name = self.__get_per_range_df_name(report_range, f'{report_threading}_Region_Issues')
        self.tp.add_dataframe_from_dict(region_issues_name, {"Performance Issues": region_issues_dict.keys(), "Issues Found": region_issues_dict.values()}, None, False)

        region_overview_name = self.__get_per_range_df_name(report_range, 'Region_Overview')
        self.tp.add_dataframe_from_dict(region_overview_name, {"Region Info": region_overview_dict.keys(), "Values": region_overview_dict.values()}, None, False)
