﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>
#include <aws/iam/model/Tag.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class CreateInstanceProfileRequest : public IAMRequest {
 public:
  AWS_IAM_API CreateInstanceProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateInstanceProfile"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the instance profile to create.</p> <p>This parameter allows
   * (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>) a
   * string of characters consisting of upper and lowercase alphanumeric characters
   * with no spaces. You can also include any of the following characters:
   * _+=,.@-</p>
   */
  inline const Aws::String& GetInstanceProfileName() const { return m_instanceProfileName; }
  inline bool InstanceProfileNameHasBeenSet() const { return m_instanceProfileNameHasBeenSet; }
  template <typename InstanceProfileNameT = Aws::String>
  void SetInstanceProfileName(InstanceProfileNameT&& value) {
    m_instanceProfileNameHasBeenSet = true;
    m_instanceProfileName = std::forward<InstanceProfileNameT>(value);
  }
  template <typename InstanceProfileNameT = Aws::String>
  CreateInstanceProfileRequest& WithInstanceProfileName(InstanceProfileNameT&& value) {
    SetInstanceProfileName(std::forward<InstanceProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The path to the instance profile. For more information about paths, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/Using_Identifiers.html">IAM
   * Identifiers</a> in the <i>IAM User Guide</i>.</p> <p>This parameter is optional.
   * If it is not included, it defaults to a slash (/).</p> <p>This parameter allows
   * (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>) a
   * string of characters consisting of either a forward slash (/) by itself or a
   * string that must begin and end with forward slashes. In addition, it can contain
   * any ASCII character from the ! (<code>\u0021</code>) through the DEL character
   * (<code>\u007F</code>), including most punctuation characters, digits, and upper
   * and lowercased letters.</p>
   */
  inline const Aws::String& GetPath() const { return m_path; }
  inline bool PathHasBeenSet() const { return m_pathHasBeenSet; }
  template <typename PathT = Aws::String>
  void SetPath(PathT&& value) {
    m_pathHasBeenSet = true;
    m_path = std::forward<PathT>(value);
  }
  template <typename PathT = Aws::String>
  CreateInstanceProfileRequest& WithPath(PathT&& value) {
    SetPath(std::forward<PathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags that you want to attach to the newly created IAM instance
   * profile. Each tag consists of a key name and an associated value. For more
   * information about tagging, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html">Tagging IAM
   * resources</a> in the <i>IAM User Guide</i>.</p>  <p>If any one of the tags
   * is invalid or if you exceed the allowed maximum number of tags, then the entire
   * request fails and the resource is not created.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateInstanceProfileRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateInstanceProfileRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceProfileName;

  Aws::String m_path;

  Aws::Vector<Tag> m_tags;
  bool m_instanceProfileNameHasBeenSet = false;
  bool m_pathHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
