﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/vpc-lattice/VPCLatticeRequest.h>
#include <aws/vpc-lattice/VPCLattice_EXPORTS.h>
#include <aws/vpc-lattice/model/AuthType.h>

#include <utility>

namespace Aws {
namespace VPCLattice {
namespace Model {

/**
 */
class UpdateServiceNetworkRequest : public VPCLatticeRequest {
 public:
  AWS_VPCLATTICE_API UpdateServiceNetworkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateServiceNetwork"; }

  AWS_VPCLATTICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID or ARN of the service network.</p>
   */
  inline const Aws::String& GetServiceNetworkIdentifier() const { return m_serviceNetworkIdentifier; }
  inline bool ServiceNetworkIdentifierHasBeenSet() const { return m_serviceNetworkIdentifierHasBeenSet; }
  template <typename ServiceNetworkIdentifierT = Aws::String>
  void SetServiceNetworkIdentifier(ServiceNetworkIdentifierT&& value) {
    m_serviceNetworkIdentifierHasBeenSet = true;
    m_serviceNetworkIdentifier = std::forward<ServiceNetworkIdentifierT>(value);
  }
  template <typename ServiceNetworkIdentifierT = Aws::String>
  UpdateServiceNetworkRequest& WithServiceNetworkIdentifier(ServiceNetworkIdentifierT&& value) {
    SetServiceNetworkIdentifier(std::forward<ServiceNetworkIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of IAM policy.</p> <ul> <li> <p> <code>NONE</code>: The resource
   * does not use an IAM policy. This is the default.</p> </li> <li> <p>
   * <code>AWS_IAM</code>: The resource uses an IAM policy. When this type is used,
   * auth is enabled and an auth policy is required.</p> </li> </ul>
   */
  inline AuthType GetAuthType() const { return m_authType; }
  inline bool AuthTypeHasBeenSet() const { return m_authTypeHasBeenSet; }
  inline void SetAuthType(AuthType value) {
    m_authTypeHasBeenSet = true;
    m_authType = value;
  }
  inline UpdateServiceNetworkRequest& WithAuthType(AuthType value) {
    SetAuthType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_serviceNetworkIdentifier;

  AuthType m_authType{AuthType::NOT_SET};
  bool m_serviceNetworkIdentifierHasBeenSet = false;
  bool m_authTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace VPCLattice
}  // namespace Aws
