﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codepipeline/CodePipelineRequest.h>
#include <aws/codepipeline/CodePipeline_EXPORTS.h>
#include <aws/codepipeline/model/CurrentRevision.h>
#include <aws/codepipeline/model/ExecutionDetails.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodePipeline {
namespace Model {

/**
 * <p>Represents the input of a <code>PutThirdPartyJobSuccessResult</code>
 * action.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codepipeline-2015-07-09/PutThirdPartyJobSuccessResultInput">AWS
 * API Reference</a></p>
 */
class PutThirdPartyJobSuccessResultRequest : public CodePipelineRequest {
 public:
  AWS_CODEPIPELINE_API PutThirdPartyJobSuccessResultRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutThirdPartyJobSuccessResult"; }

  AWS_CODEPIPELINE_API Aws::String SerializePayload() const override;

  AWS_CODEPIPELINE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the job that successfully completed. This is the same ID returned
   * from <code>PollForThirdPartyJobs</code>.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  PutThirdPartyJobSuccessResultRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The clientToken portion of the clientId and clientToken pair used to verify
   * that the calling entity is allowed access to the job and its details.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  PutThirdPartyJobSuccessResultRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents information about a current revision.</p>
   */
  inline const CurrentRevision& GetCurrentRevision() const { return m_currentRevision; }
  inline bool CurrentRevisionHasBeenSet() const { return m_currentRevisionHasBeenSet; }
  template <typename CurrentRevisionT = CurrentRevision>
  void SetCurrentRevision(CurrentRevisionT&& value) {
    m_currentRevisionHasBeenSet = true;
    m_currentRevision = std::forward<CurrentRevisionT>(value);
  }
  template <typename CurrentRevisionT = CurrentRevision>
  PutThirdPartyJobSuccessResultRequest& WithCurrentRevision(CurrentRevisionT&& value) {
    SetCurrentRevision(std::forward<CurrentRevisionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token generated by a job worker, such as a CodeDeploy deployment ID, that a
   * successful job provides to identify a partner action in progress. Future jobs
   * use this token to identify the running instance of the action. It can be reused
   * to return more information about the progress of the partner action. When the
   * action is complete, no continuation token should be supplied.</p>
   */
  inline const Aws::String& GetContinuationToken() const { return m_continuationToken; }
  inline bool ContinuationTokenHasBeenSet() const { return m_continuationTokenHasBeenSet; }
  template <typename ContinuationTokenT = Aws::String>
  void SetContinuationToken(ContinuationTokenT&& value) {
    m_continuationTokenHasBeenSet = true;
    m_continuationToken = std::forward<ContinuationTokenT>(value);
  }
  template <typename ContinuationTokenT = Aws::String>
  PutThirdPartyJobSuccessResultRequest& WithContinuationToken(ContinuationTokenT&& value) {
    SetContinuationToken(std::forward<ContinuationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details of the actions taken and results produced on an artifact as it
   * passes through stages in the pipeline. </p>
   */
  inline const ExecutionDetails& GetExecutionDetails() const { return m_executionDetails; }
  inline bool ExecutionDetailsHasBeenSet() const { return m_executionDetailsHasBeenSet; }
  template <typename ExecutionDetailsT = ExecutionDetails>
  void SetExecutionDetails(ExecutionDetailsT&& value) {
    m_executionDetailsHasBeenSet = true;
    m_executionDetails = std::forward<ExecutionDetailsT>(value);
  }
  template <typename ExecutionDetailsT = ExecutionDetails>
  PutThirdPartyJobSuccessResultRequest& WithExecutionDetails(ExecutionDetailsT&& value) {
    SetExecutionDetails(std::forward<ExecutionDetailsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  Aws::String m_clientToken;

  CurrentRevision m_currentRevision;

  Aws::String m_continuationToken;

  ExecutionDetails m_executionDetails;
  bool m_jobIdHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
  bool m_currentRevisionHasBeenSet = false;
  bool m_continuationTokenHasBeenSet = false;
  bool m_executionDetailsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodePipeline
}  // namespace Aws
