import { IrcBridge } from "./IrcBridge";
import { RoomAccessSyncer } from "./RoomAccessSyncer";
import { IrcServer } from "../irc/IrcServer";
import { BridgeRequest, BridgeRequestErr } from "../models/BridgeRequest";
import { BridgedClient } from "../irc/BridgedClient";
import { MatrixRoom, MembershipQueue } from "matrix-appservice-bridge";
import { IrcUser } from "../models/IrcUser";
import { IrcAction } from "../models/IrcAction";
import { Message } from "matrix-org-irc";
import { PrivacyProtection } from "../irc/PrivacyProtection";
export type MatrixDirectoryVisibility = "private" | "public";
export type MatrixMembership = "join" | "invite" | "leave" | "ban";
export interface IrcHandlerConfig {
    mapIrcMentionsToMatrix?: "on" | "off" | "force-off";
    powerLevelGracePeriodMs?: number;
}
type MetricNames = "join.names" | "join" | "part" | "pm" | "invite" | "topic" | "message" | "kick" | "mode";
export declare class IrcHandler {
    private readonly ircBridge;
    private readonly membershipQueue;
    private readonly privacyProtection;
    private readonly roomIdToPrivateMember;
    private readonly topicQueues;
    private readonly pmRoomPromises;
    private readonly nickUserIdMapCache;
    private mentionMode;
    readonly roomAccessSyncer: RoomAccessSyncer;
    private callCountMetrics?;
    private readonly registeredNicks;
    private pendingAdminRooms;
    constructor(ircBridge: IrcBridge, config: IrcHandlerConfig | undefined, membershipQueue: MembershipQueue, privacyProtection: PrivacyProtection);
    onMatrixMemberEvent(event: {
        room_id: string;
        state_key: string;
        content: {
            membership: MatrixMembership;
        };
    }): void;
    private ensureMatrixUserJoined;
    /**
     * Create a new matrix PM room for an IRC user with nick `fromUserNick` and another
     * matrix user with user ID `toUserId`.
     * @param req An associated request for contextual logging.
     * @param toUserId The user ID of the recipient.
     * @param fromUserId The user ID of the sender.
     * @param fromUserNick The nick of the sender.
     * @param server The sending IRC server.
     * @return A Promise which is resolved when the PM room has been created.
     */
    private createPmRoom;
    /**
     * Called when the AS receives an IRC message event.
     * @param {IrcServer} server The sending IRC server.
     * @param {IrcUser} fromUser The sender.
     * @param {IrcUser} toUser The target.
     * @param {Object} action The IRC action performed.
     * @return {Promise} which is resolved/rejected when the request
     * finishes.
     */
    onPrivateMessage(req: BridgeRequest, server: IrcServer, fromUser: IrcUser, toUser: IrcUser, action: IrcAction): Promise<BridgeRequestErr | void>;
    /**
     * Called when the AS receives an IRC invite event.
     * @param {IrcServer} server The sending IRC server.
     * @param {IrcUser} fromUser The sender.
     * @param {IrcUser} toUser The target.
     * @param {String} channel The channel.
     * @return {Promise} which is resolved/rejected when the request
     * finishes.
     */
    onInvite(req: BridgeRequest, server: IrcServer, fromUser: IrcUser, toUser: IrcUser, channel: string): Promise<BridgeRequestErr.ERR_VIRTUAL_USER | BridgeRequestErr.ERR_DROPPED | undefined>;
    private serviceTopicQueue;
    /**
     * Called when the AS receives an IRC topic event.
     * @param {IrcServer} server The sending IRC server.
     * @param {IrcUser} fromUser The sender.
     * @param {string} channel The target channel.
     * @param {Object} action The IRC action performed.
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onTopic(req: BridgeRequest, server: IrcServer, fromUser: IrcUser, channel: string, action: IrcAction): Promise<BridgeRequestErr.ERR_VIRTUAL_USER | BridgeRequestErr.ERR_NOT_MAPPED | undefined>;
    /**
     * Called when the AS receives an IRC message event.
     * @param {IrcServer} server The sending IRC server.
     * @param {IrcUser} fromUser The sender.
     * @param {string} channel The target channel.
     * @param {Object} action The IRC action performed.
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onMessage(req: BridgeRequest, server: IrcServer, fromUser: IrcUser, channel: string, action: IrcAction): Promise<BridgeRequestErr.ERR_VIRTUAL_USER | BridgeRequestErr.ERR_DROPPED | undefined>;
    /**
     * Called when the AS receives an IRC join event.
     * @param {IrcServer} server The sending IRC server.
     * @param {IrcUser} joiningUser The user who joined.
     * @param {string} chan The channel that was joined.
     * @param {string} kind The kind of join (e.g. from a member list if
     * the bot just connected, or an actual JOIN command)
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onJoin(req: BridgeRequest, server: IrcServer, joiningUser: IrcUser, chan: string, kind: "names" | "join" | "nick"): Promise<BridgeRequestErr.ERR_VIRTUAL_USER | BridgeRequestErr.ERR_NOT_MAPPED | undefined>;
    onKick(req: BridgeRequest, server: IrcServer, kicker: IrcUser, kickee: IrcUser, chan: string, reason: string): Promise<void>;
    /**
     * Called when the AS receives an IRC part event.
     * @param server The sending IRC server.
     * @param leavingUser The user who parted.
     * @param chan The channel that was left.
     * @param kind The kind of part (e.g. PART, KICK, BAN, QUIT, netsplit, etc)
     * @param reason: The reason why the client parted, if given.
     * @return A promise which is resolved/rejected when the request finishes.
     */
    onPart(req: BridgeRequest, server: IrcServer, leavingUser: IrcUser, chan: string, kind: string, reason?: string): Promise<BridgeRequestErr | undefined>;
    /**
     * Called when a user sets a mode in a channel.
     * @param {Request} req The metadata request
     * @param {IrcServer} server The sending IRC server.
     * @param {string} channel The channel that has the given mode.
     * @param {string} mode The mode that the channel is in, e.g. +sabcdef
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onMode(req: BridgeRequest, server: IrcServer, channel: string, by: string, mode: string, enabled: boolean, arg: string | null): Promise<BridgeRequestErr | undefined>;
    /**
     * Called when channel mode information is received
     * @param {Request} req The metadata request
     * @param {IrcServer} server The sending IRC server.
     * @param {string} channel The channel that has the given mode.
     * @param {string} mode The mode that the channel is in, e.g. +sabcdef
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onModeIs(req: BridgeRequest, server: IrcServer, channel: string, mode: string): Promise<void>;
    getOrCreateAdminRoom(req: BridgeRequest, userId: string, server: IrcServer, newRoomMsg?: string): Promise<MatrixRoom>;
    /**
     * Called when the AS connects/disconnects a Matrix user to IRC.
     * @param {Request} req The metadata request
     * @param {BridgedClient} client The client who is acting on behalf of the Matrix user.
     * @param {string} msg The message to share with the Matrix user.
     * @param {boolean} force True if ignoring startup suppresion.
     * @param ircMsg Optional data about the metadata.
     * @return {Promise} which is resolved/rejected when the request finishes.
     */
    onMetadata(req: BridgeRequest, client: BridgedClient, msg: string, force: boolean, ircMsg?: Message): Promise<void | BridgeRequestErr.ERR_DROPPED>;
    invalidateCachingForUserId(userId: string): boolean;
    incrementMetric(metric: MetricNames): void;
    getMetrics(): {
        join: number;
        message: number;
        kick: number;
        topic: number;
        pm: number;
        invite: number;
        part: number;
        "join.names": number;
        mode: number;
    };
    onConfigChanged(config: IrcHandlerConfig): void;
    private invalidateNickUserIdMap;
}
export {};
