/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.managers;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.Time;
import org.keycloak.cookie.CookieProvider;
import org.keycloak.cookie.CookieType;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.crypto.SignatureSignerContext;
import org.keycloak.forms.login.LoginFormsProvider;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.UserSessionProvider;
import org.keycloak.models.utils.SessionExpiration;
import org.keycloak.protocol.RestartLoginCookie;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.sessions.RootAuthenticationSessionModel;
import org.keycloak.sessions.StickySessionEncoderProvider;

public class AuthenticationSessionManager {
    private static final Logger log = Logger.getLogger(AuthenticationSessionManager.class);
    private static final Base64.Encoder BASE_64_ENCODER_NO_PADDING = Base64.getEncoder().withoutPadding();
    private final KeycloakSession session;

    public AuthenticationSessionManager(KeycloakSession session) {
        this.session = session;
    }

    public RootAuthenticationSessionModel createAuthenticationSession(RealmModel realm, boolean browserCookie) {
        RootAuthenticationSessionModel rootAuthSession = this.session.authenticationSessions().createRootAuthenticationSession(realm);
        if (browserCookie) {
            this.setAuthSessionCookie(rootAuthSession.getId());
            this.setAuthSessionIdHashCookie(rootAuthSession.getId());
        }
        return rootAuthSession;
    }

    public RootAuthenticationSessionModel getCurrentRootAuthenticationSession(RealmModel realm) {
        AuthSessionCookie authSession = this.getAuthSessionCookies(realm);
        if (authSession == null) {
            return null;
        }
        this.reEncodeAuthSessionCookie(authSession);
        return authSession.rootSession();
    }

    public AuthenticationSessionModel getCurrentAuthenticationSession(RealmModel realm, ClientModel client, String tabId) {
        AuthSessionCookie rootAuth = this.getAuthSessionCookies(realm);
        if (rootAuth == null) {
            return null;
        }
        AuthenticationSessionModel authSession = rootAuth.rootSession().getAuthenticationSession(client, tabId);
        if (authSession != null) {
            this.reEncodeAuthSessionCookie(rootAuth);
        }
        return authSession;
    }

    public void setAuthSessionCookie(String authSessionId) {
        StickySessionEncoderProvider encoder = (StickySessionEncoderProvider)this.session.getProvider(StickySessionEncoderProvider.class);
        String signedAuthSessionId = this.signAndEncodeToBase64AuthSessionId(authSessionId);
        String encodedWithRoute = encoder.encodeSessionId(signedAuthSessionId, authSessionId);
        ((CookieProvider)this.session.getProvider(CookieProvider.class)).set(CookieType.AUTH_SESSION_ID, encodedWithRoute);
        log.debugf("Set AUTH_SESSION_ID cookie with value %s", (Object)encodedWithRoute);
    }

    public void setAuthSessionIdHashCookie(String authSessionId) {
        String authSessionIdHash = BASE_64_ENCODER_NO_PADDING.encodeToString(HashUtils.hash((String)"SHA-256", (byte[])authSessionId.getBytes(StandardCharsets.UTF_8)));
        ((CookieProvider)this.session.getProvider(CookieProvider.class)).set(CookieType.AUTH_SESSION_ID_HASH, authSessionIdHash);
        log.debugf("Set KC_AUTH_SESSION_HASH cookie with value %s", (Object)authSessionIdHash);
    }

    private void reEncodeAuthSessionCookie(AuthSessionCookie authSessionCookie) {
        if (authSessionCookie.routeChanged()) {
            this.setAuthSessionCookie(authSessionCookie.sessionId());
        }
    }

    public String decodeBase64AndValidateSignature(String encodedBase64AuthSessionId) {
        try {
            String decodedAuthSessionId = new String(Base64Url.decode((String)encodedBase64AuthSessionId), StandardCharsets.UTF_8);
            int dotIndex = decodedAuthSessionId.lastIndexOf(46);
            if (dotIndex == -1) {
                return null;
            }
            String authSessionId = decodedAuthSessionId.substring(0, dotIndex);
            String signature = decodedAuthSessionId.substring(dotIndex + 1);
            return this.validateAuthSessionIdSignature(authSessionId, signature);
        }
        catch (Exception e) {
            log.errorf("Error decoding auth session id with value: %s", (Object)encodedBase64AuthSessionId, (Object)e);
            return null;
        }
    }

    private String validateAuthSessionIdSignature(String authSessionId, String signature) {
        if (signature.equals(this.session.getAttribute(authSessionId))) {
            return authSessionId;
        }
        SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, "HS512");
        SignatureSignerContext signer = signatureProvider.signer();
        try {
            boolean valid = signatureProvider.verifier(signer.getKid()).verify(authSessionId.getBytes(StandardCharsets.UTF_8), Base64Url.decode((String)signature));
            if (!valid) {
                return null;
            }
            this.session.setAttribute(authSessionId, (Object)signature);
            return authSessionId;
        }
        catch (Exception e) {
            log.errorf("Signature validation failed for auth session id: %s", (Object)authSessionId, (Object)e);
            return null;
        }
    }

    public String signAndEncodeToBase64AuthSessionId(String authSessionId) {
        SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, "HS512");
        SignatureSignerContext signer = signatureProvider.signer();
        StringBuilder buffer = new StringBuilder();
        byte[] signature = signer.sign(authSessionId.getBytes(StandardCharsets.UTF_8));
        buffer.append(authSessionId);
        if (signature != null) {
            buffer.append('.');
            buffer.append(Base64Url.encode((byte[])signature));
        }
        return Base64Url.encode((byte[])buffer.toString().getBytes(StandardCharsets.UTF_8));
    }

    AuthSessionCookie getAuthSessionCookies(RealmModel realm) {
        boolean routeChanged;
        String oldEncodedId = ((CookieProvider)this.session.getProvider(CookieProvider.class)).get(CookieType.AUTH_SESSION_ID);
        if (oldEncodedId == null || oldEncodedId.isEmpty()) {
            return null;
        }
        StickySessionEncoderProvider routeEncoder = (StickySessionEncoderProvider)this.session.getProvider(StickySessionEncoderProvider.class);
        StickySessionEncoderProvider.SessionIdAndRoute sessionIdAndRoute = routeEncoder.decodeSessionIdAndRoute(oldEncodedId);
        String decodedAuthSessionId = this.decodeBase64AndValidateSignature(sessionIdAndRoute.sessionId());
        if (decodedAuthSessionId == null) {
            return null;
        }
        RootAuthenticationSessionModel rootAuthenticationSession = this.session.authenticationSessions().getRootAuthenticationSession(realm, decodedAuthSessionId);
        if (rootAuthenticationSession == null) {
            return null;
        }
        String newRoute = routeEncoder.sessionIdRoute(decodedAuthSessionId);
        boolean bl = routeChanged = !sessionIdAndRoute.isSameRoute(newRoute);
        if (routeChanged) {
            log.debugf("Route changed. Will update authentication session cookie. Old: '%s', New: '%s'", (Object)sessionIdAndRoute.route(), (Object)newRoute);
        }
        return new AuthSessionCookie(rootAuthenticationSession, routeChanged);
    }

    public void removeAuthenticationSession(RealmModel realm, AuthenticationSessionModel authSession, boolean expireRestartCookie) {
        RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
        log.debugf("Removing root authSession '%s'. Expire restart cookie: %b", (Object)rootAuthSession.getId(), (Object)expireRestartCookie);
        this.session.authenticationSessions().removeRootAuthenticationSession(realm, rootAuthSession);
        if (expireRestartCookie) {
            RestartLoginCookie.expireRestartCookie(this.session);
            ((LoginFormsProvider)this.session.getProvider(LoginFormsProvider.class)).setDetachedAuthSession();
        }
    }

    public boolean removeTabIdInAuthenticationSession(RealmModel realm, AuthenticationSessionModel authSession) {
        RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
        rootAuthSession.removeAuthenticationSessionByTabId(authSession.getTabId());
        if (rootAuthSession.getAuthenticationSessions().isEmpty()) {
            this.removeAuthenticationSession(realm, authSession, true);
            return true;
        }
        return false;
    }

    public void updateAuthenticationSessionAfterSuccessfulAuthentication(RealmModel realm, AuthenticationSessionModel authSession) {
        boolean removedRootAuthSession = this.removeTabIdInAuthenticationSession(realm, authSession);
        if (removedRootAuthSession) {
            return;
        }
        if (realm.getSsoSessionIdleTimeout() < SessionExpiration.getAuthSessionLifespan((RealmModel)realm) && realm.getSsoSessionMaxLifespan() < SessionExpiration.getAuthSessionLifespan((RealmModel)realm)) {
            this.removeAuthenticationSession(realm, authSession, true);
            return;
        }
        RootAuthenticationSessionModel rootAuthSession = authSession.getParentSession();
        int authSessionExpiresIn = realm.getAccessCodeLifespan();
        int authSessionExpirationTime = Time.currentTime() - SessionExpiration.getAuthSessionLifespan((RealmModel)realm) + authSessionExpiresIn;
        rootAuthSession.setTimestamp(authSessionExpirationTime);
        log.tracef("Removed authentication session of root session '%s' with tabId '%s'. But there are remaining tabs in the root session. Root authentication session will expire in %d seconds", (Object)rootAuthSession.getId(), (Object)authSession.getTabId(), (Object)authSessionExpiresIn);
    }

    public UserSessionModel getUserSession(AuthenticationSessionModel authSession) {
        return this.getUserSessionProvider().getUserSession(authSession.getRealm(), authSession.getParentSession().getId());
    }

    public AuthenticationSessionModel getAuthenticationSessionByIdAndClient(RealmModel realm, String authSessionId, ClientModel client, String tabId) {
        RootAuthenticationSessionModel rootAuthSession = this.session.authenticationSessions().getRootAuthenticationSession(realm, authSessionId);
        return rootAuthSession == null ? null : rootAuthSession.getAuthenticationSession(client, tabId);
    }

    public AuthenticationSessionModel getAuthenticationSessionByEncodedIdAndClient(RealmModel realm, String encodedAuthSessionId, ClientModel client, String tabId) {
        String decodedAuthSessionId = this.decodeBase64AndValidateSignature(encodedAuthSessionId);
        return decodedAuthSessionId == null ? null : this.getAuthenticationSessionByIdAndClient(realm, decodedAuthSessionId, client, tabId);
    }

    public UserSessionModel getUserSessionFromAuthenticationCookie(RealmModel realm) {
        AuthSessionCookie rootAuth = this.getAuthSessionCookies(realm);
        if (rootAuth == null) {
            return this.getUserSessionFromIdentityCookie(realm);
        }
        UserSessionModel userSession = this.getUserSessionProvider().getUserSession(realm, rootAuth.sessionId());
        if (userSession != null) {
            this.reEncodeAuthSessionCookie(rootAuth);
        }
        return userSession;
    }

    private UserSessionModel getUserSessionFromIdentityCookie(RealmModel realm) {
        AuthenticationManager.AuthResult authResult = AuthenticationManager.authenticateIdentityCookie(this.session, realm, true);
        if (authResult == null) {
            return null;
        }
        assert (authResult.session() != null);
        this.setAuthSessionCookie(authResult.session().getId());
        return authResult.session();
    }

    private UserSessionProvider getUserSessionProvider() {
        return this.session.sessions();
    }

    record AuthSessionCookie(RootAuthenticationSessionModel rootSession, boolean routeChanged) {
        public String sessionId() {
            return this.rootSession.getId();
        }
    }
}

