#!/bin/sh
#
# Copyright (C) 2019 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#

#
# File: prepare-gpu-hardware-metrics
#
# Description: script add user into video group and set paranoid = 0
# for Enable GPU analysis for non-privileged users
#
# Version: 1.0
#

default_groups=
pci_devices="/sys/bus/pci/devices/*"
for device in $pci_devices
do
  drm_path="${device}/drm/"
  if [ -d "$drm_path" ] ; then
    i915_is_load=`readlink ${device}/driver | tail -c5 | grep i915`
    xe_is_load=`readlink ${device}/driver | tail -c5 | grep xe`
    if [ ! -z "${i915_is_load=}" ] || [ ! -z "${xe_is_load=}" ] ; then
      for driver_folder in $drm_path*
      do
        driver_interface_path=/dev/dri/$(basename $driver_folder)
        if [ -e "$driver_interface_path" ] ; then
          group=$(stat -c '%G' $driver_interface_path)
          case "$default_groups" in
            *$group*) ;;
            *       ) default_groups="$default_groups $group" ;;
          esac
        fi
      done
    fi
  fi
done
if [ ! -z "${default_groups}" ] ; then
default_groups=`echo ${default_groups} | cut -c 1-`
fi


default_user=
SCRIPT=$0
SCRIPT_ARGS="$@"
set_user=0
interactive=1
permanently=0
set_groups=0

print_usage_and_exit()
{
    err=${1:-0}
    echo ""
    echo "Usage: $0 [ option ]"
    echo ""
    echo " where \"option\" is one of the following:"
    echo ""
    echo "    -u | --user <user>"
    echo "      Specifies the user of the environment to be configured."
    echo ""
    echo "    -g | --group <group>"
    echo "      Use this option to specify groups other than "${default_groups}" to access video device."
    echo ""
    echo "    -i | --install"
    echo "      Sets /proc/sys/dev/i915/perf_stream_paranoid=0 permanently."
    echo ""
    echo "    -b | --batch"
    echo "      Run in non-interactive mode."
    echo ""
    echo " Without options, script will adds user into groups: ${default_groups} "
    echo " sets /proc/sys/dev/i915/perf_stream_paranoid value to 0 and "
    echo " sets /proc/sys/dev/xe/observation_paranoid to 0."
    exit $err
}

# help.
if [ $# -eq 1 ] ; then
    case "$1" in
        -h | --help)
            print_usage_and_exit 0
        ;;
    esac
fi

# parse the options
while [ $# -gt 0 ] ; do
    case "$1" in
        -g | --group)
            default_groups="$2"
            set_groups=1
            shift
        ;;
        -u | --user)
            default_user="$2"
            set_groups=1
            set_user=1
            shift
        ;;
        -b | --batch)
            interactive=0
        ;;
        -i | --install)
            permanently=1
        ;;
        *)
            echo ""
            echo "ERROR: unrecognized option $1"
            print_usage_and_exit 3
        ;;
    esac
    shift
done

# check if USER is root.
if [ -z "${BUSYBOX_SHELL}" ] ; then
    if [ "root" != "${USER}" ] ; then
        if [ ! -w /dev ] ; then
            if [ ${interactive} -eq 0 ] ; then
                echo "Error:"
                echo "Super-user or \"root\" privileges are required."
                exit 2
            fi
            echo "NOTE:  super-user or \"root\" privileges are required in order to continue."
            echo "Please enter \"root\" "
            exec su -c "/bin/sh ${SCRIPT} ${SCRIPT_ARGS}"
        fi
    fi
fi

add_to_group(){
  group=$1
  if [ ${interactive} -eq 1 ] ; then
      if [ ${set_user} -eq 0 ] ||  [ ${set_groups} -eq 0 ]  ; then
          echo "Do you want add user to \"${group}\" groups [y/n]"
          read answer
          if [ "$answer" = "y" ] && [ ${set_user} -eq 0 ] ; then
              echo "Enter the user name for whom you want to set permissions:"
              while read answer && [ -z "$answer" ];
              do
                  echo "ERROR: "
                  echo "The user name can not be empty."
              done
              default_user=${answer}
              set_user=1
          fi
      fi
  fi

  if [ ${set_user} -eq 1 ] ; then
      tmp=`id ${default_user} | grep  ${group}`
      if [ -z "$tmp" ] ; then
          if [ 'grep  ${group} /etc/group' ] ; then
              echo "Adding user: \"${default_user}\" into group: \"${group}\"."
              usermod -a -G ${group} ${default_user}
              if [ $? -ne 0 ]; then
                  echo "Failed to add user \"${default_user}\" to group \"${group}\"."
                  return 1
              fi
              echo "User \"${default_user}\" successfully added into group: \"${group}\"."
              echo "Re-login required"
          else
              echo "Group \"${group}\" does not exist"
          fi
      else
          echo "User is already member of group ${group}"
      fi
  fi
  return 0
}

user_added=0
for group in $default_groups
do
    add_to_group $group
done

check_i915=`sysctl -a 2> /dev/null | grep dev.i915.perf_stream_paranoid`
check_xe=`sysctl -a 2> /dev/null | grep dev.xe.observation_paranoid`
paranoid_i915="dev.i915.perf_stream_paranoid"
paranoid_xe="dev.xe.observation_paranoid"
if [ -z "$check_i915" ] && [ -z "$check_xe" ] ; then
    echo "error :"
    echo "dev.i915.perf_stream_paranoid and dev.xe.observation_paranoid not found"
    exit 2
fi

if [ ${interactive} -eq 1 ] && [ ${permanently} -eq 0 ] ; then 
    echo "Do you want set perf_stream_paranoid to 0?"
    while true
    do
        echo "(R)eject, set (t)emporarily or set (p)ermanently?"
        read answer
        if [ "$answer" = "t" ] ; then
            echo 'Selected: temporarily'
            if [ ! -z "$check_i915" ] ; then
                set `sysctl -w ${paranoid_i915}=0` via /etc/sysctl.d/
            fi
            if [ ! -z "$check_xe" ] ; then
                set `sysctl -w ${paranoid_xe}=0` via /etc/sysctl.d/
            fi
            exit 0
        fi
        if [ "$answer" = "p" ] ; then
            echo 'Selected: permanently'
            if [ ! -z "$check_i915" ] ; then
                tmp=`grep "${paranoid_i915}=" /etc/sysctl.conf`
                if  [ -z "$tmp" ] ; then
                    echo "${paranoid_i915}=0" >> /etc/sysctl.conf
                else
                    sed '/${paranoid_i915}/d' /etc/sysctl.conf > /etc/sysctl.conf_tmp
                    cat /etc/sysctl.conf_tmp > /etc/sysctl.conf
                    rm -f /etc/sysctl.conf_tmp
                    echo "${paranoid_i915}=0" >> /etc/sysctl.conf
                fi
                set `sysctl -w ${paranoid_i915}=0` via /etc/sysctl.d/
            fi
            if [ ! -z "$check_xe" ] ; then
                tmp=`grep "${paranoid_xe}=" /etc/sysctl.conf`
                if  [ -z "$tmp" ] ; then
                    echo "${paranoid_xe}=0" >> /etc/sysctl.conf
                else
                    sed '/${paranoid_xe}/d' /etc/sysctl.conf > /etc/sysctl.conf_tmp
                    cat /etc/sysctl.conf_tmp > /etc/sysctl.conf
                    rm -f /etc/sysctl.conf_tmp
                    echo "${paranoid_xe}=0" >> /etc/sysctl.conf
                fi
                set `sysctl -w ${paranoid_xe}=0` via /etc/sysctl.d/
            fi
            exit 0
        fi
        if [ "$answer" = "R" ] ;  then
            echo 'Selected: reject.'
            echo "The value of /proc/sys/dev/i915/perf_stream_paranoid and /proc/sys/dev/xe/observation_paranoid are not set."
            exit 0
        fi
        echo "Invalid value ${answer}"
    done
else
    if [ ${permanently} -eq 0 ] ; then
        if [ ! -z "$check_i915" ] ; then
            set `sysctl -w ${paranoid_i915}=0` via /etc/sysctl.d/
        fi
        if [ ! -z "$check_xe" ] ; then
            set `sysctl -w ${paranoid_xe}=0` via /etc/sysctl.d/
        fi
        exit 0
    fi
    if [ ${permanently} -eq 1 ] ;  then
        if [ ! -z "$check_i915" ] ; then
            tmp=`grep "${paranoid_i915}=" /etc/sysctl.conf`
            if  [ -z "$tmp" ] ; then
                echo "${paranoid_i915}=0" >> /etc/sysctl.conf
            else
                sed '/${paranoid_i915}/d' /etc/sysctl.conf > /etc/sysctl.conf_tmp
                cat /etc/sysctl.conf_tmp > /etc/sysctl.conf
                rm -f /etc/sysctl.conf_tmp
                echo "${paranoid_i915}=0" >> /etc/sysctl.conf
            fi
            set `sysctl -w ${paranoid_i915}=0` via /etc/sysctl.d/
        fi
        if [ ! -z "$check_xe" ] ; then
            tmp=`grep "${paranoid_xe}=" /etc/sysctl.conf`
            if  [ -z "$tmp" ] ; then
                echo "${paranoid_xe}=0" >> /etc/sysctl.conf
            else
                sed '/${paranoid_xe}/d' /etc/sysctl.conf > /etc/sysctl.conf_tmp
                cat /etc/sysctl.conf_tmp > /etc/sysctl.conf
                rm -f /etc/sysctl.conf_tmp
                echo "${paranoid_xe}=0" >> /etc/sysctl.conf
            fi
            set `sysctl -w ${paranoid_xe}=0` via /etc/sysctl.d/
        fi
        exit 0
    fi
fi
