/* Target interface for Intel GT based on level-zero for gdbserver.

   Copyright (C) 2020-2022 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include "server.h"
#include "ze-low.h"

#include "arch/intelgt.h"

#include <level_zero/zet_intel_gpu_debug.h>
#include <iomanip>
#include <sstream>


/* FIXME make into a target method?  */
int using_threads = 1;

/* Convenience macros.  */

#define dprintf(fmt, ...) \
  debug_prefixed_printf_cond (debug_threads, "ze-low", fmt, ##__VA_ARGS__)

/* Determine the most suitable type to be used for a register with bit size
   BITSIZE and element size ELEMSIZE.  */

static const char *
intelgt_uint_reg_type (tdesc_feature *feature, uint32_t bitsize,
		       uint32_t elemsize)
{
  if (0 != (bitsize % elemsize))
    error (_("unsupported combination of bitsize %" PRIu32 "and elemsize %"
	     PRIu32), bitsize, elemsize);
  if ((elemsize < 8) || (elemsize > 128) || ((elemsize & (elemsize - 1)) != 0))
    error (_("unsupported elemsize %" PRIu32), elemsize);

  char type_name[20];
  snprintf (type_name, sizeof (type_name), "uint%u", elemsize);
  tdesc_type *type = tdesc_named_type (feature, type_name);

  if (elemsize == bitsize)
    return type->name.c_str ();

  uint32_t elements = bitsize / elemsize;
  snprintf (type_name, sizeof (type_name), "vector%ux%u", elements,
	    elemsize);
  tdesc_type *vector
    = tdesc_create_vector (feature, type_name, type, elements);

  return vector->name.c_str ();
}

/* Add a (uniform) register set to FEATURE.  */

static void
intelgt_add_regset (tdesc_feature *feature, long &regnum, const char *prefix,
		    uint32_t count, const char *group, uint32_t bitsize,
		    bool is_writable, const char *type,
		    const std::initializer_list<unsigned int> &expedite = {})
{
  for (uint32_t reg = 0; reg < count; ++reg)
    {
      std::string name = std::string (prefix) + std::to_string (reg);

      bool is_expedited = (std::find (expedite.begin (), expedite.end (), reg)
			  != expedite.end ());

      tdesc_create_reg (feature, name.c_str (), regnum++, is_writable, group,
			bitsize, type, is_expedited);
    }
}

/* Control Register details.  */

enum
{
  /* The position of the Breakpoint Suppress bit in CR0.0.  */
  intelgt_cr0_0_breakpoint_suppress = 15,

  /* The position of the Breakpoint Status and Control bit in CR0.1.  */
  intelgt_cr0_1_breakpoint_status = 31,

  /* The position of the External Halt Status and Control bit in CR0.1.  */
  intelgt_cr0_1_external_halt_status = 30,

  /* The position of the Software Exception Control bit in CR0.1.  */
  intelgt_cr0_1_software_exception_control = 29,

  /* The position of the Illegal Opcode Exception Status bit in CR0.1.  */
  intelgt_cr0_1_illegal_opcode_status = 28,

  /* The position of the Force Exception Status and Control bit in CR0.1.  */
  intelgt_cr0_1_force_exception_status = 26,

  /* The position of the Page Fault Status bit in CR0.1.
     This is a software convention using a reserved bit to indicate
     page faults by the user mode driver.  */
  intelgt_cr0_1_pagefault_status = 16,
};

static uint32_t
get_device_id (const ze_device_info *device)
{
  gdb_assert (device != nullptr);
  return device->properties.deviceId;
}

/* Read DST.size () bytes from register REGNUM at OFFSET in REGCACHE and write
   to DST.  */

static void
intelgt_read_register_part (regcache *regcache, int regnum, int offset,
			    gdb::array_view<gdb_byte> dst)
{
  int regsize = register_size (regcache->tdesc, regnum);
  gdb_assert (offset >= 0);
  gdb_assert (offset + dst.size () <= regsize);

  collect_register_part (regcache, regnum, offset, dst);

  enum register_status status = regcache->get_register_status (regnum);
  switch (status)
    {
    case REG_VALID:
    case REG_DIRTY:
      return;

    case REG_UNKNOWN:
      internal_error (_("unknown register '%s'."),
		      register_name (regcache->tdesc, regnum));

    case REG_UNAVAILABLE:
      error (_("Register '%s' is not available"),
	     register_name (regcache->tdesc, regnum));
    }

  internal_error (_("unknown register status: %d."), status);
}

/* Write SRC.size () bytes to register REGNUM at OFFSET in REGCACHE.  */

static void
intelgt_write_register_part (regcache *regcache, int regnum, int offset,
			     gdb::array_view<const gdb_byte> src)
{
  int regsize = register_size (regcache->tdesc, regnum);
  gdb_assert (offset >= 0);
  gdb_assert (offset + src.size () <= regsize);

  std::vector<gdb_byte> buf (src.size ());
  collect_register_part (regcache, regnum, offset,
			 gdb::array_view<gdb_byte> (std::move (buf)));

  enum register_status status = regcache->get_register_status (regnum);
  switch (status)
    {
    case REG_VALID:
    case REG_DIRTY:
      supply_register_part (regcache, regnum, offset, src);
      return;

    case REG_UNKNOWN:
      internal_error (_("unknown register '%s'."),
		      register_name (regcache->tdesc, regnum));

    case REG_UNAVAILABLE:
      error (_("Register '%s' is not available"),
	     register_name (regcache->tdesc, regnum));
    }

  internal_error (_("unknown register status: %d."), status);
}

/* Return CR0.SUBREG in REGCACHE.  */

static uint32_t
intelgt_read_cr0 (regcache *regcache, int subreg)
{
  int regno = find_regno (regcache->tdesc, "cr0");

  uint32_t value = 0;
  gdb::array_view<gdb_byte> buf ((gdb_byte *) &value, sizeof (value));
  intelgt_read_register_part (regcache, regno, subreg * sizeof (value), buf);

  return value;
}

/* Write VALUE into CR0.SUBREG in REGCACHE.  */

static void
intelgt_write_cr0 (regcache *regcache, int subreg, uint32_t value)
{
  int regno = find_regno (regcache->tdesc, "cr0");
  gdb::array_view<const gdb_byte> buf ((const gdb_byte *) &value,
				       sizeof (value));
  intelgt_write_register_part (regcache, regno, subreg * sizeof (value), buf);
}

static unsigned int
intelgt_decode_tagged_address (CORE_ADDR addr)
{
  /* Generic pointers are tagged in order to preserve the address space to
     which they are pointing.  Tags are encoded into bits [61:63] of an
     address:

     000/111 - global,
     001 - private,
     010 - local (SLM)

     We currently cannot decode this tag in GDB, as the information
     cannot be added to the (cached) type instance flags, as it changes at
     runtime.  */
  if ((addr >> 61) == 0x2ul)
    return (unsigned int) ZET_DEBUG_MEMORY_SPACE_TYPE_SLM;

  return (unsigned int) ZET_DEBUG_MEMORY_SPACE_TYPE_DEFAULT;
}

static CORE_ADDR
intelgt_untag_address (CORE_ADDR addr)
{
  /* We need to clear tags from addresses before passing the read / write
     request to level-zero.  Copy bits [56:59] to [61:63] to preserve
     canonical form.  */
  constexpr CORE_ADDR clear_bit_mask = ~(0x7ull << 61);
  constexpr CORE_ADDR copy_from_bit_mask = (0x7ull << 56);

  return (addr & clear_bit_mask)
	  | ((addr & copy_from_bit_mask) << 5);
}

/* Return a human-readable device UUID string.  */

static std::string
device_uuid_str (const uint8_t uuid[], size_t size)
{
  std::stringstream sstream;
  for (int i = size - 1; i >= 0; --i)
    sstream << std::hex << std::setfill ('0') << std::setw (2)
	    << static_cast<int> (uuid[i]);

  return sstream.str ();
}

/* Target op definitions for Intel GT target based on level-zero.  */

class intelgt_ze_target : public ze_target
{
public:
  const gdb_byte *sw_breakpoint_from_kind (int kind, int *size) override;

  bool supports_stopped_by_sw_breakpoint () override { return true; }
  bool stopped_by_sw_breakpoint () override;
  bool supports_run_command () override;

  CORE_ADDR read_pc (regcache *regcache) override;
  void write_pc (regcache *regcache, CORE_ADDR pc) override;

protected:
  bool is_device_supported
    (const ze_device_properties_t &,
     const std::vector<zet_debug_regset_properties_t> &) override;

  target_desc *create_tdesc
    (ze_device_info *dinfo,
     const std::vector<zet_debug_regset_properties_t> &) override;

  target_stop_reason get_stop_reason (thread_info *, gdb_signal &) override;

  void prepare_thread_resume (thread_info *tp) override;

  /* Read one instruction from memory at PC into BUFFER and return the
     number of bytes read on success or a negative errno error code.

     BUFFER must be intelgt::MAX_INST_LENGTH bytes long.  */
  int read_inst (thread_info *tp, CORE_ADDR pc, unsigned char *buffer);

  bool is_at_breakpoint (thread_info *tp) override;
  bool is_at_eot (thread_info *tp);

  bool erratum_18020355813 (thread_info *tp);

  /* Read the memory in the context of thread TP.  */
  int read_memory (thread_info *tp, CORE_ADDR memaddr,
		   unsigned char *myaddr, int len,
		   unsigned int addr_space = 0) override;

  /* Write the memory in the context of thread TP.  */
  int write_memory (thread_info *tp, CORE_ADDR memaddr,
		    const unsigned char *myaddr, int len,
		    unsigned int addr_space = 0) override;

private:
  /* Add a register set for REGPROP on DEVICE to REGSETS and increment REGNUM
     accordingly.  */
  void add_regset (target_desc *tdesc, const ze_device_info &dinfo,
		   const zet_debug_regset_properties_t &regprop,
		   long &regnum, ze_regset_info_t &regsets);
};

bool
intelgt_ze_target::supports_run_command ()
{
  return false;
}

const gdb_byte *
intelgt_ze_target::sw_breakpoint_from_kind (int kind, int *size)
{
  /* We do not support breakpoint instructions.

     Use gdbarch methods that use read/write memory target operations for
     setting s/w breakopints.  */
  *size = 0;
  return nullptr;
}

bool
intelgt_ze_target::stopped_by_sw_breakpoint ()
{
  const ze_thread_info *zetp = ze_thread (current_thread);
  if (zetp == nullptr)
    return false;

  ptid_t ptid = current_thread->id;

  if (zetp->exec_state != ze_thread_state_stopped)
    {
      dprintf ("not-stopped thread %s", ptid.to_string ().c_str ());
      return false;
    }

  return (zetp->stop_reason == TARGET_STOPPED_BY_SW_BREAKPOINT);
}

CORE_ADDR
intelgt_ze_target::read_pc (regcache *regcache)
{
  uint32_t ip = intelgt_read_cr0 (regcache, 2);
  uint64_t isabase;
  collect_register_by_name (regcache, "isabase", &isabase);

  if (UINT32_MAX < ip)
    warning (_("IP '0x%" PRIx32 "' outside of ISA range."), ip);

  CORE_ADDR pc = (CORE_ADDR) isabase + (CORE_ADDR) ip;
  if (pc < isabase)
    warning (_("PC '%s' outside of ISA range."),
	     core_addr_to_string_nz (pc));

  return pc;
}

void
intelgt_ze_target::write_pc (regcache *regcache, CORE_ADDR pc)
{
  uint64_t isabase;
  collect_register_by_name (regcache, "isabase", &isabase);

  if (pc < isabase)
    error (_("PC '%s' outside of ISA range."), core_addr_to_string_nz (pc));

  pc -= isabase;
  if (UINT32_MAX < pc)
    error (_("PC '%s' outside of ISA range."), core_addr_to_string_nz (pc));

  intelgt_write_cr0 (regcache, 2, (uint32_t) pc);
}

bool
intelgt_ze_target::is_device_supported
  (const ze_device_properties_t &properties,
   const std::vector<zet_debug_regset_properties_t> &regset_properties)
{
  if (properties.type != ZE_DEVICE_TYPE_GPU)
    {
      dprintf ("non-gpu (%x) device (%" PRIx32 "): %s", properties.type,
	       properties.deviceId, properties.name);
      return false;
    }

  if (properties.vendorId != 0x8086)
    {
      dprintf ("unknown vendor (%" PRIx32 ") of device (%" PRIx32 "): %s",
	       properties.vendorId, properties.deviceId, properties.name);
      return false;
    }

  /* We need a few registers to support an Intel GT device.

     Those are registers that GDB itself uses.  Without those, we might run into
     internal errors at some point.  We need others, too, that may be referenced
     in debug information.  */
  bool have_grf = false;
  bool have_isabase = false;
  bool have_cr = false;
  bool have_sr = false;
  bool have_ce = false;
  bool have_sba = false;
  bool have_scrbase = false;
  bool have_modeflags = false;
  for (const zet_debug_regset_properties_t &regprop : regset_properties)
    {
      if (regprop.count < 1)
	{
	  warning (_("Ignoring empty regset %u in %s."), regprop.type,
		   properties.name);
	  continue;
	}

      switch (regprop.type)
	{
	case ZET_DEBUG_REGSET_TYPE_GRF_INTEL_GPU:
	  have_grf = true;
	  break;

	case ZET_DEBUG_REGSET_TYPE_CE_INTEL_GPU:
	  have_ce = true;
	  break;

	case ZET_DEBUG_REGSET_TYPE_CR_INTEL_GPU:
	  have_cr = true;
	  break;

	case ZET_DEBUG_REGSET_TYPE_SR_INTEL_GPU:
	  have_sr = true;
	  break;

	case ZET_DEBUG_REGSET_TYPE_SBA_INTEL_GPU:
	  have_sba = true;
	  /* We need 'isabase', which is at position 5 in version 1.  */
	  if ((regprop.version == 0) && (regprop.count >= 5))
	    have_isabase = true;
	  else
	    warning (_("Ignoring unknown SBA regset version %u in %s."),
		     regprop.version, properties.name);
	  break;

	case ZET_DEBUG_REGSET_TYPE_THREAD_SCRATCH_INTEL_GPU:
	  have_scrbase = true;
	  break;

	case ZET_DEBUG_REGSET_TYPE_MODE_FLAGS_INTEL_GPU:
	  have_modeflags = true;
	  break;
	}
    }

  if (have_sba && have_scrbase)
    {
      warning (_("Unsupported device (ID: %" PRIx32 "): "
		 "Cannot have both SBA and THREAD_SCRATCH register sets."),
		 properties.deviceId);
      return false;
    }

  if (have_grf && have_cr && have_sr && have_ce
      && (have_isabase || have_modeflags))
    return true;

  dprintf ("unsupported device (%" PRIx32 "): %s", properties.deviceId,
	   properties.name);
  return false;
}

target_desc *
intelgt_ze_target::create_tdesc
  (ze_device_info *dinfo,
   const std::vector<zet_debug_regset_properties_t> &regset_properties)
{
  const ze_device_properties_t &properties = dinfo->properties;

  if (properties.vendorId != 0x8086)
    error (_("unknown vendor (%" PRIx32 ") of device (%" PRIx32 "): %s"),
	   properties.vendorId, properties.deviceId, properties.name);

  target_desc_up tdesc = allocate_target_description ();
  set_tdesc_architecture (tdesc.get (), "intelgt");
  set_tdesc_osabi (tdesc.get (), GDB_OSABI_LINUX);

  std::string device_uuid = device_uuid_str (
    dinfo->properties.uuid.id, sizeof (dinfo->properties.uuid.id));
  const uint32_t total_cores = (properties.numSlices
				* properties.numSubslicesPerSlice
				* properties.numEUsPerSubslice);
  const uint32_t total_threads = (total_cores * properties.numThreadsPerEU);

  tdesc_device *device_info = new tdesc_device ();
  device_info->vendor_id = properties.vendorId;
  device_info->target_id = properties.deviceId;
  device_info->name = properties.name;
  device_info->pci_slot = dinfo->pci_slot;
  device_info->uuid = device_uuid;
  device_info->total_cores = total_cores;
  device_info->total_threads = total_threads;

  if (properties.flags & ZE_DEVICE_PROPERTY_FLAG_SUBDEVICE)
    device_info->subdevice_id = properties.subdeviceId;

  set_tdesc_device_info (tdesc.get (), device_info);

  long regnum = 0;
  ze_regset_info_up regset_info { new ze_regset_info_t };
  for (const zet_debug_regset_properties_t &regprop : regset_properties)
    add_regset (tdesc.get (), *dinfo, regprop, regnum,
		*regset_info);


  target_desc *tret = tdesc.get ();
  init_target_desc (tret, nullptr, GDB_OSABI_LINUX);

  /* Devices can have several tdescs.  One is created during attach to the
     device.  Others may be created during thread stop event
     handling as threads may have their own regset properties.  E.g. the number
     of GRF registers is dependent on the GRF mode.  */
  dinfo->tdesc_cache.add (regset_properties, std::move (tdesc),
			  std::move (regset_info));

  return tret;
}

target_stop_reason
intelgt_ze_target::get_stop_reason (thread_info *tp, gdb_signal &signal)
{
  ze_device_thread_t thread = ze_thread_id (tp);
  regcache *regcache = get_thread_regcache (tp, /* fetch = */ false);

  uint32_t cr0[3] = {
    intelgt_read_cr0 (regcache, 0),
    intelgt_read_cr0 (regcache, 1),
    intelgt_read_cr0 (regcache, 2)
  };

  dprintf ("thread %s (%s) stopped, cr0.0=%" PRIx32 ", .1=%" PRIx32
	   " [ %s%s%s%s%s%s], .2=%" PRIx32 ".", tp->id.to_string ().c_str (),
	   ze_thread_id_str (thread).c_str (), cr0[0], cr0[1],
	   (((cr0[1] & (1 << intelgt_cr0_1_breakpoint_status)) != 0)
	    ? "bp " : ""),
	   (((cr0[1] & (1 << intelgt_cr0_1_illegal_opcode_status)) != 0)
	    ? "ill " : ""),
	   (((cr0[1] & (1 << intelgt_cr0_1_force_exception_status)) != 0)
	    ? "fe " : ""),
	   (((cr0[1] & (1 << intelgt_cr0_1_software_exception_control)) != 0)
	    ? "sw " : ""),
	   (((cr0[1] & (1 << intelgt_cr0_1_external_halt_status)) != 0)
	    ? "eh " : ""),
	   (((cr0[1] & (1 << intelgt_cr0_1_pagefault_status)) != 0)
	    ? "pf " : ""),
	   cr0[2]);

  if ((cr0[1] & (1 << intelgt_cr0_1_pagefault_status)) != 0)
    {
      cr0[1] &= ~(1 << intelgt_cr0_1_pagefault_status);
      intelgt_write_cr0 (regcache, 1, cr0[1]);

      signal = GDB_SIGNAL_SEGV;
      return TARGET_STOPPED_BY_NO_REASON;
    }

  if ((cr0[1] & (1 << intelgt_cr0_1_breakpoint_status)) != 0)
    {
      cr0[1] &= ~(1 << intelgt_cr0_1_breakpoint_status);
      intelgt_write_cr0 (regcache, 1, cr0[1]);

      /* We cannot distinguish a single step exception from a breakpoint
	 exception just by looking at CR0.

	 We could inspect the instruction to see if the breakpoint bit is
	 set.  Or we could check the resume type and assume that we set
	 things up correctly for single-stepping before we resumed.  */
      const ze_thread_info *zetp = ze_thread (tp);
      gdb_assert (zetp != nullptr);

      switch (zetp->resume_state)
	{
	case ze_thread_resume_step:
	  signal = GDB_SIGNAL_TRAP;
	  return TARGET_STOPPED_BY_SINGLE_STEP;

	case ze_thread_resume_run:
	case ze_thread_resume_none:
	  /* On some devices, we may get spurious breakpoint exceptions.  */
	  if (erratum_18020355813 (tp))
	    {
	      ze_device_thread_t zeid = ze_thread_id (tp);

	      dprintf ("applying #18020355813 workaround for thread "
		       "%s (%s)", tp->id.to_string ().c_str (),
		       ze_thread_id_str (zeid).c_str ());

	      signal = GDB_SIGNAL_0;
	      return TARGET_STOPPED_BY_NO_REASON;
	    }

	  [[fallthrough]];
	case ze_thread_resume_stop:
	  signal = GDB_SIGNAL_TRAP;
	  return TARGET_STOPPED_BY_SW_BREAKPOINT;
	}
    }

  if ((cr0[1] & (1 << intelgt_cr0_1_illegal_opcode_status)) != 0)
    {
      cr0[1] &= ~(1 << intelgt_cr0_1_illegal_opcode_status);
      intelgt_write_cr0 (regcache, 1, cr0[1]);

      signal = GDB_SIGNAL_ILL;
      return TARGET_STOPPED_BY_NO_REASON;
    }

  if ((cr0[1] & (1 << intelgt_cr0_1_software_exception_control)) != 0)
    {
      cr0[1] &= ~(1 << intelgt_cr0_1_software_exception_control);
      intelgt_write_cr0 (regcache, 1, cr0[1]);

      signal = GDB_EXC_SOFTWARE;
      return TARGET_STOPPED_BY_NO_REASON;
    }

  if ((cr0[1] & ((1 << intelgt_cr0_1_force_exception_status)
		 | (1 << intelgt_cr0_1_external_halt_status))) != 0)
    {
      cr0[1] &= ~(1 << intelgt_cr0_1_force_exception_status);
      cr0[1] &= ~(1 << intelgt_cr0_1_external_halt_status);
      intelgt_write_cr0 (regcache, 1, cr0[1]);

      signal = GDB_SIGNAL_TRAP;
      return TARGET_STOPPED_BY_NO_REASON;
    }

  signal = GDB_SIGNAL_UNKNOWN;
  return TARGET_STOPPED_BY_NO_REASON;
}

int
intelgt_ze_target::read_memory (thread_info *tp, CORE_ADDR memaddr,
				unsigned char *myaddr, int len,
				unsigned int addr_space)
{
  if (addr_space == (unsigned int) ZET_DEBUG_MEMORY_SPACE_TYPE_DEFAULT)
    addr_space = intelgt_decode_tagged_address (memaddr);

  memaddr = intelgt_untag_address (memaddr);

  return ze_target::read_memory (tp, memaddr, myaddr, len, addr_space);
}

int
intelgt_ze_target::write_memory (thread_info *tp, CORE_ADDR memaddr,
				 const unsigned char *myaddr, int len,
				 unsigned int addr_space)
{
  if (addr_space == (unsigned int) ZET_DEBUG_MEMORY_SPACE_TYPE_DEFAULT)
    addr_space = intelgt_decode_tagged_address (memaddr);

  memaddr = intelgt_untag_address (memaddr);

  return ze_target::write_memory (tp, memaddr, myaddr, len, addr_space);
}

int
intelgt_ze_target::read_inst (thread_info *tp, CORE_ADDR pc,
			      unsigned char *buffer)
{
  int status = read_memory (tp, pc, buffer, intelgt::MAX_INST_LENGTH);
  if (status == 0)
    return intelgt::MAX_INST_LENGTH;

  status = read_memory (tp, pc, buffer, intelgt::COMPACT_INST_LENGTH);
  if (status > 0)
    return status;

  uint32_t device_id = get_device_id (ze_thread_device (tp));
  if (intelgt::inst_length (buffer, device_id) == intelgt::MAX_INST_LENGTH)
    return -EIO;

  memset (buffer + intelgt::COMPACT_INST_LENGTH, 0,
	  intelgt::MAX_INST_LENGTH - intelgt::COMPACT_INST_LENGTH);

  return intelgt::COMPACT_INST_LENGTH;
}

bool
intelgt_ze_target::is_at_breakpoint (thread_info *tp)
{
  regcache *regcache = get_thread_regcache (tp, /* fetch = */ false);
  CORE_ADDR pc = read_pc (regcache);

  gdb_byte inst[intelgt::MAX_INST_LENGTH];
  int status = read_inst (tp, pc, inst);
  if (status < 0)
    return false;

  uint32_t device_id = get_device_id (ze_thread_device (tp));
  return intelgt::has_breakpoint (inst, device_id);
}

bool
intelgt_ze_target::is_at_eot (thread_info *tp)
{
  regcache *regcache = get_thread_regcache (tp, /* fetch = */ false);
  CORE_ADDR pc = read_pc (regcache);

  gdb_byte inst[intelgt::MAX_INST_LENGTH];
  int status = read_inst (tp, pc, inst);
  if (status < 0)
    {
      ze_device_thread_t zeid = ze_thread_id (tp);

      warning (_("error reading memory for thread %s (%s) at 0x%"
		 PRIx64), tp->id.to_string ().c_str (),
	       ze_thread_id_str (zeid).c_str (), pc);
      return false;
    }

  uint32_t device_id = get_device_id (ze_thread_device (tp));
  intelgt::xe_version device_version = intelgt::get_xe_version (device_id);
  switch (device_version)
    {
    case intelgt::XE_HP:
    case intelgt::XE_HPG:
    case intelgt::XE_HPC:
    case intelgt::XE2:
    case intelgt::XE3:
      {
	/* The opcode mask for bits 6:0.  */
	constexpr uint8_t OPC_MASK = 0x7f;
	switch (inst[0] & OPC_MASK)
	  {
	  case 0x31: /* send */
	  case 0x32: /* sendc */
	    {
	      /* The End Of Thread control.  Only used for SEND and
		 SENDC.  */
	      constexpr uint8_t CTRL_EOT_SEND = 34;
	      return intelgt::get_inst_bit (inst, CTRL_EOT_SEND);
	    }

	  default:
	    return false;
	  }
      }

    case intelgt::XE_INVALID:
      break;
    }

  error (_("Unsupported device id 0x%" PRIx32), device_id);
}

/* Return whether erratum #18020355813 applies.  */

bool
intelgt_ze_target::erratum_18020355813 (thread_info *tp)
{
  ze_device_info *device = ze_thread_device (tp);

  /* We may not have a device if we got detached.  */
  if (device == nullptr)
    return false;

  /* The erratum only applies to Intel devices.  */
  if (device->properties.vendorId != 0x8086)
    return false;

  uint32_t device_id = get_device_id (device);

  /* The erratum only applies to a range of devices.  */
  switch (intelgt::get_xe_version (device_id))
    {
      case intelgt::XE_HPG:
      case intelgt::XE_HPC:
	break;

      default:
	return false;
    }

  regcache *regcache = get_thread_regcache (tp, /* fetch = */ false);
  CORE_ADDR pc = read_pc (regcache);

  gdb_byte inst[intelgt::MAX_INST_LENGTH];
  int status = read_inst (tp, pc, inst);
  if (status < 0)
    {
      ze_device_thread_t zeid = ze_thread_id (tp);

      warning (_("error reading memory for thread %s (%s) at 0x%"
		 PRIx64), tp->id.to_string ().c_str (),
	       ze_thread_id_str (zeid).c_str (), pc);
      return false;
    }

  /* The erratum applies to instructions without breakpoint control.  */
  return !intelgt::has_breakpoint (inst, device_id);
}

void
intelgt_ze_target::prepare_thread_resume (thread_info *tp)
{
  ze_thread_info *zetp = ze_thread (tp);
  gdb_assert (zetp != nullptr);

  regcache *regcache = get_thread_regcache (tp, /* fetch = */ false);
  uint32_t cr0[3] = {
    intelgt_read_cr0 (regcache, 0),
    intelgt_read_cr0 (regcache, 1),
    intelgt_read_cr0 (regcache, 2)
  };

  /* The thread is running.  We may need to overwrite this below.  */
  zetp->exec_state = ze_thread_state_running;

  /* Clear any potential interrupt indication.

     We leave other exception indications so the exception would be
     reported again and can be handled by GDB.  */
  cr0[1] &= ~(1 << intelgt_cr0_1_force_exception_status);
  cr0[1] &= ~(1 << intelgt_cr0_1_external_halt_status);

  /* Distinguish stepping and continuing.  */
  switch (zetp->resume_state)
    {
    case ze_thread_resume_step:
      /* We step by indicating a breakpoint exception, which will be
	 considered on the next instruction.

	 This does not work for EOT, though.  */
      if (!is_at_eot (tp))
	{
	  cr0[0] |= (1 << intelgt_cr0_0_breakpoint_suppress);
	  cr0[1] |= (1 << intelgt_cr0_1_breakpoint_status);
	  break;
	}

      /* At EOT, the thread dispatch ends and the thread becomes idle.

	 There's no point in requesting a single-step exception but we
	 need to inject an event to tell GDB that the step completed.  */
      zetp->exec_state = ze_thread_state_unavailable;
      zetp->waitstatus.set_unavailable ();

      [[fallthrough]];
    case ze_thread_resume_run:
      cr0[1] &= ~(1 << intelgt_cr0_1_breakpoint_status);
      break;

    default:
      internal_error (_("bad resume kind: %d."), zetp->resume_state);
    }

  /* When stepping over a breakpoint, we need to suppress the breakpoint
     exception we would otherwise get immediately.

     This requires breakpoints to be already inserted when this function
     is called.  It also handles permanent breakpoints.  */
  if (is_at_breakpoint (tp))
    cr0[0] |= (1 << intelgt_cr0_0_breakpoint_suppress);

  intelgt_write_cr0 (regcache, 0, cr0[0]);
  intelgt_write_cr0 (regcache, 1, cr0[1]);
  intelgt_write_cr0 (regcache, 2, cr0[2]);

  dprintf ("thread %s (%s) resumed, cr0.0=%" PRIx32 " .1=%" PRIx32
	   " .2=%" PRIx32 ".", tp->id.to_string ().c_str (),
	   ze_thread_id_str (zetp->id).c_str (), cr0[0], cr0[1], cr0[2]);
}

void
intelgt_ze_target::add_regset (target_desc *tdesc, const ze_device_info &dinfo,
			       const zet_debug_regset_properties_t &regprop,
			       long &regnum, ze_regset_info_t &regsets)
{
  tdesc_feature *feature = nullptr;
  const ze_device_properties_t &device = dinfo.properties;

  ze_regset_info regset = {};
  regset.type = (uint32_t) regprop.type;
  regset.size = regprop.byteSize;
  regset.begin = regnum;
  regset.is_writeable
    = ((regprop.generalFlags & ZET_DEBUG_REGSET_FLAG_WRITEABLE) != 0);

  if (regprop.count < 1)
    {
      warning (_("Ignoring empty regset %u in %s."), regprop.type,
	       device.name);
      return;
    }

  if ((regprop.generalFlags & ZET_DEBUG_REGSET_FLAG_READABLE) == 0)
    {
      warning (_("Ignoring non-readable regset %u in %s."), regprop.type,
	       device.name);
      return;
    }

  if (regprop.bitSize == 0)
    {
      warning (_("Ignoring regset %u in %s with 0 bitsize."),
	       regprop.type, device.name);
      return;
    }

  switch (regprop.type)
    {
    case ZET_DEBUG_REGSET_TYPE_GRF_INTEL_GPU:
      {
	intelgt::xe_version device_version
	  = intelgt::get_xe_version (device.deviceId);

	switch (device_version)
	  {
	  case intelgt::XE_HP:
	  case intelgt::XE_HPG:
	  case intelgt::XE_HPC:
	  case intelgt::XE2:
	  case intelgt::XE3:
	    feature = tdesc_create_feature (tdesc, intelgt::feature_grf);

	    intelgt_add_regset (feature, regnum, "r", regprop.count, "grf",
				regprop.bitSize, regset.is_writeable,
				intelgt_uint_reg_type (feature,
						       regprop.bitSize, 32u),
				{ 0, regprop.count - 1 });
	    break;

	  default:
	    gdb_assert_not_reached ("Unexpected device id 0x%" PRIx32,
				    device.deviceId);
	  }
	break;
      }

    case ZET_DEBUG_REGSET_TYPE_ADDR_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_addr);

      intelgt_add_regset (feature, regnum, "a", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 16u));
      break;

    case ZET_DEBUG_REGSET_TYPE_FLAG_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_flag);

      intelgt_add_regset (feature, regnum, "f", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 16u));
      break;

    case ZET_DEBUG_REGSET_TYPE_CE_INTEL_GPU:
      /* We expect a single 'ce' register.  */
      if (regprop.count != 1)
	warning (_("Ignoring %u unexpected 'ce' registers in %s."),
		 regprop.count - 1, device.name);

      feature = tdesc_create_feature (tdesc, intelgt::feature_ce);

      tdesc_create_reg (feature, "ce", regnum++, regset.is_writeable, "arf",
			regprop.bitSize,
			intelgt_uint_reg_type (feature, regprop.bitSize,
					       32u),
			true /* expedited */);
      break;

    case ZET_DEBUG_REGSET_TYPE_SR_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_sr);

      intelgt_add_regset (feature, regnum, "sr", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u),
			  { 0 });
      break;

    case ZET_DEBUG_REGSET_TYPE_CR_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_cr);

      intelgt_add_regset (feature, regnum, "cr", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u),
			  { 0 });
      break;

    case ZET_DEBUG_REGSET_TYPE_TDR_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_tdr);

      intelgt_add_regset (feature, regnum, "tdr", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 16u));
      break;

    case ZET_DEBUG_REGSET_TYPE_ACC_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_acc);

      intelgt_add_regset (feature, regnum, "acc", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u));
      break;

    case ZET_DEBUG_REGSET_TYPE_MME_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_mme);

      intelgt_add_regset (feature, regnum, "mme", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u));
      break;

    case ZET_DEBUG_REGSET_TYPE_SP_INTEL_GPU:
      /* We expect a single 'sp' register.  */
      if (regprop.count != 1)
	warning (_("Ignoring %u unexpected 'sp' registers in %s."),
		 regprop.count - 1, device.name);

      feature = tdesc_create_feature (tdesc, intelgt::feature_sp);

      tdesc_create_reg (feature, "sp", regnum++, regset.is_writeable, "arf",
			regprop.bitSize,
			intelgt_uint_reg_type (feature, regprop.bitSize,
					       regprop.bitSize));
      break;

    case ZET_DEBUG_REGSET_TYPE_SBA_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_sba);

      switch (regprop.version)
	{
	case 0:
	  {
	    const char *regtype = intelgt_uint_reg_type (feature,
							 regprop.bitSize,
							 regprop.bitSize);
	    const char *sbaregs[] = {
	      "genstbase",
	      "sustbase",
	      "dynbase",
	      "iobase",
	      "isabase",
	      "blsustbase",
	      "blsastbase",
	      "btbase",
	      "scrbase0",
	      "scrbase1",
	      nullptr
	    };
	    int reg = 0;
	    for (; (reg < regprop.count) && (sbaregs[reg] != nullptr); ++reg)
	      {
		bool is_expedited = false;
		if ((strcmp (sbaregs[reg], "genstbase") == 0)
		    || (strcmp (sbaregs[reg], "isabase") == 0))
		  is_expedited = true;

		tdesc_create_reg (feature, sbaregs[reg], regnum++,
				  regset.is_writeable, "virtual",
				  regprop.bitSize, regtype, is_expedited);
	      }
	  }
	  break;

	default:
	  warning (_("Ignoring unknown SBA regset version %u in %s"),
		   regprop.version, device.name);
	  break;
	}
      break;

    case ZET_DEBUG_REGSET_TYPE_DBG_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_dbg);

      intelgt_add_regset (feature, regnum, "dbg", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u));
      break;

    case ZET_DEBUG_REGSET_TYPE_FC_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_fc);

      intelgt_add_regset (feature, regnum, "fc", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u));
      break;

    case ZET_DEBUG_REGSET_TYPE_MSG_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_msg);

      intelgt_add_regset (feature, regnum, "msg", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 32u));
      break;

    case ZET_DEBUG_REGSET_TYPE_MODE_FLAGS_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_mf);

      intelgt_add_regset (feature, regnum, "mf", regprop.count, "virtual",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 regprop.bitSize),
			  { 0 });
      break;

    case ZET_DEBUG_REGSET_TYPE_DEBUG_SCRATCH_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_debugger);

      switch (regprop.version)
	{
	case 0:
	  {
	    const char *regtype = intelgt_uint_reg_type (feature,
							 regprop.bitSize,
							 regprop.bitSize);
	    const char *debugregs[] = {
	      "dbgscrbase",
	      "dbgscrsize",
	      nullptr
	    };
	    int reg = 0;
	    for (; (reg < regprop.count) && (debugregs[reg] != nullptr); ++reg)
	      {
		tdesc_create_reg (feature, debugregs[reg], regnum++,
				  regset.is_writeable, "virtual",
				  regprop.bitSize, regtype);
	      }
	  }
	  break;

	default:
	  warning (_("Ignoring unknown DEBUG_SCRATCH regset version %u in %s"),
		   regprop.version, device.name);
	  break;
	}
      break;

    case ZET_DEBUG_REGSET_TYPE_THREAD_SCRATCH_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_scratch);

      intelgt_add_regset (feature, regnum, "scrbase", regprop.count, "virtual",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize,
						 regprop.bitSize));
      break;

    case ZET_DEBUG_REGSET_TYPE_SCALAR_INTEL_GPU:
      feature = tdesc_create_feature (tdesc, intelgt::feature_scalar);

      intelgt_add_regset (feature, regnum, "s", regprop.count, "arf",
			  regprop.bitSize, regset.is_writeable,
			  intelgt_uint_reg_type (feature, regprop.bitSize, 8u));
      break;

    case ZET_DEBUG_REGSET_TYPE_INVALID_INTEL_GPU:
    case ZET_DEBUG_REGSET_TYPE_FORCE_UINT32:
      break;
    }

  if (feature == nullptr)
    {
      warning (_("Ignoring unknown regset %u in %s."), regprop.type,
	       device.name);

      return;
    }

  regset.end = regnum;
  regsets.push_back (regset);
}


/* The Intel GT target ops object.  */

static intelgt_ze_target the_intelgt_ze_target;

extern void initialize_low ();
void
initialize_low ()
{
  /* Delayed initialization of level-zero targets.  See ze-low.h.  */
  the_intelgt_ze_target.init ();
  set_target_ops (&the_intelgt_ze_target);
}
