/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.commons.reflect;

import static org.apache.juneau.commons.utils.AssertionUtils.*;
import static org.apache.juneau.commons.utils.ThrowableUtils.*;
import static org.apache.juneau.commons.utils.Utils.*;

import java.beans.*;
import java.lang.annotation.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.function.*;
import java.util.stream.*;

import org.apache.juneau.commons.utils.*;

/**
 * Lightweight utility class for introspecting information about a Java method.
 *
 * <p>
 * This class provides a convenient wrapper around {@link Method} that extends the standard Java reflection
 * API with additional functionality for method introspection, annotation handling, and hierarchy traversal.
 * It's designed to be lightweight, thread-safe, and cached for efficient reuse.
 *
 * <h5 class='section'>Features:</h5>
 * <ul class='spaced-list'>
 * 	<li>Method introspection - access method metadata, parameters, return type, exceptions
 * 	<li>Annotation support - get annotations from method and overridden methods in hierarchy
 * 	<li>Hierarchy traversal - find matching methods in parent classes and interfaces
 * 	<li>Type-safe access - wrapper around reflection with convenient methods
 * 	<li>Thread-safe - instances are immutable and safe for concurrent access
 * </ul>
 *
 * <h5 class='section'>Use Cases:</h5>
 * <ul class='spaced-list'>
 * 	<li>Introspecting method metadata for code generation or analysis
 * 	<li>Finding annotations on methods including those from parent classes
 * 	<li>Discovering method hierarchies and overridden methods
 * 	<li>Working with method parameters and return types
 * 	<li>Building frameworks that need to analyze method signatures
 * </ul>
 *
 * <h5 class='section'>Usage:</h5>
 * <p class='bjava'>
 * 	<jc>// Get MethodInfo from a class</jc>
 * 	ClassInfo <jv>ci</jv> = ClassInfo.<jsm>of</jsm>(MyClass.<jk>class</jk>);
 * 	MethodInfo <jv>method</jv> = <jv>ci</jv>.getMethod(<js>"myMethod"</js>);
 *
 * 	<jc>// Get return type</jc>
 * 	ClassInfo <jv>returnType</jv> = <jv>method</jv>.getReturnType();
 *
 * 	<jc>// Get annotations including from parent methods</jc>
 * 	List&lt;AnnotationInfo&lt;MyAnnotation&gt;&gt; <jv>annotations</jv> =
 * 		<jv>method</jv>.getAnnotations(MyAnnotation.<jk>class</jk>).toList();
 *
 * 	<jc>// Find matching methods in hierarchy</jc>
 * 	List&lt;MethodInfo&gt; <jv>matching</jv> = <jv>method</jv>.getMatchingMethods();
 * </p>
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='jc'>{@link ClassInfo} - Class introspection
 * 	<li class='jc'>{@link FieldInfo} - Field introspection
 * 	<li class='jc'>{@link ConstructorInfo} - Constructor introspection
 * 	<li class='jc'>{@link ParameterInfo} - Parameter introspection
 * 	<li class='link'><a class="doclink" href="https://juneau.apache.org/docs/topics/JuneauCommonsReflection">Reflection Package</a>
 * </ul>
 */
public class MethodInfo extends ExecutableInfo implements Comparable<MethodInfo>, Annotatable {
	/**
	 * Creates a MethodInfo wrapper for the specified method.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	Method <jv>m</jv> = MyClass.<jk>class</jk>.getMethod(<js>"myMethod"</js>);
	 * 	MethodInfo <jv>mi</jv> = MethodInfo.<jsm>of</jsm>(MyClass.<jk>class</jk>, <jv>m</jv>);
	 * </p>
	 *
	 * @param declaringClass The class that declares this method. Must not be <jk>null</jk>.
	 * @param inner The method being wrapped. Must not be <jk>null</jk>.
	 * @return A new MethodInfo object wrapping the method.
	 */
	public static MethodInfo of(Class<?> declaringClass, Method inner) {
		return ClassInfo.of(declaringClass).getMethod(inner);
	}

	/**
	 * Creates a MethodInfo wrapper for the specified method.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	ClassInfo <jv>ci</jv> = ClassInfo.<jsm>of</jsm>(MyClass.<jk>class</jk>);
	 * 	Method <jv>m</jv> = MyClass.<jk>class</jk>.getMethod(<js>"myMethod"</js>);
	 * 	MethodInfo <jv>mi</jv> = MethodInfo.<jsm>of</jsm>(<jv>ci</jv>, <jv>m</jv>);
	 * </p>
	 *
	 * @param declaringClass The ClassInfo for the class that declares this method. Must not be <jk>null</jk>.
	 * @param inner The method being wrapped. Must not be <jk>null</jk>.
	 * @return A new MethodInfo object wrapping the method.
	 */
	public static MethodInfo of(ClassInfo declaringClass, Method inner) {
		assertArgNotNull("declaringClass", declaringClass);
		return declaringClass.getMethod(inner);
	}

	/**
	 * Creates a MethodInfo wrapper for the specified method.
	 *
	 * <p>
	 * This convenience method automatically determines the declaring class from the method.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	Method <jv>m</jv> = MyClass.<jk>class</jk>.getMethod(<js>"myMethod"</js>);
	 * 	MethodInfo <jv>mi</jv> = MethodInfo.<jsm>of</jsm>(<jv>m</jv>);
	 * </p>
	 *
	 * @param inner The method being wrapped. Must not be <jk>null</jk>.
	 * @return A new MethodInfo object wrapping the method.
	 */
	public static MethodInfo of(Method inner) {
		assertArgNotNull("inner", inner);
		return ClassInfo.of(inner.getDeclaringClass()).getMethod(inner);
	}

	private final Method inner;
	private final Supplier<ClassInfo> returnType;
	private final Supplier<List<MethodInfo>> matchingMethods;
	private final Supplier<List<AnnotationInfo<Annotation>>> annotations;

	/**
	 * Constructor.
	 *
	 * <p>
	 * Creates a new MethodInfo wrapper for the specified method. This constructor is protected
	 * and should not be called directly. Use the static factory methods {@link #of(Method)} or
	 * obtain MethodInfo instances from {@link ClassInfo#getMethod(Method)}.
	 *
	 * @param declaringClass The ClassInfo for the class that declares this method.
	 * @param inner The method being wrapped.
	 */
	protected MethodInfo(ClassInfo declaringClass, Method inner) {
		super(declaringClass, inner);
		this.inner = inner;
		this.returnType = mem(() -> ClassInfo.of(inner.getReturnType(), inner.getGenericReturnType()));
		this.matchingMethods = mem(this::findMatchingMethods);
		this.annotations = mem(() -> getMatchingMethods().stream().flatMap(m -> m.getDeclaredAnnotations().stream()).toList());
	}

	@Override /* Overridden from ExecutableInfo */
	public MethodInfo accessible() {
		super.accessible();
		return this;
	}

	@Override
	public int compareTo(MethodInfo o) {
		int i = cmp(getSimpleName(), o.getSimpleName());
		if (i == 0) {
			var params = getParameters();
			var oParams = o.getParameters();
			i = params.size() - oParams.size();
			if (i == 0) {
				for (var j = 0; j < params.size() && i == 0; j++) {
					i = cmp(params.get(j).getParameterType().getName(), oParams.get(j).getParameterType().getName());
				}
			}
		}
		return i;
	}

	@Override /* Annotatable */
	public AnnotatableType getAnnotatableType() { return AnnotatableType.METHOD_TYPE; }

	/**
	 * Returns an {@link AnnotatedType} object that represents the use of a type to specify the return type of the method.
	 *
	 * <p>
	 * Same as calling {@link Method#getAnnotatedReturnType()}.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	<jc>// For method: public @NotNull String getName()</jc>
	 * 	MethodInfo <jv>mi</jv> = ClassInfo.<jsm>of</jsm>(MyClass.<jk>class</jk>).getMethod(<js>"getName"</js>);
	 * 	AnnotatedType <jv>aType</jv> = <jv>mi</jv>.getAnnotatedReturnType();
	 * 	<jc>// Check for @NotNull on the return type</jc>
	 * </p>
	 *
	 * @return An {@link AnnotatedType} object representing the return type.
	 * @see Method#getAnnotatedReturnType()
	 */
	public AnnotatedType getAnnotatedReturnType() { return inner.getAnnotatedReturnType(); }

	/**
	 * Returns all annotations on this method and parent overridden methods in child-to-parent order.
	 *
	 * <p>
	 * Results include annotations from:
	 * <ul>
	 * 	<li>This method
	 * 	<li>Matching methods in parent classes
	 * 	<li>Matching methods in interfaces
	 * </ul>
	 *
	 * <p>
	 * <b>Note on Repeatable Annotations:</b>
	 * Repeatable annotations (those marked with {@link java.lang.annotation.Repeatable @Repeatable}) are automatically
	 * expanded into their individual annotation instances. For example, if a method has multiple {@code @Bean} annotations,
	 * this method returns each {@code @Bean} annotation separately, rather than the container annotation.
	 *
	 * <p>
	 * List is unmodifiable.
	 *
	 * @return
	 * 	A list of all annotations on this method and overridden methods.
	 * 	<br>Repeatable annotations are expanded into individual instances.
	 */
	public List<AnnotationInfo<Annotation>> getAnnotations() { return annotations.get(); }

	/**
	 * Returns all annotations of the specified type on this method and parent overridden methods in child-to-parent order.
	 *
	 * <p>
	 * Results include annotations from:
	 * <ul>
	 * 	<li>This method
	 * 	<li>Matching methods in parent classes
	 * 	<li>Matching methods in interfaces
	 * </ul>
	 *
	 * <p>
	 * <b>Note on Repeatable Annotations:</b>
	 * If the specified annotation type is repeatable (marked with {@link java.lang.annotation.Repeatable @Repeatable}),
	 * this method automatically expands container annotations into individual instances. This allows you to filter for
	 * a repeatable annotation and get back all individual occurrences without manually handling the container.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	<jc>// Get all @Bean annotations on this method and overridden methods</jc>
	 * 	Stream&lt;AnnotationInfo&lt;Bean&gt;&gt; <jv>beans</jv> = <jv>methodInfo</jv>.getAnnotations(Bean.<jk>class</jk>);
	 * 	<jc>// If method has @Beans({@Bean(...), @Bean(...)}), both individual @Bean instances are returned</jc>
	 * </p>
	 *
	 * @param <A> The annotation type.
	 * @param type The annotation type to filter by.
	 * @return
	 * 	A stream of matching annotation infos in child-to-parent order.
	 * 	<br>Repeatable annotations are expanded into individual instances.
	 */
	@SuppressWarnings("unchecked")
	public <A extends Annotation> Stream<AnnotationInfo<A>> getAnnotations(Class<A> type) {
		assertArgNotNull("type", type);
		return getAnnotations().stream().filter(a -> a.isType(type)).map(a -> (AnnotationInfo<A>)a);
	}

	/**
	 * Returns the default value for the annotation member represented by this method.
	 *
	 * <p>
	 * Same as calling {@link Method#getDefaultValue()}.
	 *
	 * <p>
	 * Returns <jk>null</jk> if this method is not an annotation member, or if the annotation member has no default value.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	<jc>// For annotation: @interface MyAnnotation { String value() default "default"; }</jc>
	 * 	MethodInfo <jv>mi</jv> = ClassInfo.<jsm>of</jsm>(MyAnnotation.<jk>class</jk>).getMethod(<js>"value"</js>);
	 * 	Object <jv>defaultValue</jv> = <jv>mi</jv>.getDefaultValue();
	 * 	<jc>// defaultValue is "default"</jc>
	 * </p>
	 *
	 * @return The default value, or <jk>null</jk> if none.
	 * @see Method#getDefaultValue()
	 */
	public Object getDefaultValue() { return inner.getDefaultValue(); }

	/**
	 * Returns a {@link Type} object that represents the formal return type of the method.
	 *
	 * <p>
	 * Same as calling {@link Method#getGenericReturnType()}.
	 *
	 * <p>
	 * If the return type is a parameterized type, the {@link Type} object returned reflects the actual type parameters used in the source code.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	<jc>// For method: public List&lt;String&gt; getValues()</jc>
	 * 	MethodInfo <jv>mi</jv> = ClassInfo.<jsm>of</jsm>(MyClass.<jk>class</jk>).getMethod(<js>"getValues"</js>);
	 * 	Type <jv>returnType</jv> = <jv>mi</jv>.getGenericReturnType();
	 * 	<jk>if</jk> (<jv>returnType</jv> <jk>instanceof</jk> ParameterizedType) {
	 * 		ParameterizedType <jv>pType</jv> = (ParameterizedType)<jv>returnType</jv>;
	 * 		<jc>// pType.getActualTypeArguments()[0] is String.class</jc>
	 * 	}
	 * </p>
	 *
	 * @return A {@link Type} object representing the formal return type.
	 * @see Method#getGenericReturnType()
	 */
	public Type getGenericReturnType() { return inner.getGenericReturnType(); }

	@Override /* Annotatable */
	public String getLabel() { return getDeclaringClass().getNameSimple() + "." + getShortName(); }

	/**
	 * Returns this method and all matching methods up the hierarchy chain.
	 *
	 * <p>
	 * Searches parent classes and interfaces for methods with matching name and parameter types.
	 * Results are returned in the following order:
	 * <ol>
	 * 	<li>This method
	 * 	<li>Any matching methods on declared interfaces of this class
	 * 	<li>Matching method on the parent class
	 * 	<li>Any matching methods on the declared interfaces of the parent class
	 * 	<li>Continue up the hierarchy
	 * </ol>
	 *
	 * <h5 class='section'>Examples:</h5>
	 * <p class='bjava'>
	 * 	<jc>// Interface and class hierarchy:</jc>
	 * 	<jk>interface</jk> I1 {
	 * 		<jk>void</jk> foo(String <jv>s</jv>);
	 * 	}
	 * 	<jk>class</jk> A {
	 * 		<jk>void</jk> foo(String <jv>s</jv>) {}
	 * 	}
	 * 	<jk>interface</jk> I2 {
	 * 		<jk>void</jk> foo(String <jv>s</jv>);
	 * 	}
	 * 	<jk>class</jk> B <jk>extends</jk> A <jk>implements</jk> I2 {
	 * 		&#64;Override
	 * 		<jk>void</jk> foo(String <jv>s</jv>) {}
	 * 	}
	 * 	<jc>// For B.foo(), returns: [B.foo, I2.foo, A.foo, I1.foo]</jc>
	 * 	MethodInfo <jv>mi</jv> = ...;
	 * 	List&lt;MethodInfo&gt; <jv>matching</jv> = <jv>mi</jv>.getMatchingMethods();
	 * </p>
	 *
	 * @return A list of matching methods including this one, in child-to-parent order.
	 */
	public List<MethodInfo> getMatchingMethods() { return matchingMethods.get(); }

	/**
	 * Returns the name of this method.
	 *
	 * @return The name of this method
	 */
	public String getName() { return inner.getName(); }

	/**
	 * Returns the bean property name if this is a getter or setter.
	 *
	 * @return The bean property name, or <jk>null</jk> if this isn't a getter or setter.
	 */
	public String getPropertyName() {
		String n = inner.getName();
		if ((n.startsWith("get") || n.startsWith("set")) && n.length() > 3)
			return Introspector.decapitalize(n.substring(3));
		if (n.startsWith("is") && n.length() > 2)
			return Introspector.decapitalize(n.substring(2));
		return n;
	}

	/**
	 * Returns the generic return type of this method as a {@link ClassInfo} object.
	 *
	 * @return The generic return type of this method.
	 */
	public ClassInfo getReturnType() { return returnType.get(); }

	/**
	 * Returns the signature of this method.
	 *
	 * <p>
	 * For no-arg methods, the signature will be a simple string such as <js>"toString"</js>.
	 * For methods with one or more args, the arguments will be fully-qualified class names (e.g.
	 * <js>"append(java.util.StringBuilder,boolean)"</js>)
	 *
	 * @return The methods signature.
	 */
	public String getSignature() {
		var sb = new StringBuilder(128);
		sb.append(inner.getName());
		var params = getParameters();
		if (params.size() > 0) {
			sb.append('(');
			for (var i = 0; i < params.size(); i++) {
				if (i > 0)
					sb.append(',');
				params.get(i).getParameterType().appendNameFormatted(sb, ClassNameFormat.FULL, true, '$', ClassArrayFormat.BRACKETS);
			}
			sb.append(')');
		}
		return sb.toString();
	}

	/**
	 * Returns <jk>true</jk> if this method has at least the specified parameters.
	 *
	 * <p>
	 * Method may or may not have additional parameters besides those specified.
	 *
	 * @param requiredParams The parameter types to check for.
	 * @return <jk>true</jk> if this method has at least the specified parameters.
	 */
	public boolean hasAllParameters(Class<?>...requiredParams) {
		var paramTypes = getParameters().stream().map(p -> p.getParameterType().inner()).toList();

		for (var c : requiredParams)
			if (! paramTypes.contains(c))
				return false;

		return true;
	}

	/**
	 * Returns <jk>true</jk> if the specified annotation is present on this method or any matching methods in parent classes/interfaces.
	 *
	 * <p>
	 * This method searches through all matching methods in the hierarchy.
	 *
	 * @param <A> The annotation type to look for.
	 * @param type The annotation to look for.
	 * @return <jk>true</jk> if the specified annotation is present on this method.
	 */
	@Override
	public <A extends Annotation> boolean hasAnnotation(Class<A> type) {
		return getAnnotations(type).findAny().isPresent();
	}

	/**
	 * Returns <jk>true</jk> if the parameters on the method only consist of the types specified in the list.
	 *
	 * <p>
	 * <b>Note:</b> This method is not meant to be used on methods with duplicate parameter types.
	 * It checks if each parameter type is present in the specified list, but does not verify
	 * that the count of each type matches exactly.
	 *
	 * <h5 class='figure'>Example:</h5>
	 * <p class='bjava'>
	 *
	 * 	<jc>// Example method:</jc>
	 * 	<jk>public void</jk> foo(String <jv>bar</jv>, Integer <jv>baz</jv>);
	 *
	 * 	<jv>fooMethod</jv>.hasOnlyParameterTypes(String.<jk>class</jk>, Integer.<jk>class</jk>);  <jc>// True.</jc>
	 * 	<jv>fooMethod</jv>.hasOnlyParameterTypes(String.<jk>class</jk>, Integer.<jk>class</jk>, Map.<jk>class</jk>);  <jc>// True.</jc>
	 * 	<jv>fooMethod</jv>.hasOnlyParameterTypes(String.<jk>class</jk>);  <jc>// False.</jc>
	 * </p>
	 *
	 * @param args The valid class types (exact) for the arguments.
	 * @return <jk>true</jk> if the method parameters only consist of the types specified in the list.
	 */
	public boolean hasOnlyParameterTypes(Class<?>...args) {
		for (var param : getParameters()) {
			var c1 = param.getParameterType().inner();
			var foundMatch = false;
			for (var c2 : args)
				if (c1 == c2)
					foundMatch = true;
			if (! foundMatch)
				return false;
		}
		return true;
	}

	/**
	 * Returns <jk>true</jk> if this method has the specified parameter.
	 *
	 * <p>
	 * Method may or may not have additional parameters besides the one specified.
	 *
	 * @param requiredParam The parameter type to check for.
	 * @return <jk>true</jk> if this method has at least the specified parameter.
	 */
	public boolean hasParameter(Class<?> requiredParam) {
		return hasAllParameters(requiredParam);
	}

	/**
	 * Returns <jk>true</jk> if this method has this return type.
	 *
	 * @param c The return type to test for.
	 * @return <jk>true</jk> if this method has this return type.
	 */
	public boolean hasReturnType(Class<?> c) {
		return inner.getReturnType() == c;
	}

	/**
	 * Returns <jk>true</jk> if this method has this return type.
	 *
	 * @param ci The return type to test for.
	 * @return <jk>true</jk> if this method has this return type.
	 */
	public boolean hasReturnType(ClassInfo ci) {
		return hasReturnType(ci.inner());
	}

	/**
	 * Returns <jk>true</jk> if this method has this parent return type.
	 *
	 * @param c The return type to test for.
	 * @return <jk>true</jk> if this method has this parent return type.
	 */
	public boolean hasReturnTypeParent(Class<?> c) {
		return ClassInfo.of(c).isParentOf(inner.getReturnType());
	}

	/**
	 * Returns <jk>true</jk> if this method has this parent return type.
	 *
	 * @param ci The return type to test for.
	 * @return <jk>true</jk> if this method has this parent return type.
	 */
	public boolean hasReturnTypeParent(ClassInfo ci) {
		return hasReturnTypeParent(ci.inner());
	}

	/**
	 * Returns the wrapped method.
	 *
	 * @return The wrapped method.
	 */
	public Method inner() {
		return inner;
	}

	/**
	 * Compares this MethodInfo with the specified object for equality.
	 *
	 * <p>
	 * Two MethodInfo objects are considered equal if they wrap the same underlying {@link Method} object.
	 * This delegates to the underlying {@link Method#equals(Object)} method.
	 *
	 * <p>
	 * This method makes MethodInfo suitable for use as keys in hash-based collections such as {@link HashMap}
	 * and {@link HashSet}.
	 *
	 * @param obj The object to compare with.
	 * @return <jk>true</jk> if the objects are equal, <jk>false</jk> otherwise.
	 */
	@Override
	public boolean equals(Object obj) {
		return obj instanceof MethodInfo other && eq(this, other, (x, y) -> eq(x.inner, y.inner) && eq(x.declaringClass, y.declaringClass));
	}

	/**
	 * Returns a hash code value for this MethodInfo.
	 *
	 * <p>
	 * This combines the hash code of the underlying {@link Method} with the hash code of the declaring class
	 * to ensure that methods from different subclasses that inherit the same method are not considered equal.
	 *
	 * <p>
	 * This method makes MethodInfo suitable for use as keys in hash-based collections such as {@link HashMap}
	 * and {@link HashSet}.
	 *
	 * @return A hash code value for this MethodInfo.
	 */
	@Override
	public int hashCode() {
		return 31 * inner.hashCode() + declaringClass.hashCode();
	}

	/**
	 * Shortcut for calling the invoke method on the underlying method.
	 *
	 * @param <T> The method return type.
	 * @param obj the object the underlying method is invoked from.
	 * @param args the arguments used for the method call
	 * @return The object returned from the method.
	 * @throws ExecutableException Exception occurred on invoked constructor/method/field.
	 */
	@SuppressWarnings("unchecked")
	public <T> T invoke(Object obj, Object...args) throws ExecutableException {
		return safe(() -> {
			try {
				return (T)inner.invoke(obj, args);
			} catch (InvocationTargetException e) {
				throw exex(e.getTargetException());
			}
		}, e -> exex(e));
	}

	/**
	 * Invokes the specified method using lenient argument matching.
	 *
	 * <p>
	 * Lenient matching allows arguments to be matched to parameters based on parameter types.
	 * <br>Arguments can be in any order.
	 * <br>Extra arguments will be ignored.
	 * <br>Missing arguments will be left <jk>null</jk>.
	 *
	 * <p>
	 * Note that this only works for methods that have distinguishable argument types.
	 * <br>It's not going to work on methods with generic argument types like <c>Object</c>
	 *
	 * @param pojo
	 * 	The POJO the method is being called on.
	 * 	<br>Can be <jk>null</jk> for static methods.
	 * @param args
	 * 	The arguments to pass to the method.
	 * @return
	 * 	The results of the method invocation.
	 * @throws ExecutableException Exception occurred on invoked constructor/method/field.
	 */
	public Object invokeLenient(Object pojo, Object...args) throws ExecutableException {
		return safe(() -> {
			return inner.invoke(pojo, ClassUtils.getMatchingArgs(inner.getParameterTypes(), args));
		}, e -> exex(e instanceof InvocationTargetException ? ((InvocationTargetException)e).getTargetException() : e));
	}

	@Override
	public boolean is(ElementFlag flag) {
		return switch (flag) {
			case BRIDGE -> isBridge();
			case NOT_BRIDGE -> ! isBridge();
			case DEFAULT -> isDefault();
			case NOT_DEFAULT -> ! isDefault();
			default -> super.is(flag);
		};
	}

	//-----------------------------------------------------------------------------------------------------------------
	// High Priority Methods (direct Method API compatibility)
	//-----------------------------------------------------------------------------------------------------------------

	/**
	 * Returns <jk>true</jk> if this method is a bridge method.
	 *
	 * @return <jk>true</jk> if this method is a bridge method.
	 */
	public boolean isBridge() { return inner.isBridge(); }

	/**
	 * Returns <jk>true</jk> if this method is a default method (Java 8+ interface default method).
	 *
	 * <p>
	 * Same as calling {@link Method#isDefault()}.
	 *
	 * <p>
	 * A default method is a public non-abstract instance method (i.e., non-static method with a body) declared in an interface.
	 *
	 * <h5 class='section'>Example:</h5>
	 * <p class='bjava'>
	 * 	<jc>// For interface: interface MyInterface { default String getName() { return "default"; } }</jc>
	 * 	MethodInfo <jv>mi</jv> = ClassInfo.<jsm>of</jsm>(MyInterface.<jk>class</jk>).getMethod(<js>"getName"</js>);
	 * 	<jk>if</jk> (<jv>mi</jv>.isDefault()) {
	 * 		<jc>// This is a default interface method</jc>
	 * 	}
	 * </p>
	 *
	 * @return <jk>true</jk> if this method is a default method.
	 * @see Method#isDefault()
	 */
	public boolean isDefault() { return inner.isDefault(); }

	/**
	 * Returns <jk>true</jk> if this method matches the specified method by name and parameter types.
	 *
	 * @param m The method to compare against.
	 * @return <jk>true</jk> if this method has the same name and parameter types as the specified method.
	 */
	public boolean matches(MethodInfo m) {
		return hasName(m.getName()) && hasMatchingParameters(m.getParameters());
	}

	private void addMatchingMethodsFromInterface(List<MethodInfo> result, ClassInfo iface) {
		// Add matching methods from this interface
		iface.getDeclaredMethods().stream().filter(this::matches).forEach(result::add);

		// Recursively search parent interfaces
		iface.getDeclaredInterfaces().stream().forEach(pi -> addMatchingMethodsFromInterface(result, pi));
	}

	//-----------------------------------------------------------------------------------------------------------------
	// Annotatable interface methods
	//-----------------------------------------------------------------------------------------------------------------

	private List<MethodInfo> findMatchingMethods() {
		var result = new ArrayList<MethodInfo>();
		result.add(this); // 1. This method

		var cc = getDeclaringClass();

		while (nn(cc)) {
			// 2. Add matching methods from declared interfaces of current class
			cc.getDeclaredInterfaces().stream().forEach(di -> addMatchingMethodsFromInterface(result, di));

			// 3. Move to parent class
			cc = cc.getSuperclass();
			if (nn(cc)) {
				// Add matching method from parent class
				cc.getDeclaredMethods().stream().filter(this::matches).findFirst().ifPresent(result::add);
			}
		}

		return result;
	}
}